"""Extents class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import collections

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.file_io.report.point import Point


class Extents(collections.UserList):
    """Class to keep track of extents (bounding box) of points."""
    def __init__(self):
        """Initializes the class."""
        super().__init__([Point(), Point()])
        self._first_time = True

    @property
    def min(self):
        """Returns the min point.

        Returns:
            See description.
        """
        return self.data[0]

    @min.setter
    def min(self, minimum):
        """Sets the min point."""
        self.data[0] = minimum

    @property
    def max(self):
        """Returns the max point.

        Returns:
            See description.
        """
        return self.data[1]

    @max.setter
    def max(self, maximum):
        """Sets the max point."""
        self.data[1] = maximum

    def add_xy(self, x, y):
        """Adjusts the extents (if necessary) given the point x and y values.

        Args:
            x (:obj:`float`): X coordinate.
            y (:obj:`float`): Y coordinate.
        """
        if self._first_time:
            self.min.x = x
            self.min.y = y
            self.max.x = x
            self.max.y = y
            self._first_time = False
            return

        # Check the minimums
        if x < self.min.x:
            self.min.x = x
        if y < self.min.y:
            self.min.y = y

        # Check the maximums
        if x > self.max.x:
            self.max.x = x
        if y > self.max.y:
            self.max.y = y

    def add_point(self, point):
        """Adjusts the extents (if necessary) given the point.

        Args:
            point: A list-like or tuple-like x,y point (like the Point class).
        """
        self.add_xy(point[0], point[1])

    def add_points(self, points):
        """Adjusts the extents (if necessary) given the point.

        Args:
            points: A list of list-like or tuple-like x,y points (like the Point class).
        """
        for point in points:
            self.add_point(point)
