"""Obstruction coverage looper class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.srh.components.obstruction_component import ObstructionComponent
from xms.srh.file_io.report import report_util
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase


class ObstructionCoverageLooper(CoverageLooperBase):
    """Adds obstruction coverages to the report."""
    def __init__(self, notes_db, report_dir, query, logger):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            logger(:obj:`logger`): The logger
        """
        super().__init__(notes_db, report_dir, query, 'Obstructions', 'Obstruction_Component')
        self._logger = logger

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return ObstructionComponent(main_file)

    def _load_component_ids(self, obstruction_component):  # pragma: no cover
        """Loads the feature arc/point IDs into the component.

        Args:
            obstruction_component (:obj:`ObstructionComponent`): The obstructions component to load the IDs into.
        """
        self._query.load_component_ids(obstruction_component, points=True, arcs=True)

    def _store_coverage_data(self, coverage, obstruction_component):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            obstruction_component (:obj:`ObstructionComponent`): The obstructions component.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(coverage, obstruction_component, self._notes_db, self._logger)

    @staticmethod
    def get_jinja(coverage, obstruction_component, notes_db, logger):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            obstruction_component (:obj:`ObstructionComponent`): The obstructions component.
            notes_db (:obj:`Notes`): Notes object.
            logger(:obj:`logger`): The logger.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        if logger:
            logger.info(f'Getting data for obstruction coverage "{coverage.name}"')

        # Store all jinja stuff in a dict
        coverage_jinja = {'name': coverage.name}
        report_util.add_object_notes(notes_db, coverage.uuid, coverage_jinja)

        # Add arc obstructions
        arcs = coverage.arcs
        coverage_jinja['arc_count'] = str(len(arcs))
        obstruction_arcs = []
        for arc in arcs:
            obstruction_jinja = {'id': str(arc.id)}
            comp_id = obstruction_component.get_comp_id(TargetType.arc, arc.id)
            p = obstruction_component.data.obstruction_data_param_from_id(comp_id)
            obstruction_jinja['width'] = p.width
            obstruction_jinja['thickness'] = p.thickness
            obstruction_jinja['drag_coefficient'] = p.drag
            obstruction_arcs.append(obstruction_jinja)

        # Add point obstructions
        points = coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        coverage_jinja['point_count'] = str(len(points))
        obstruction_points = []
        for point in points:
            obstruction_jinja = {'id': str(point.id)}
            comp_id = obstruction_component.get_comp_id(TargetType.point, point.id)
            p = obstruction_component.data.obstruction_data_param_from_id(comp_id)
            obstruction_jinja['width'] = p.width
            obstruction_jinja['thickness'] = p.thickness
            obstruction_jinja['invert_elevation'] = point.z
            obstruction_jinja['drag_coefficient'] = p.drag
            obstruction_points.append(obstruction_jinja)

        coverage_jinja['obstruction_arcs'] = obstruction_arcs
        coverage_jinja['obstruction_points'] = obstruction_points
        return coverage_jinja
