"""The Compare Simulation Data dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QDialogButtonBox

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.srh.floodway.simulation_compare_calc import SimulationCompareCalc
from xms.srh.gui.compare_simulation_data_dialog_ui import Ui_CompareSimulationDataDialog  # noqa: I202
from xms.srh.gui.compare_simulation_data_output_dialog import CompareSimulationDataOutputDialog  # noqa: I202


class CompareSimulationDataDialog(XmsDlg):
    """The Compare Simulation Data dialog."""
    def __init__(self, parent=None, help_url=None, query=None):
        """Initializes the dialog.

        Args:
            parent (:obj:`QObject`): The Qt parent object
            help_url (:obj:`str`): The url that we want the help button to bring up
            query (:obj:`Query`): XMS interprocess communication object
        """
        super().__init__(parent, 'xms.srh.gui.compare_simulation_data_dialog')

        self.ui = Ui_CompareSimulationDataDialog()
        self.ui.setupUi(self)
        self.pe_tree = None
        self.fp_geom_uuid = None
        self.fp_geom_name = ""
        self.fw_geom_uuid = None
        self.fw_geom_name = ""
        self.fp_wse_uuid = None
        self.fp_vmag_uuid = None
        self.fw_wse_uuid = None
        self.fw_vmag_uuid = None
        self.cross_section_uuid = None

        if query:  # Setup project explorer trees we will need.
            self.pe_tree = query.copy_project_tree()
            tree_util.filter_project_explorer(self.pe_tree, DatasetSelector.is_scalar_if_dset)
            # Get the simulation tree item
            sim_uuid = query.parent_item_uuid()
            # Look for a linked mesh under the simulation
            sim_item = tree_util.find_tree_node_by_uuid(self.pe_tree, sim_uuid)
            grid_items = tree_util.descendants_of_type(sim_item, xms_types=['TI_MESH2D_PTR'], allow_pointers=True)
            if grid_items:  # Have a linked mesh
                # Find the real mesh tree item
                self.fp_geom_uuid = grid_items[0].uuid
                self.fw_geom_uuid = grid_items[0].uuid
        self.query = query

        # example of help URL
        # help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"
        if help_url:
            self.help_url = help_url
        else:
            self.help_url = 'https://www.xmswiki.com/wiki/SMS:Floodway_Tools#Compare_Simulation_Data'
            # self.ui.btn_help.hide()

        self.setup_controls()
        self.setup_connections()

        self.set_enabled_or_disabled_state()

    def setup_controls(self):
        """Set up the controls for the dialog."""
        self.set_fp_geom_name()
        self.set_fw_geom_name()

    def setup_connections(self):
        """Setup signals and slots for the dialog's controls."""
        self.ui.btn_help.clicked.connect(self.help_requested)

        self.ui.btn_geo_floodplain.clicked.connect(self.select_floodplain_geometry_source)
        self.ui.btn_fp_wse.clicked.connect(self.select_fp_wse_dataset)
        self.ui.btn_fp_vmag.clicked.connect(self.select_fp_vmag_dataset)

        self.ui.btn_geo_floodway.clicked.connect(self.select_floodway_geometry_source)
        self.ui.btn_fw_wse.clicked.connect(self.select_fw_wse_dataset)
        self.ui.btn_fw_vmag.clicked.connect(self.select_fw_vmag_dataset)

        self.ui.btn_crosssection.clicked.connect(self.select_cross_section_coverage)

    def select_floodplain_geometry_source(self):
        """Dialog to select a coverage."""
        dialog = TreeItemSelectorDlg(
            title='Select Floodplain Geometry',
            pe_tree=self.pe_tree,
            target_type=xmd.UGridItem,
            previous_selection=self.fp_geom_uuid,
            show_root=True,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS'],
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fp_geom_uuid = dialog.get_selected_item_uuid()
            self.set_fp_geom_name()
            self.fp_wse_uuid = None
            self.ui.txt_fp_wse_selected.setText("(none selected)")
            self.fp_vmag_uuid = None
            self.ui.txt_fp_vmag_selected.setText("(none selected)")
        self.set_enabled_or_disabled_state()
        self.setup_controls()

    def set_fp_geom_name(self):
        """Set floodplain geometry name."""
        if self.fp_geom_uuid:
            geom_node = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fp_geom_uuid)
            geom_name = geom_node.name
            self.fp_geom_name = geom_name
            self.ui.txt_floodplain_selected.setText(geom_name)
        else:
            self.ui.txt_floodplain_selected.setText("(none selected)")

    def select_floodway_geometry_source(self):
        """Dialog to select a coverage."""
        dialog = TreeItemSelectorDlg(
            title='Select Floodway Geometry',
            pe_tree=self.pe_tree,
            target_type=xmd.UGridItem,
            previous_selection=self.fw_geom_uuid,
            show_root=True,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS'],
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fw_geom_uuid = dialog.get_selected_item_uuid()
            self.set_fw_geom_name()
            self.fw_wse_uuid = None
            self.ui.txt_fw_wse_selected.setText("(none selected)")
            self.fw_vmag_uuid = None
            self.ui.txt_fw_vmag_selected.setText("(none selected)")
        self.set_enabled_or_disabled_state()
        self.setup_controls()

    def set_fw_geom_name(self):
        """Set floodway geometry name."""
        if self.fw_geom_uuid:
            geom_node = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fw_geom_uuid)
            geom_name = geom_node.name
            self.fw_geom_name = geom_name
            self.ui.txt_floodway_selected.setText(geom_name)
        else:
            self.ui.txt_floodway_selected.setText("(none selected)")

    def select_cross_section_coverage(self):
        """Select floodplain geometry."""
        dialog = TreeItemSelectorDlg(
            title='Select Cross Section Coverage',
            pe_tree=self.pe_tree,
            target_type=xmd.CoverageItem,
            previous_selection=self.cross_section_uuid,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.cross_section_uuid = dialog.get_selected_item_uuid()
            if self.cross_section_uuid:
                cover_node = tree_util.find_tree_node_by_uuid(self.pe_tree, self.cross_section_uuid)
                cross_name = cover_node.name
                self.ui.txt_crossection_selected.setText(cross_name)
            else:
                self.ui.txt_crossection_selected.setText("(none selected)")
        self.check_if_ok()

    def select_fp_wse_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        grid_tree = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fp_geom_uuid)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select WSE Dataset',
            pe_tree=grid_tree,
            selected_dataset=self.fp_wse_uuid,
            query=self.query,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fp_wse_uuid = dialog.get_selected_item_uuid()
            if self.fp_wse_uuid:
                wse_name = tree_util.build_tree_path(self.pe_tree, self.fp_wse_uuid)
                self.ui.txt_fp_wse_selected.setText(wse_name)
            else:
                self.ui.txt_fp_wse_selected.setText("(none selected)")
        self.check_if_ok()

    def select_fp_vmag_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        grid_tree = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fp_geom_uuid)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select Velocity Magnitude Dataset',
            pe_tree=grid_tree,
            selected_dataset=self.fp_vmag_uuid,
            query=self.query,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fp_vmag_uuid = dialog.get_selected_item_uuid()
            if self.fp_vmag_uuid:
                vmag_name = tree_util.build_tree_path(self.pe_tree, self.fp_vmag_uuid)
                self.ui.txt_fp_vmag_selected.setText(vmag_name)
            else:
                self.ui.txt_fp_vmag_selected.setText("(none selected)")
        self.check_if_ok()

    def select_fw_wse_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        grid_tree = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fw_geom_uuid)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select WSE Dataset',
            pe_tree=grid_tree,
            selected_dataset=self.fw_wse_uuid,
            query=self.query,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fw_wse_uuid = dialog.get_selected_item_uuid()
            if self.fw_wse_uuid:
                wse_name = tree_util.build_tree_path(self.pe_tree, self.fw_wse_uuid)
                self.ui.txt_fw_wse_selected.setText(wse_name)

            else:
                self.ui.txt_fw_wse_selected.setText("(none selected)")
        self.check_if_ok()

    def select_fw_vmag_dataset(self):
        """Launches the select dataset dialog and sets the cross section values."""
        grid_tree = tree_util.find_tree_node_by_uuid(self.pe_tree, self.fw_geom_uuid)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select Velocity Magnitude Dataset',
            pe_tree=grid_tree,
            selected_dataset=self.fw_vmag_uuid,
            query=self.query,
            show_root=True,
            parent=self
        )

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.fw_vmag_uuid = dialog.get_selected_item_uuid()
            if self.fw_vmag_uuid:
                vmag_name = tree_util.build_tree_path(self.pe_tree, self.fw_vmag_uuid)
                self.ui.txt_fw_vmag_selected.setText(vmag_name)
            else:
                self.ui.txt_fw_vmag_selected.setText("(none selected)")
        self.check_if_ok()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Bring up the compare simulation dialog and send back to XMS when dialog is accepted."""
        super().accept()
        simulation_comparison = SimulationCompareCalc(
            self.query, self.fp_geom_uuid, self.fp_geom_name, self.fp_wse_uuid, self.fp_vmag_uuid, self.fw_geom_uuid,
            self.fw_geom_name, self.fw_wse_uuid, self.fw_vmag_uuid, self.cross_section_uuid
        )
        input_data = simulation_comparison.compute_data()
        significant_figures = self.ui.cbx_significant_figures.currentIndex()
        dlg = CompareSimulationDataOutputDialog(data=input_data, significant_figures=significant_figures)
        dlg.exec()

    def set_enabled_or_disabled_state(self):
        """Set the enabled/disabled states of the dialog controls."""
        fp_sim_defined = True if self.fp_geom_uuid else False
        self.ui.txt_fp_wse.setEnabled(fp_sim_defined)
        self.ui.btn_fp_wse.setEnabled(fp_sim_defined)
        self.ui.txt_fp_wse_selected.setEnabled(fp_sim_defined)
        self.ui.txt_fp_vmag.setEnabled(fp_sim_defined)
        self.ui.btn_fp_vmag.setEnabled(fp_sim_defined)
        self.ui.txt_fp_vmag_selected.setEnabled(fp_sim_defined)

        fw_sim_defined = True if self.fw_geom_uuid else False
        self.ui.txt_fw_wse.setEnabled(fw_sim_defined)
        self.ui.btn_fw_wse.setEnabled(fw_sim_defined)
        self.ui.txt_fw_wse_selected.setEnabled(fw_sim_defined)
        self.ui.txt_fw_vmag.setEnabled(fw_sim_defined)
        self.ui.btn_fw_vmag.setEnabled(fw_sim_defined)
        self.ui.txt_fw_vmag_selected.setEnabled(fw_sim_defined)

        self.check_if_ok()

    def check_if_ok(self):
        """Check if the 'OK' button should be enabled."""
        enable_ok = True
        if not self.fp_geom_uuid:
            enable_ok = False
        if not self.fw_geom_uuid:
            enable_ok = False
        if not self.fp_wse_uuid:
            enable_ok = False
        if not self.fp_vmag_uuid:
            enable_ok = False
        if not self.fw_wse_uuid:
            enable_ok = False
        if not self.fw_vmag_uuid:
            enable_ok = False
        if not self.cross_section_uuid:
            enable_ok = False

        if enable_ok:
            self.ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(True)
        else:
            self.ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(False)
        return enable_ok
