"""A child XY series editor dialog for the parameters dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
import pandas
from PySide2.QtCore import QItemSelectionModel, Qt
from PySide2.QtWidgets import QAbstractItemView, QSplitter

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.srh.gui.parameters_xy_series_dialog_ui import Ui_parameters_xy_series_dialog


class ParametersXySeriesDialog(XmsDlg):
    """A dialog for all xy series used in the parameters dialog."""

    LIST_DIALOG_COLUMN = 2  # Hidden column that holds the xy series editor dialogs

    def __init__(self, xy_series_list, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            xy_series_list (:obj:`list[tuple]`): List of XY series.
                Example: [('name': {'X': [1, 2], 'Y': [3, 4]}), ('name2': {'X': [1, 2], 'Y': [3, 4]})]
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, 'xms.srh.gui.parameters_xy_series_dialog.py')

        self.xy_series_list = xy_series_list
        self.ui = None
        self.model = None
        self.actions = None
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D'

        self.setup_ui()

    def add_accessible_names_for_tests(self):
        """Adds accessible names for some widgets so that they can be found by GuiTester."""
        self.ui.grp_xy_series_editor.setAccessibleName('XY Series Editor group box')
        self.ui.wid_xy_list.setAccessibleName('XY Series list widget')
        self.ui.tbar_tools.setAccessibleName('XY Series list toolbar')
        self.ui.tbl_xy_list.setAccessibleName('XY Series list table')

    def delete_all_xyseries_dialog(self):
        """Deletes all the XY series dialogs to avoid a script error when returning."""
        for row in range(self.model.rowCount()):
            self.model.data_frame['Dialog'].iloc[row].deleteLater()

    def accept(self):
        """Clean up child XY series editors when dialog closes."""
        self.delete_all_xyseries_dialog()
        super().accept()

    def reject(self):
        """Clean up child XY series editors when dialog closes."""
        self.delete_all_xyseries_dialog()
        super().reject()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def setup_ui(self):
        """Setup the dialog widgets."""
        self.ui = Ui_parameters_xy_series_dialog()
        self.ui.setupUi(self)

        self.ui.tbl_xy_list.size_to_contents = True
        widget_builder.style_table_view(self.ui.tbl_xy_list)

        self.create_tools()
        self.create_list()
        self.add_splitter()

    def create_tools(self):
        """Create a toolbar for the XY series table."""
        button_list = [
            [':/resources/icons/row-insert.svg', 'Insert Row', self.on_btn_insert],
            [':/resources/icons/row-delete.svg', 'Delete Row', self.on_btn_delete],
            [':/resources/icons/row-up.svg', 'Move Up', self.on_btn_up],
            [':/resources/icons/row-down.svg', 'Move Down', self.on_btn_down]
        ]
        self.actions = widget_builder.setup_toolbar(self.ui.tbar_tools, button_list)

    def on_btn_insert(self):
        """Inserts a row into XY series table."""
        index_list = self.ui.tbl_xy_list.selectedIndexes()
        row = 0 if not index_list else index_list[0].row()
        self.model.insertRows(row, 1)

        # Put data into the new row
        dlg = self.create_xy_series_dialog({'X': [], 'Y': []})
        index = self.model.createIndex(row, self.LIST_DIALOG_COLUMN)
        self.model.setData(index, dlg)
        self.update_xy_list_selection(row)

    def on_btn_delete(self):
        """Deletes currently selected row from the XY series table."""
        index_list = self.ui.tbl_xy_list.selectedIndexes()
        if not index_list:
            return
        row = index_list[0].row()
        self.model.data_frame['Dialog'].iloc[row].deleteLater()
        self.model.removeRows(row, 1)
        # Sometimes when deleting a row, nothing is selected.  This makes
        # sure a row is selected.
        sel_rows = self.ui.tbl_xy_list.selectionModel().selectedRows()
        if not sel_rows and self.model.rowCount() > 0:
            self.update_xy_list_selection(0)

    def on_btn_up(self):
        """Move current row up in the XY series table."""
        self.move_row(True)

    def on_btn_down(self):
        """Move current row down in the XY series table."""
        self.move_row(False)

    def move_row(self, up):
        """Move current row up or down in the XY series table.

        Args:
            up (:obj:`int`): True if the row should be moved up
        """
        index_list = self.ui.tbl_xy_list.selectedIndexes()
        if not index_list:
            return
        source_row = index_list[0].row()
        begin_row = 0 if up else self.model.rowCount() - 1
        if source_row == begin_row:
            return
        source_idx = source_row + 1
        dest_row = source_row - 1 if up else source_row + 1
        dest_idx = dest_row + 1
        self.model.swap_rows(source_idx=source_idx, dest_idx=dest_idx, source_row=source_row, dest_row=dest_row)

        # Update the selection
        self.update_xy_list_selection(dest_row)

    def update_xy_list_selection(self, row):
        """Set the current selection in the XY series curve list.

        Args:
            row (:obj:`int`): Index of the row to select
        """
        new_index = self.model.index(row, 0)
        self.ui.tbl_xy_list.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )

    def add_splitter(self):
        """Adds a QSplitter between the tables so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        splitter = QSplitter(self)
        splitter.setOrientation(Qt.Horizontal)
        splitter.addWidget(self.ui.wid_xy_list)
        splitter.addWidget(self.ui.grp_xy_series_editor)
        # Just use a fixed starting width of 300 for the table for now
        width = self.ui.wid_xy_list.sizeHint().width()
        if width < 100 or width > 350:
            width = 200
        splitter.setSizes([width, 600])
        splitter.setChildrenCollapsible(False)
        splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        pos = self.ui.lay_horz.indexOf(self.ui.grp_xy_series_editor)
        self.ui.lay_horz.insertWidget(pos, splitter)

    def create_list(self):
        """Creates the list of XY series."""
        df_dict = {'Name': [], 'Rating\nCurve': [], 'Dialog': []}

        # Add items to list
        if self.xy_series_list:
            for row, series in enumerate(self.xy_series_list):
                df_dict['Name'].append(series[0])
                df_dict['Rating\nCurve'].append(series[2])
                df_dict['Dialog'].append(self.create_xy_series_dialog(self.xy_series_list[row][1]))

        df = pandas.DataFrame(data=df_dict)
        df.index += 1
        self.model = QxPandasTableModel(df)
        self.model.set_default_values({'Name': 'new series', 'Rating\nCurve': False, 'Dialog': None})
        self.model.set_checkbox_columns({1})
        self.ui.tbl_xy_list.setModel(self.model)
        self.ui.tbl_xy_list.setColumnHidden(self.LIST_DIALOG_COLUMN, True)
        self.ui.tbl_xy_list.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.ui.tbl_xy_list.setSelectionMode(QAbstractItemView.SingleSelection)
        self.ui.tbl_xy_list.selectionModel().selectionChanged.connect(self.on_list_selection_changed)
        self.ui.tbl_xy_list.setFocus()
        # self.ui.lst_xy_list.setMinimumWidth(self.ui.lst_xy_list.sizeHintForColumn(0))
        self.update_xy_list_selection(0)
        self.ui.tbl_xy_list.resizeColumnsToContents()
        self.ui.tbl_xy_list.horizontalHeader().setStretchLastSection(True)

    def create_xy_series_dialog(self, xyseries_dict):
        """Returns the XY Series dialog from a dict.

        Args:
            xyseries_dict (:obj:`dict`): Mapping of x/y column name to x/y column values. {str: [float]}

        Returns:
            (:obj:`XySeriesEditor`): See description
        """
        df = pandas.DataFrame(data=xyseries_dict)
        dlg = XySeriesEditor(data_frame=df, series_name='XY Series', parent=self)
        flags = dlg.windowFlags()
        dlg.setWindowFlags(flags & ~Qt.Dialog)
        dlg.ui.buttonBox.hide()
        # self.ui.lay_horz.addWidget(dlg)
        self.ui.hlay_group_box.addWidget(dlg)
        dlg.hide()
        return dlg

    def on_list_selection_changed(self, selected, deselected):
        """Handles list selection change.

        Args:
            selected (:obj:`QItemSelection`): New selected item.
            deselected (:obj:`QItemSelection`): Old selected item.
        """
        if deselected.count() > 0:
            old_index = deselected.indexes()[0]
            old_row = old_index.row()
            dialog = self.model.data_frame['Dialog'].iloc[old_row]
            dialog.hide()

        if selected.count() > 0:
            index = selected.indexes()[0]
            row = index.row()
            dialog = self.model.data_frame['Dialog'].iloc[row]
            dialog.show()

    def get_xy_series_data(self):
        """Returns the xy series data as a list of tuples of (names, dicts of x, y lists, rating curve).

        Returns:
            See description.
        """
        xy_series_data = []
        row_count = self.model.rowCount()
        for row in range(row_count):
            name_index = self.model.createIndex(row, 0)
            name = self.model.data(name_index)
            rating_curve_index = self.model.createIndex(row, 1)
            rating_curve = (self.model.data(rating_curve_index, Qt.CheckStateRole) == Qt.Checked)
            df = self.model.data_frame['Dialog'].iloc[row].model.data_frame
            xy_dict = {
                df.columns.values[0]: df[df.columns[0]].to_list(),
                df.columns.values[1]: df[df.columns[1]].to_list()
            }
            xy_series_data.append((name, xy_dict, rating_curve))

        return xy_series_data
