"""Qt delegate for the sediment curve button widget."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSize, Qt
from PySide2.QtGui import QBrush, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules
from xms.srh.data.material_data import MaterialData


class SrhSedCurveButtonDelegate(QStyledItemDelegate):
    """Qt delegate for the sediment curve button widget."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (:obj:`QObject`): The parent Qt object.
        """
        super().__init__(parent)
        self.parent_dlg = parent

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton('Edit Curve...')
            height = option.rect.height()
            width = option.rect.width()
            min_dim = min(height, width)
            icon_size = QSize(min_dim, min_dim)
            btn.setIconSize(icon_size)
            # TODO: Get a curve preview image on the button
            btn.setFixedWidth(width)
            btn.setFixedHeight(height)

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the depth varied Manning's N XY series editor button is clicked.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                # get parent dialog
                parent = self.parent_dlg
                mat_id = parent.mat_att_material_id
                # get attribute id
                att_id_index = model.index(index.row(), MaterialData.COL_PROP_ID)
                att_id = int(model.data(att_id_index, Qt.DisplayRole))
                # get the curve for the material and attribute
                df = parent.material_data.gradation_curve_from_mat_id_prop_id(mat_id, att_id).to_dataframe()
                if not df.index.empty and df.index[0] == 0:
                    df.index += 1  # Start index at 1, not 0

                # xy_series dialog
                df.columns = ['Particle diameter (mm)', '% finer']
                # TODO: log_x=True causes a crash, check next version of matplotlib
                dlg = XySeriesEditor(
                    data_frame=df,
                    series_name='Gradation Curve',
                    dialog_title='Gradation Curve',
                    parent=self.parent_dlg,
                    log_x=False
                )
                if dlg.exec():
                    df = dlg.model.data_frame
                    df.columns = ['Particle diameter (mm)', 'Percent finer']
                    parent.material_data.set_gradation_curve(mat_id, att_id, df.to_xarray())
                return True
        return super().editorEvent(event, model, option, index)
