"""ListDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard python modules
import os
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Qt
from PySide2.QtGui import QDoubleValidator, QIntValidator, QKeySequence
from PySide2.QtWidgets import QApplication, QDialogButtonBox, QToolBar, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.number_corrector import NumberCorrector
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView


# 4. Local modules


def get_unique_selected_rows(selected_list):
    """Returns a list of selected rows.

     Selection may be disjoint.

    Args:
        selected_list (:obj:`list[QModelIndex]`): The selected list.

    Returns:
        (:obj:`tuple`): tuple containing:

        (:obj:`int`): Minimum selected row

        (:obj:`int`): Maximum selected row

        (:obj:`int`): Number of unique rows selected.
    """
    row_set = set()
    for index in selected_list:
        row_set.add(index.row())
    return list(row_set)


class TableDialog(XmsDlg):
    """A dialog showing a table. Used for land use and soil tables."""

    def __init__(self, dialog_title, data, parent=None):
        """Initializes the dialog.

        Args:
            dialog_title (:obj:`str`): The text to show in the dialog window title.
            data (:obj:`pandas.DataFrame`): The data to appear in the dialog.
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, 'xms.srhw.gui.table_dialog')

        self.df = data
        self.actions = {}
        self._editing = False
        self.setWindowTitle(dialog_title)

        vertical_layout = QVBoxLayout(self)
        self._create_table_view(vertical_layout)
        self._create_toolbar(vertical_layout)
        self._create_button_box(vertical_layout)

        # Signals
        self.table_view.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.table_view.setContextMenuPolicy(Qt.CustomContextMenu)
        self.table_view.customContextMenuRequested.connect(self.on_right_click)
        self.btn_box.helpRequested.connect(self.help_requested)

        self.on_selection_changed()

    def _create_table_view(self, vertical_layout):
        """Creates the table widget.

        Args:
            vertical_layout: The layout.
        """
        self.data_model = QxPandasTableModel(self.df)
        self.table_view = QxTableView(self)
        self.table_view.setModel(self.data_model)
        columns = list(self.df.columns)
        defaults = {}
        dbl_validator = QDoubleValidator(self)
        dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        dbl_delegate = EditFieldValidator(dbl_validator, self)
        int_validator = QIntValidator()
        int_validator.setBottom(1)
        int_delegate = EditFieldValidator(int_validator, self)
        for col, col_name in enumerate(columns):
            if col_name == 'Name':
                defaults[col_name] = 'Untitled'
            elif col_name == 'ID':
                defaults[col_name] = 1
                self.table_view.setItemDelegateForColumn(col, int_delegate)
            else:
                defaults[col_name] = 0.0
                self.table_view.setItemDelegateForColumn(col, dbl_delegate)
        self.table_view.model().set_default_values(defaults)
        vertical_layout.addWidget(self.table_view)
        widget_builder.style_table_view(self.table_view)
        self.table_view.resizeColumnsToContents()
        self.table_view.horizontalHeader().setStretchLastSection(True)
        self.table_view.verticalHeader().hide()
        self.table_view.clearSelection()

    def on_right_click(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(:obj:`QPoint`): The point clicked.
        """
        menu_list = [
            [':/resources/icons/row-insert.svg', 'Insert', self.on_btn_insert],
            [':/resources/icons/row-delete.svg', 'Delete', self.on_btn_delete],
            # ['copy', 'Copy', self.on_copy],
            # ['paste', 'Paste', self.on_paste]
        ]
        menu = widget_builder.setup_context_menu(self.table_view, menu_list)
        menu.popup(self.table_view.viewport().mapToGlobal(point))

    def _create_toolbar(self, vertical_layout):
        """Creates the toolbar.

        Args:
            vertical_layout: The layout
        """
        self.toolbar = QToolBar(self)
        button_list = [[':/resources/icons/row-insert.svg', 'Insert Row', self.on_btn_insert],
                       [':/resources/icons/row-add.svg', 'Add Row', self.on_btn_add],
                       [':/resources/icons/row-delete.svg', 'Delete Row', self.on_btn_delete],
                       [':/resources/icons/row-up.svg', 'Move Up', self.on_btn_up],
                       [':/resources/icons/row-down.svg', 'Move Down', self.on_btn_down]]
        self.actions = widget_builder.setup_toolbar(self.toolbar, button_list)
        vertical_layout.addWidget(self.toolbar)

    def _create_button_box(self, vertical_layout):
        """Creates the button box.

        Args:
            vertical_layout: The layout
        """
        self.btn_box = QDialogButtonBox()
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        vertical_layout.addWidget(self.btn_box)

    def on_selection_changed(self):
        """Called when the user clicks in the table and changes what cells are selected."""
        selection_list = self.table_view.selectedIndexes()
        selections_exist = len(selection_list) > 0
        self.toolbar.widgetForAction(self.actions[':/resources/icons/row-insert.svg']).setEnabled(selections_exist)
        self.toolbar.widgetForAction(self.actions[':/resources/icons/row-add.svg']).setEnabled(True)
        self.toolbar.widgetForAction(self.actions[':/resources/icons/row-delete.svg']).setEnabled(selections_exist)
        self.toolbar.widgetForAction(self.actions[':/resources/icons/row-up.svg']).setEnabled(selections_exist)
        self.toolbar.widgetForAction(self.actions[':/resources/icons/row-down.svg']).setEnabled(selections_exist)

    def add_or_insert(self, add):
        """Called when the Add or Insert buttons are clicked. Adds rows to the table.

        Args:
            add (:obj:`bool`): True if adding, False if inserting
        """
        unique_rows = get_unique_selected_rows(self.table_view.selectedIndexes())
        row = self.table_view.model().rowCount() if not unique_rows else min(unique_rows)
        if add and row < self.table_view.model().rowCount():
            row = row + 1

        # Get the default land use ID and set it
        df = self.table_view.model().data_frame
        ids = df['ID'].tolist()
        max_id = 0
        if len(ids):
            max_id = max(ids)
        columns = list(df.columns)
        defaults = {}
        for col_name in columns:
            if col_name == 'Name':
                defaults[col_name] = 'Untitled'
            elif col_name == 'ID':
                defaults[col_name] = max_id + 1
            else:
                defaults[col_name] = 0.0
        self.table_view.model().set_default_values(defaults)

        # Insert the row
        self.table_view.model().insertRows(row, 1)

    def on_btn_insert(self):
        """Called when the Insert button is clicked. Inserts rows to the spreadsheet."""
        self.add_or_insert(False)

    def on_btn_add(self):
        """Called when the Add button is clicked. Adds rows to the table."""
        self.add_or_insert(True)

    def on_btn_delete(self):
        """Called when the Delete button is clicked. Deletes rows from the table."""
        selected_list = self.table_view.selectedIndexes()
        if not selected_list:
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(parent=self, message='Must select a row to delete.', app_name=app_name)
            return

        # Get the count of unique rows and the span from the selection
        unique_rows = get_unique_selected_rows(selected_list)
        min_row = min(unique_rows)
        self.table_view.model().removeRows(min_row, len(unique_rows))
        self.on_selection_changed()  # To update enabling

    def on_btn_up(self):
        """Called when the Up button is clicked. Moves rows up in the table."""
        selected_list = self.table_view.selectedIndexes()
        if not selected_list:
            return
        self.move(True, selected_list)

    def on_btn_down(self):
        """Called when the Down button is clicked. Moves rows down in the table."""
        selected_list = self.table_view.selectedIndexes()
        if not selected_list:
            return
        self.move(False, selected_list)

    def move(self, up, selected_list):
        """Moves a row up or down. From https://www.qtcentre.org/threads/3386-QTableWidget-move-row.

        Args:
            up (:obj:`bool`): True if moving up, else False
            selected_list (:obj:'list[PySide2.QtCore.QModelIndex]'): The list of selected cells
        """
        self._editing = True
        source_row = selected_list[0].row()
        source_col = selected_list[0].column()
        dest_row = source_row - 1 if up else source_row + 1
        if dest_row < 0 or dest_row > self.table_view.model().rowCount() - 1:
            return
        self.table_view.model().swap_rows(source_idx=source_row + 1, dest_idx=dest_row + 1, source_row=source_row,
                                          dest_row=dest_row)
        self.table_view.selectionModel().select(self.table_view.model().index(dest_row, source_col),
                                                QItemSelectionModel.ClearAndSelect)
        self._editing = False

    def copy_paste(self, do_copy):
        """Handles copy/paste.

        Args:
            do_copy (:obj:`bool`): True if copying, False if pasting.
        """
        selection_list = self.table_view.selectedIndexes()
        if len(selection_list) > 0:
            item = self.table_view.model().data(selection_list[0])
            if item:
                if do_copy:
                    QApplication.clipboard().setText(item.text())
                else:
                    text = QApplication.clipboard().text()
                    item.setText(text)

    def on_copy(self):
        """Handles copy."""
        self.copy_paste(True)

    def on_paste(self):
        """Handles paste."""
        self.copy_paste(False)

    def keyPressEvent(self, event):  # noqa N802 (function name 'keyPressEvent' should be lowercase)
        """Handle Copy/Paste via CTRL+C, CTRL+V.

        Args:
            event (:obj:`QKeyEvent`): The event
        """
        if event.matches(QKeySequence.Copy):
            self.on_copy()
        elif event.matches(QKeySequence.Paste):
            self.on_paste()

    def _save_data(self):
        self.df = self.table_view.model().data_frame

    def accept(self):
        """Dialog accept."""
        super().accept()
        self._save_data()

    @staticmethod
    def help_requested():
        """Opens a web browser to a URL when the user clicks the Help button."""
        url = 'https://www.xmswiki.com/wiki/SMS:SMS'
        webbrowser.open(url)

# def main():
#     """Main function, for testing."""
#     import sys
#
#     app = QApplication(sys.argv)
#     dialog = TableDialog('title', ['A', 'B', 'C'], '')
#
#     if dialog.exec() == QDialog.Accepted:
#         pass
#
#     sys.exit(app.exec_())
#
#
# if __name__ == "__main__":
#     main()
