"""Handles all the model global and arc data."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel


# 4. Local modules


class ConduitShape:
    """Types of conduits."""
    CIRCULAR = 'Circular'
    FILLED_CIRCULAR = 'Filled Circular'
    CLOSED_RECTANGULAR = 'Closed Rectangular'
    HORIZONTAL_ELLIPTICAL = 'Horizontal Elliptical'
    VERTICAL_ELLIPTICAL = 'Vertical Elliptical'


class OutfallType:
    """Types of conduits."""
    FREE = 'FREE'
    NORMAL = 'NORMAL'
    FIXED = 'FIXED'
    TIDAL = 'TIDAL'
    TIMESERIES = 'TIMESERIES'


class FlowUnits:
    """Flow units."""
    CFS = 'CFS'
    GPM = 'GPM'
    MGD = 'MGD'
    CMS = 'CMS'
    LPS = 'LPS'
    MLD = 'MLD'


class RoutingModel:
    """Routing model."""
    STEADY = 'Steady Flow'
    KINWAVE = 'Kinematic Wave'
    DYNWAVE = 'Dynamic Wave'


class ForceMain:
    """Force main equation."""
    H_W = 'Hazen-Williams'
    D_W = 'Darcy-Weisbach'


class LengthUnits:
    """Length units."""
    FEET = 'ft'
    METERS = 'm'


class AreaUnits:
    """Length units."""
    SQ_FEET = 'ft²'
    SQ_METERS = 'm²'


def get_swmm_model() -> GenericModel:
    """Returns the model."""
    model = GenericModel(exclusive_point_conditions=True)
    _add_global_parameters(model)
    length_units, area_units, flow_units = _get_units(model)
    _add_arc_parameters(model, length_units, flow_units)
    _add_point_parameters(model, length_units, area_units, flow_units)
    return model


def _add_global_parameters(model: GenericModel):
    """Adds SWMM model control parameters."""
    section = model.global_parameters

    group = section.add_group('general', 'General')
    group.add_option('flow_units', 'Flow Units', FlowUnits.CFS,
                     [FlowUnits.CFS, FlowUnits.GPM, FlowUnits.MGD, FlowUnits.CMS, FlowUnits.LPS, FlowUnits.MLD],
                     'The choice of flow units determines whether US or metric units are used for all '
                     'other quantities. Default values are not automatically adjusted when the unit '
                     'system is changed from US to metric (or vice versa).')
    group.add_option('routing_model', 'Routing Model', RoutingModel.DYNWAVE,
                     [RoutingModel.STEADY, RoutingModel.KINWAVE, RoutingModel.DYNWAVE],
                     'This option determines which method is used to route flows through the conveyance '
                     'system.')
    group.add_boolean('allow_ponding', 'Allow Ponding', True,
                      'Checking this option will allow excess water to collect atop nodes and be '
                      're-introduced into the system as conditions permit. In order for ponding to '
                      'actually occur at a particular node, a non-zero value for its Ponded Area attribute '
                      'must be used.')

    group = section.add_group('dates', 'Dates')
    current_datetime = datetime.datetime.now()
    next_day_datetime = current_datetime + datetime.timedelta(days=1)
    group.add_date_time('start_analysis', 'Start Analysis On', current_datetime,
                        'Enter the date (month/day/year) and time of day when the simulation begins.')
    group.add_date_time('start_reporting', 'Start Reporting On', current_datetime,
                        'Enter the date and time of day when reporting of simulation results is to begin. '
                        'Using a date prior to the start date is the same as using the start date.')
    group.add_date_time('end_analysis', 'End Analysis On', next_day_datetime,
                        'Enter the date and time when the simulation is to end.')

    group = section.add_group('time_steps', 'Time Steps')
    group.add_float('reporting_time_step', 'Reporting Time Step (minutes)', 15.0, low=0.0,
                    description='[minutes] Time interval for reporting of computed results.')
    group.add_float('dry_weather_time_step', 'Dry Weather Runoff Time Step (minutes)', 60.0, low=0.0,
                    description='[minutes] Enter the time step length used for runoff computations during periods when '
                                'there is no rainfall, no ponded water, and LID controls are dry. This must be '
                                'greater or equal to the Wet Weather time step.')
    group.add_float('wet_weather_time_step', 'Wet Weather Runoff Time Step (minutes)', 5.0, low=0.0,
                    description='[minutes] Enter the time step length used to compute runoff during periods of '
                                'rainfall, or when ponded water still remains on the surface.')
    group.add_float('routing_time_step', 'Routing Time Step (seconds)', 20.0, low=0.0,
                    description='[seconds] Enter the time step length used for routing flows through the conveyance '
                                'system. Note that Dynamic Wave routing requires a much smaller time step than the '
                                'other methods of flow routing.')

    group = section.add_group('dynamic_wave', 'Dynamic Wave')
    group.add_option('force_main_equation', 'Force Main Equation', ForceMain.H_W, [ForceMain.H_W, ForceMain.D_W],
                     'Selects which equation will be used to compute friction losses during pressurized flow for '
                     'conduits that have been assigned a Circular Force Main cross-section.')


def _add_point_parameters(model: GenericModel, length_units: str, area_units: str, flow_units: str):
    """Adds SWMM node parameters."""
    section = model.point_parameters
    group = section.add_group('junction', 'Junction')
    group.add_text('name', 'Name', '', 'User-assigned junction name.', required=True)
    group.add_curve('inflows', 'Inflows', ['Time (Minutes)', f'Flow ({flow_units})'],
                    description=f'Time series of inflows in {flow_units} for the selected junction.')
    group.add_float('max_depth', f'Max Depth ({length_units})', 0.0,
                    description=f'[{length_units}] Maximum depth at the junction (i.e., the distance from the invert '
                                f'to the ground surface) ({length_units}). If zero, then the distance from the invert '
                                'to the top of the highest connecting link will be used.')
    group.add_float('surcharge_depth', f'Surcharge Depth ({length_units})', 0.0,
                    description=f'[{length_units}] Additional depth of water beyond the maximum depth that is allowed '
                                'before the junction floods. This parameter can be used to simulate bolted manhole '
                                'covers or force main connections.')
    group.add_float('ponded_area', f'Ponded Area ({area_units})', 0.0,
                    description=f'[{area_units}] Area occupied by ponded water atop the junction after flooding '
                                'occurs. If the ALLOW PONDING analysis option is turned on, a non-zero value of this '
                                'parameter will allow ponded water to be stored and subsequently returned to the '
                                'drainage system when capacity exists.')

    group = section.add_group('outfall', 'Outfall')
    group.add_text('name', 'Name', '', 'User-assigned outfall name.', required=True)
    group.add_curve('inflows', 'Inflows', ['Time (Minutes)', f'Flow ({flow_units})'],
                    description=f'Time series of inflows in {flow_units} for the selected junction.')
    group.add_boolean('tide_gate', 'Tide Gate Exists', False, 'Toggle on if a tide gate exists that prevents backflow, '
                                                              'or off if no tide gate exists.')
    opts = [OutfallType.FREE, OutfallType.NORMAL, OutfallType.FIXED, OutfallType.TIDAL, OutfallType.TIMESERIES]
    o = group.add_option('type', 'Type', OutfallType.FREE, opts,
                         "Type of outfall boundary condition: 1) FREE: outfall stage determined by minimum of critical "
                         "flow depth and normal flow depth in the connecting conduit. 2) NORMAL: outfall stage based "
                         "on normal flow depth in the connecting conduit. 3) FIXED: outfall stage set to a fixed "
                         "value. 4) TIDAL: outfall stage given by a table of tide elevation versus time of day. "
                         "5) TIMESERIES: outfall stage supplied from a time series of elevations.")
    flags = {}
    for opt in opts:
        flags[opt] = False
    p = group.add_float('fixed_stage', f'Fixed Stage ({length_units})', 1.0,
                        description=f'[{length_units}] Water elevation for a "Fixed" type of outfall.')
    cur_flags = _copy_flags(flags, [OutfallType.FIXED])
    p.add_dependency(o, cur_flags)
    p = group.add_curve('tidal_outfall_curve', 'Tidal Outfall Curve',
                        ['Time (Hour of Day)', f'Water Elevation ({length_units})'],
                        description=f'Tidal Curve relating water elevation in {length_units} to hour of the day for '
                                    'the selected tidal outfall.')
    cur_flags = _copy_flags(flags, [OutfallType.TIDAL])
    p.add_dependency(o, cur_flags)
    p = group.add_curve('timeseries_curve', 'Timeseries Curve',
                        ['Time (Minutes)', f'Water Elevation ({length_units})'],
                        description=f'Time series containing time history of outfall stage in {flow_units} for the '
                                    'selected outfall.')
    cur_flags = _copy_flags(flags, [OutfallType.TIMESERIES])
    p.add_dependency(o, cur_flags)


def _copy_flags(flags: dict, values: list):
    """Helper function to return a copy of flags dict with the given flags set to True."""
    flags_copy = flags.copy()
    for value in values:
        flags_copy[value] = True
    return flags_copy


def _get_units(model: GenericModel) -> tuple[str, str, str]:
    flow_units = model.global_parameters.group('general').parameter('flow_units').value
    length_units = LengthUnits.METERS
    area_units = AreaUnits.SQ_METERS
    if flow_units in [FlowUnits.CFS, FlowUnits.GPM, FlowUnits.MGD]:
        length_units = LengthUnits.FEET
        area_units = AreaUnits.SQ_FEET
    return length_units, area_units, flow_units


def _add_arc_parameters(model: GenericModel, length_units: str, flow_units: str):
    """Adds SWMM arc parameters."""
    section = model.arc_parameters
    group = section.add_group('conduit', 'Conduit')
    group.add_text('name', 'Name', '', 'User-assigned conduit name.', required=True)
    opts = [ConduitShape.CIRCULAR, ConduitShape.FILLED_CIRCULAR, ConduitShape.CLOSED_RECTANGULAR,
            ConduitShape.HORIZONTAL_ELLIPTICAL, ConduitShape.VERTICAL_ELLIPTICAL]
    o = group.add_option('shape', 'Shape', ConduitShape.CIRCULAR, opts,
                         "Change the geometric properties of the conduit's cross section.")
    group.add_integer('number_barrels', 'Number of Barrels', 1, 1, 100)
    group.add_float('height_diameter', f'Height/Diameter ({length_units})', 1.0,
                    description=f'[{length_units}] Conduit height or diameter.')
    p = group.add_float('width', f'Width ({length_units})', 1.0, description=f'[{length_units}] Conduit width.')
    flags = {}
    for opt in opts:
        flags[opt] = False
    cur_flags = _copy_flags(flags, [ConduitShape.CLOSED_RECTANGULAR, ConduitShape.HORIZONTAL_ELLIPTICAL,
                                    ConduitShape.VERTICAL_ELLIPTICAL])
    p.add_dependency(o, cur_flags)
    p = group.add_float('filled_depth', f'Filled Depth ({length_units})', 0.0,
                        description=f'[{length_units}] Depth of fill in the circular conduit.')
    cur_flags = _copy_flags(flags, [ConduitShape.FILLED_CIRCULAR])
    p.add_dependency(o, cur_flags)
    group.add_float('roughness', 'Roughness', 0.01, description="Manning's roughness coefficient.")
    group.add_float('inlet_offset', f'Inlet Offset ({length_units})', 0.0,
                    description=f'[{length_units}] Depth of the conduit invert above the node invert at the inlet end '
                                'of the conduit.')
    group.add_float('outlet_offset', f'Outlet Offset ({length_units})', 0.0,
                    description=f'[{length_units}] Depth of the conduit invert above the node invert at the outlet '
                                f'end of the conduit.')
    group.add_float('initial_flow', 'Initial Flow', 0.0, description=f'[{flow_units}] Initial flow in the conduit at '
                                                                     'the start of the simulation.')
    group.add_float('max_flow', 'Maximum Flow', 0.0, description=f'[{flow_units}] Maximum flow allowed in the conduit '
                                                                 '- use 0 if not applicable.')
    group.add_float('entry_loss_coeff', 'Entry Loss Coefficient', 0.0,
                    description='Head loss coefficient associated with energy losses at the entrance of the conduit.')
    group.add_float('exit_loss_coeff', 'Exit Loss Coefficient', 0.0,
                    description='Head loss coefficient associated with energy losses at the exit of the conduit.')
    group.add_float('avg_loss_coeff', 'Avg. Loss Coefficient', 0.0,
                    description='Head loss coefficient associated with energy losses along the length of the conduit.')
    group.add_boolean('flap_gate', 'Flap Gate Exists', False, 'Toggle on if a flap gate exists that prevents backflow '
                                                              'through the conduit, or off if no flap gate exists.')
