"""
Test fixtures for XMS Python packages.

If you want to use these fixtures, ensure xmstesting is installed and then list the desired fixture(s) in your test
function's parameters, just like any other fixture. It's probably best to only list this as a dependency in your Tox
file so it doesn't get included in XMS itself.
"""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
import os
from pathlib import Path
import shutil
from typing import Callable, TypeAlias

# 2. Third party modules
import pytest

# 3. Aquaveo modules
try:
    from xms.constraint.grid_reader import read_grid_from_file
    from xms.constraint.grid import Grid as CoGrid
except ImportError:
    read_grid_from_file = None
    CoGrid: TypeAlias = None

# 4. Local modules


@pytest.fixture(scope='session')
def running_multiple_tests(request) -> bool:
    """
    Check whether there are multiple tests in the current batch of tests.

    This is mainly used as a heuristic for dialog tests to decide whether they should leave the dialog open after the
    test finishes. If multiple tests are running, it's likely either the CI or a mass test to see what fails, and
    leaving dialogs open would be irritating at best. If only a single test is running though, it's probably being run
    manually because it failed and needs to be debugged, in which case leaving the dialog open is more likely to be a
    slight nuisance at worst.

    Some packages use a `running_tox` fixture instead. If you run PyTest directly, that fixture gives the OK to leave up
    tons of dialogs. Running outside of Tox can be nice because it's faster, and PyCharm integrates with it to let you
    run a subset of tests (like "only feedback tests", or "just the tests that failed last time").
    """
    # request.session.items is a list of all the tests that pytest decided should be run.
    return len(request.session.items) > 1


@pytest.fixture
def module_directory(request) -> Path:
    """
    Get the module-level directory for the test files for this test.

    If your test function can be imported like `from tests.test_package.test_module import test_function`
    then this fixture will give you the path `./files/package/module` relative to the `tests` directory.
    """
    directory = _test_dir(request.path, request.node.originalname).parent
    directory.mkdir(exist_ok=True, parents=True)
    return directory


@pytest.fixture
def test_directory(request) -> Path:
    """
    Get the path to the files for this test.

    If your test function can be imported like `from tests.test_package.test_module import test_function`
    then this fixture will give you the path `./files/package/module/function` relative to the `tests` directory.
    """
    directory = _test_dir(request.path, request.node.originalname)
    directory.mkdir(exist_ok=True, parents=True)
    return directory


@pytest.fixture
def empty_temp_directory(test_directory) -> Path:
    """
    Get a temp directory for the test.

    The directory will be located inside the test's directory, and be named `temp`. The directory will exist.
    """
    temp_directory = test_directory / 'temp'
    with contextlib.suppress(FileNotFoundError):
        shutil.rmtree(temp_directory)
    os.makedirs(temp_directory)
    return temp_directory


@pytest.fixture
def temp_file(empty_temp_directory) -> Callable[[str], Path]:
    """
    Get a generator for temp files for testing.

    Temp files will be in the test's temp folder.

    The fixture provides the test with a callable function that takes the name of a file and returns the absolute path
    to the test's temp folder with the provided name appended. The name may contain directory separators and, if so, the
    returned path will be a subdirectory of the temp folder.

    The parent of the returned file (whether that is the temp directory itself or a subdirectory thereof) is guaranteed
    to exist after the function returns. This means that after calling `temp_file('path/to/file.txt')`,
    '<temp_dir>/path/to' will exist, but '<temp_dir>/path/to/file.txt' will not.

    By default, the helper returns `pathlib.Path` objects. Passing `path=False` will make it return a string instead.

    If `touch=True` is passed to the helper, it will create the file before returning its path. The created file will
    be empty (0 bytes).
    """
    def helper(name: str, path: bool = True, touch: bool = False) -> str | Path:
        file: Path = empty_temp_directory / name
        file.parent.mkdir(parents=True, exist_ok=True)
        if touch:
            file.touch()
        return file if path else str(file)

    return helper


@pytest.fixture
def empty_work_directory(test_directory):
    """
    Get a work directory for the test.

    The directory will be located inside the test's directory, and be named `work`. The directory will exist.
    Requesting this fixture will change the current directory to the work directory.
    """
    work_directory = test_directory / 'work'
    with contextlib.suppress(FileNotFoundError):
        shutil.rmtree(work_directory)
    os.makedirs(work_directory)

    old_dir = os.getcwd()
    os.chdir(work_directory)

    yield work_directory

    os.chdir(old_dir)


@pytest.fixture
def output_path(test_directory) -> Callable[[str], Path]:
    """
    Get a generator for output files/folders for testing.

    Output paths will be in the test's directory.

    The fixture provides the test with a callable function that takes the name of a file/folder and appends it to the
    test directory. The name may contain directory separators; if so, the returned path will be in a subdirectory.

    All parents of the paths returned by the callable will exist by the time the callable returns. The last name in the
    returned path will *not* exist. If it exists at the time the callable is called, it will be deleted. This ensures
    tests don't silently pass just because a good output file from a previous run (or that was accidentally included in
    source control) wasn't cleaned up.

    ```
    def test_something(output_path):
        file = output_path('some_file.txt')
        assert file.parent.exists()
        assert not file.exists()
    ```
    """
    def helper(name: str) -> Path:
        file: Path = test_directory / name
        file.parent.mkdir(parents=True, exist_ok=True)
        if file.exists() and file.is_dir():
            shutil.rmtree(file)
        elif file.exists():
            file.unlink()

        return file

    return helper


@pytest.fixture
def test_grid(test_directory) -> CoGrid:
    """
    Fixture that provides a constrained UGrid from a file in the test's directory.

    This fixture expects the grid to be in a file named `grid.xmc` in the test's directory. If that file does not exist
    or is not a valid .xmc file, the fixture will raise an assertion error which should fail the test.
    """
    assert read_grid_from_file is not None, 'xms.constraint not installed. Is it listed in your dependencies?'

    grid_file = test_directory / 'grid.xmc'
    grid = read_grid_from_file(str(grid_file))

    # read_grid_from_file returns None if there's any sort of problem at all. Odds are the test will fail anyway due to
    # this, so make it fail close to the problem.
    assert grid is not None, f'"{grid_file}" did not exist or was not a valid constrained UGrid file.'

    return grid


def _test_dir(request_path: Path, test_name: str) -> Path:
    """
    Find the test directory for a particular test.

    The directory may or may not exist after this function returns.
    """
    tests_dir = request_path
    while tests_dir.name != 'tests' and tests_dir.parent != tests_dir:
        tests_dir = tests_dir.parent
    dirty_test_path = request_path.relative_to(tests_dir).with_suffix('') / test_name
    parts = [part.removeprefix('test_') for part in dirty_test_path.parts]
    test_directory = tests_dir.joinpath('files', *parts)
    return test_directory
