"""Qt dialog for viewing tidal constituent component data."""

__copyright__ = '(C) Copyright Aquaveo 2025'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import datetime
import webbrowser

# 2. Third party modules
from harmonica.resource import ResourceManager
from harmonica.tidal_constituents import Constituents
from PySide2.QtCore import QDateTime
from PySide2.QtWidgets import QSizePolicy, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.time_format import datetime_to_qdatetime, ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules
from xms.tides.data import tidal_data
from xms.tides.gui import gui_util
from xms.tides.gui.constituent_list import ConstituentListWidget
from xms.tides.gui.tidal_dlg_ui import Ui_TidalDlg
from xms.tides.gui.user_defined_table import (
    COL_AMP, COL_CONSTITUENT, COL_EQUILIBRIUM_ARGUMENT, COL_ETRF, COL_FORCE_VELOCITY, COL_FREQUENCY, COL_NAME,
    COL_NODAL_FACTOR, COL_PHASE, COL_POTENTIAL_AMPLITUDE, COL_VEL_AMP_X, COL_VEL_AMP_Y, COL_VEL_PHASE_X,
    COL_VEL_PHASE_Y, UserDefinedWidget
)


class TidalDlg(XmsDlg):
    """A dialog showing the Tidal Database options."""

    def __init__(self, data, pe_tree, win_cont):
        """Initializes the sediment material list and properties dialog.

        Args:
            data (TidalData): Dataset containing the tidal database parameters
            pe_tree (xms.api.tree.tree_node.TreeNode): The XMS project explorer tree
            win_cont (QWidget): Parent dialog
        """
        super().__init__(win_cont, 'xms.tides.gui.tidal_dlg')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Tidal_Constituents'
        self.data = data
        self.pe_tree = pe_tree
        self.source = 0  # Just to prevent refetching constituent list if not needed
        self.cons_list = None
        self.user_defined_table = None
        self.con_model = None
        self.available_models = ResourceManager.available_models()
        self.ui = Ui_TidalDlg()
        self.ui.setupUi(self)

        # Setup the dialog
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Populate comboboxes
        self._add_cbx_source(self.ui.cbx_source)

        # Add the constituent list
        self._add_constituent_list()

        # Add the constituent table widget
        self._add_userdefined_table()

        # Link up the help URL
        self.ui.button_box.helpRequested.connect(self.help_requested)

        # Connect slots/signals
        self.ui.cbx_source.currentIndexChanged.connect(self.on_source_changed)
        # Populate widgets from data
        self._load_data()
        self.on_source_changed(self.ui.cbx_source.currentIndex())

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def on_source_changed(self, index):
        """Called when the source combobx is changed."""
        #  Hide/show the user defined table and List view of the selected tidal database
        self.ui.grp_user.setVisible(index == tidal_data.USER_DEFINED_INDEX)
        self.ui.grp_cons.setVisible(index != tidal_data.USER_DEFINED_INDEX)
        if index != tidal_data.USER_DEFINED_INDEX and index != self.source:
            #  populate the constituents based on the selected dbase
            self.source = index
            if self.con_model is None:
                self.con_model = Constituents(self._get_harmonica_model_string(self.source))
            else:
                self.con_model.change_model(self._get_harmonica_model_string(index))
            df = self.data.get_default_con_dataset(self.con_model).to_dataframe()
            gui_util.fix_df_column_names_for_gui(df)
            self.cons_list.change_model(df)
        #  Hide/show the license notification based on the database source
        if index in [tidal_data.FES2014_INDEX, tidal_data.TPX08_INDEX, tidal_data.TPX09_INDEX]:
            model_name = self._get_harmonica_model_string(index)
            if not self.available_models[model_name]:
                self.ui.date_reftime.setEnabled(False)
                self.ui.grp_cons.setEnabled(False)
                model_name = model_name.upper()
                self.ui.lbl_license.setText(
                    f'The selected tidal harmonic model cannot be used without a valid {model_name} license. Click the '
                    f'"Help" button for instructions on installing the {model_name} model.'
                )
                self.ui.lbl_license.setStyleSheet('QLabel { color : red; }')
                self.ui.lbl_license.setVisible(True)
            else:
                self.ui.date_reftime.setEnabled(True)
                self.ui.grp_cons.setEnabled(True)
                self.ui.lbl_license.setVisible(False)
        else:
            self.ui.date_reftime.setEnabled(True)
            self.ui.grp_cons.setEnabled(True)
            self.ui.lbl_license.setVisible(False)

    @staticmethod
    def _add_cbx_source(cbx):
        """Add a combobox's option texts and user data values.

        Args:
            cbx (QCombobox): The combobox widget.
        """
        for idx, opt in enumerate(tidal_data.TDB_SOURCES):  # Loop through the list of display option texts
            cbx.addItem(opt, idx)  # Add the index value as UserData

    @staticmethod
    def _set_cbx_source_from_data(cbx, value):
        """Set a combobox's  current option index based on its user data value.

        Args::
            cbx (QCombobox): The combobox widget.
            value (int): Data associated with the option to set
        """
        index = cbx.findData(value)
        if index != -1:
            cbx.setCurrentIndex(index)

    def _add_constituent_list(self):
        """Add the constituent list widget."""
        layout = QVBoxLayout()
        self.source = int(self.data.info.attrs['source'])
        source = self.source
        if self.source == tidal_data.USER_DEFINED_INDEX:
            source = tidal_data.LEPROVOST_INDEX
        self.con_model = Constituents(self._get_harmonica_model_string(source))
        df = self.data.cons.to_dataframe()
        if df.empty:
            df = self.data.get_default_con_dataset(self.con_model).to_dataframe()
        gui_util.fix_df_column_names_for_gui(df)
        self.cons_list = ConstituentListWidget(df, self)
        layout.addWidget(self.cons_list)
        self.ui.grp_cons.setLayout(layout)

    def _add_userdefined_table(self):
        """Add the user defined table widget."""
        layout = QVBoxLayout()
        df = self.data.user.to_dataframe()
        gui_util.fix_df_column_names_for_gui(df)
        reftime = datetime.datetime.strptime(self.data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        columns = [
            COL_CONSTITUENT, COL_NAME, COL_POTENTIAL_AMPLITUDE, COL_FREQUENCY, COL_NODAL_FACTOR,
            COL_EQUILIBRIUM_ARGUMENT, COL_ETRF, COL_AMP, COL_PHASE, COL_FORCE_VELOCITY, COL_VEL_AMP_X, COL_VEL_PHASE_X,
            COL_VEL_AMP_Y, COL_VEL_PHASE_Y
        ]
        self.user_defined_table = UserDefinedWidget(df, self.pe_tree, reftime, columns, parent=self)
        layout.addWidget(self.user_defined_table)
        self.ui.grp_user.setLayout(layout)
        size_policy = self.ui.grp_user.sizePolicy()
        size_policy.setVerticalPolicy(QSizePolicy.MinimumExpanding)
        self.ui.grp_user.setSizePolicy(size_policy)

    @staticmethod
    def _get_harmonica_model_string(index):
        """Get the harmonica model string for a database source.

        Args:
            index (int): Combobox index of the database source option

        Returns:
            str: See description
        """
        if index == tidal_data.ADCIRC_INDEX:
            return 'adcirc2015'
        elif index == tidal_data.LEPROVOST_INDEX:
            return 'leprovost'
        elif index == tidal_data.FES2014_INDEX:
            return 'fes2014'
        elif index == tidal_data.TPX08_INDEX:
            return 'tpxo8'
        elif index == tidal_data.TPX09_INDEX:
            return 'tpxo9'

    def _load_data(self):
        """Populate widgets from persistent data."""
        self._set_cbx_source_from_data(self.ui.cbx_source, self.source)
        reftime = self.data.info.attrs['reftime']
        if not reftime:
            reftime = QDateTime.currentDateTime()
        else:
            reftime = datetime.datetime.strptime(reftime, ISO_DATETIME_FORMAT)
            reftime = datetime_to_qdatetime(reftime)
        self.ui.date_reftime.setDateTime(reftime)
        self.ui.date_reftime.dateTimeChanged.connect(self.user_defined_table.on_ref_date_changed)

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        df = self.user_defined_table.model.data_frame
        gui_util.fix_df_column_names_for_io(df)
        dset = df.to_xarray()
        self.data.user = dset
        self.data.cons = self.cons_list.get_dataset()

        self.data.info.attrs['source'] = self.ui.cbx_source.itemData(self.ui.cbx_source.currentIndex())
        reftime = qdatetime_to_datetime(self.ui.date_reftime.dateTime())
        self.data.info.attrs['reftime'] = reftime.strftime(ISO_DATETIME_FORMAT)
