"""Cell 2D Quality Metrics functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numba
from numba.typed import List
import numpy as np

# 3. Aquaveo modules

# 4. Local modules


# It's surprising this isn't defined in numpy, but it should help eek out
# a little bit of performance.
@numba.jit
def minmax(my_array):
    """Return minimum and maximum of numba.typed List my_array found in a single pass."""
    maximum = my_array[0]
    minimum = my_array[0]
    for i in my_array[1:]:
        if i > maximum:
            maximum = i
        elif i < minimum:
            minimum = i
    return minimum, maximum


# Constants and Enums from xmlutl/ugrid/UgQuality.h
XM_QUALITY_UNSUPPORTED_GEOMETRY = -888.0   # Unsupported geometry quality value
XM_QUALITY_UNDEFINED = -999.0              # Undefined quality value

# Constants for Activity
XM_ACTIVITY_INVALID = 0
XM_ACTIVITY_VALID = 1

# Supported UGrid quality types.
#
# The [XMS quality types](http://www.xmswiki.com/wiki/SMS:ARR_Mesh_Quality_Assessment_Plot).
# The Verdict types are documented in
# [The Verdict Library Reference Manual](http://www.vtk.org/Wiki/images/6/6b/VerdictManual-revA.pdf)
# by C.J. Stimpson, C.D. Ernst, P. Knupp, P.P. Pebay, and D. Thompson.
#                                                                           Tri Pg#    Quad Pg#
# See also xms/trunk/Dev/Shared/xmsutil/ugrid/UgQuality.{h|cpp}             -------    --------

XM_QUALITY_ALPHA_MIN = 0   # Minimum interior angle (XMS)
XM_QUALITY_LMAX_LMIN = 1   # Edge length ratio (XMS)
XM_QUALITY_ALS = 2         # Area/total edge length squared ratio (XMS)
XM_QUALITY_RR = 3          # Inner/outer radius ratio (XMS)
XM_QUALITY_LR = 4          # Inner radius/maximum length ratio (XMS)
XM_QUALITY_LH = 5          # Minimum height/maximum length ratio (XMS)
XM_QUALITY_CONDITION = 6   # Condition (Verdict)
XM_QUALITY_SHEAR = 7       # Shear (Verdict)
XM_QUALITY_END = 8         # End of the enum (not a metric)


# Some constants of convenience
ROOT_TWO = np.sqrt(2.0)
TAU = 2.0 * np.pi
FOUR_OVER_TAU = 4.0 / TAU  # The True Circle Constant, darn it!
ROOT_THREE = np.sqrt(3.0)
TWO_ROOT_THREE = 2.0 * ROOT_THREE
FOUR_ROOT_THREE = 4.0 * ROOT_THREE
TWO_OVER_ROOT_THREE = 2.0 / ROOT_THREE
THREE_OVER_PI = 3.0 / np.pi


# Floating point constants:
VERDICT_DBL_MIN = 1.0e-30
VERDICT_DBL_MAX = 1.0e+30


@numba.jit(nopython=True)
def angle_between_edges(pt1, pt2, pt3):
    """Compute the angle between 2 edges.

    Args:
        pt1 (ndarray): The first point.
        pt2 (ndarray): The second point.
        pt3 (ndarray): The third point.

    Returns:
        (float):       The angle
        (pt ndarray):  The cross product
    """
    dxp = pt1[0] - pt2[0]
    dyp = pt1[1] - pt2[1]
    dxn = pt3[0] - pt2[0]
    dyn = pt3[1] - pt2[1]
    magn = np.sqrt((dxn ** 2) + (dyn ** 2))
    magp = np.sqrt((dxp ** 2) + (dyp ** 2))
    if (magn == 0.0) or (magp == 0.0):
        return 0.0
    cos_angle = ((dxn * dxp) + (dyn * dyp)) / (magn * magp)
    cos_angle = min(cos_angle, 1.0)
    cos_angle = max(cos_angle, -1.0)
    the_angle = np.arccos(cos_angle)
    if ((dxp * dyn) - (dxn * dyp)) < 0.0:  # cross product
        the_angle = TAU - the_angle
    return the_angle


@numba.jit(nopython=True)
def triangle_quality(
        pt1, pt2, pt3,
        selected_metrics=False,
        quality_alpha_min=False,
        quality_lmax_lmin=False,
        quality_als=False,
        quality_rr=False,
        quality_lr=False,
        quality_lh=False,
        quality_condition=False,
        quality_shear=False):
    """Compute quality measures of a triangle.

    Args: three points and optional keyword bools:
        pt1 (ndarray): The first point.
        pt2 (ndarray): The second point.
        pt3 (ndarray): The third point.

        selected_metrics  (optional bool):  False if all non-verdict qualities desired else True
        quality_alpha_min (optional bool):  True if desired else False
        quality_lmax_lmin (optional bool):  True if desired else False
        quality_als       (optional bool):  True if desired else False
        quality_rr        (optional bool):  True if desired else False
        quality_lr        (optional bool):  True if desired else False
        quality_lh        (optional bool):  True if desired else False
        quality_condition (optional bool):  True if desired else False
        quality_shear     (optional bool):  True if desired else False

        Ideally, all these should be passed in as a single "desired_metrics" set, with default
        "{XM_QUALITY_END}", and then run selected measurements only if "desired_metrics"
        doesn't have "XM_QUALITY_END" as a value ... but the Numba optimizing compiler doesn't
        handle optional sets very well.

    Returns:
        (q_alpha_min, q_ll, q_als, q_rr, q_lr, q_lh, q_condition, q_shear): The XMS and Verdict quality metrics.
    """
    # my_desired_metrics = set(desired_metrics) if desired_metrics is not None else {XM_QUALITY_END}

    # compute the three angles for this element (alpha, beta, gamma)
    angle_1 = angle_between_edges(pt3, pt2, pt1)
    angle_2 = angle_between_edges(pt1, pt3, pt2)
    angle_3 = angle_between_edges(pt2, pt1, pt3)

    gamma, alpha = minmax(List([angle_1, angle_2, angle_3]))
    beta = np.pi - alpha - gamma

    # compute the components for quality measures for this element
    sin_alpha = np.sin(alpha)
    sin_beta = np.sin(beta)
    sin_gamma = np.sin(gamma)
    # scaled lengths of the triangle edges (assume l_alpha = 1)
    l_beta = sin_beta / sin_alpha
    l_gamma = sin_gamma / sin_alpha
    # scaled minimum height of the triangle
    h_min = l_gamma * sin_beta
    # scaled area
    double_area = h_min
    double_area_root_three = double_area * ROOT_THREE

    # outer radius
    outer_radius = 0.5 / sin_alpha
    # inner diameter (eg, twice inner radius)
    inner_diameter = double_area / (0.5 * (1 + l_beta + l_gamma))

    # calculate scaled dot(L1, L1), dot(L2, L2)
    l_beta_sq = l_beta ** 2
    l_gamma_sq = l_gamma ** 2

    if quality_condition or quality_shear:
        # Perhaps the area should be compared to VERDICT_DBL_MIN * 2.0,
        # however, "double_area", "l_beta", and "l_gamma" are all scaled,
        # so this is probably not as comparable to the Quadrilateral case anyway.
        if quality_condition:
            if double_area < VERDICT_DBL_MIN:
                condition = VERDICT_DBL_MAX
            else:
                # Because we don't have the coordinates for the legs LL_beta and LL_gamma,
                # we cannot find their dot product directly; however,
                # \[    dot(LL_beta, LL_gamma) == l_beta * l_gamma * np.cos(alpha)    \]
                # where l_beta is the length of LL_beta, etc, and alpha is the angle between the two.
                #
                # Note that this definition is conflicts with the Verdict description of the
                # various metrics, but is aligned to the Verdict C++ code:  the Verdict description
                # adds the dot product, rather than subtracts it.
                #
                # Verdict documentation footnotes these two papers, which can be found at:
                # - [Achieving Finite Element Mesh Quality via ...](https://www.osti.gov/servlets/purl/5009)
                # - [Algebraic Mesh Quality Metrics](https://www.osti.gov/servlets/purl/754328)
                # but it's not immediately obvious from either paper how the condition ought to be defined.
                #
                # Fun fact:  at least for triangles, condition and als seem to be inverses of each other;
                # ie, condition == 1.0 / als, despite being calculated in different ways.
                condition = (l_beta_sq + l_gamma_sq - (l_beta * l_gamma * np.cos(alpha))) / double_area_root_three

        if quality_shear:
            # If we'd like to define "shear" for triangles, this is probably a good definition.
            # This definition compares the largest possible equilateral triangle with the actual area
            # of the triangle.  This would be analogous to how "shear" is defined for quadrilaterals,
            # which takes the smallest of the four possible ratios of each adjascent pairs of sides with
            # the possible rectangle defined by those two sides.
            #
            # Just as the quadrilateral "shear" is closer to 0 the more squished the smallest parallelogram
            # gets, this measure gets closer to 0 the more squished a triangle gets.
            #
            # This test may be a little too "sensitive", depending on how squished a triangle can get
            # before it's considered "too sheared".  Perhaps it will be useful to compare a sheared triangle
            # to a similarly-sheared quadrilateral.
            #
            # if (double_area < VERDICT_DBL_MIN) or (min(l_beta, l_gamma) < VERDICT_DBL_MIN):
            #     shear = 0.0
            # else:
            #     # the area of the triangle over the area of the largest possible equilateral triangle;
            #     # recall that alpha is the smallest angle, so that the largest side of this scaled
            #     # triangle is "l_alpha" == 1, so the area is (l_alpha ** 2 * ROOT_THREE / 2.0)
            #     # or (double_area / (ROOT_THREE / 2.0)) or
            #     shear = double_area * TWO_OVER_ROOT_THREE
            shear = XM_QUALITY_UNSUPPORTED_GEOMETRY

    # compute the quality measures for this element
    if selected_metrics:
        q_alpha_min = (THREE_OVER_PI * gamma) if quality_alpha_min else XM_QUALITY_UNDEFINED
        q_ll = l_gamma if quality_lmax_lmin else XM_QUALITY_UNDEFINED
        q_als = (2.0 * double_area_root_three / (1 + l_beta_sq + l_gamma_sq)) if quality_als else XM_QUALITY_UNDEFINED
        q_rr = (inner_diameter / outer_radius) if quality_rr else XM_QUALITY_UNDEFINED
        q_lr = (ROOT_THREE * inner_diameter) if quality_lr else XM_QUALITY_UNDEFINED
        q_lh = (TWO_OVER_ROOT_THREE * h_min) if quality_lh else XM_QUALITY_UNDEFINED
        q_condition = condition if quality_condition else XM_QUALITY_UNDEFINED
        q_shear = shear if quality_shear else XM_QUALITY_UNDEFINED
    else:
        q_alpha_min = (THREE_OVER_PI * gamma)
        q_ll = l_gamma
        q_als = (2.0 * double_area_root_three / (1 + (l_beta_sq) + (l_gamma_sq)))
        q_rr = (inner_diameter / outer_radius)
        q_lr = (ROOT_THREE * inner_diameter)
        q_lh = (TWO_OVER_ROOT_THREE * h_min)
        q_condition = XM_QUALITY_UNDEFINED
        q_shear = XM_QUALITY_UNDEFINED

    return q_alpha_min, q_ll, q_als, q_rr, q_lr, q_lh, q_condition, q_shear


@numba.jit(nopython=True)
def quad_quality(
        pt0, pt1, pt2, pt3,
        selected_metrics=False,
        quality_alpha_min=False,
        quality_lmax_lmin=False,
        quality_als=False,
        quality_rr=False,
        quality_lr=False,
        quality_lh=False,
        quality_condition=False,
        quality_shear=False):
    """Compute quality measures of a quadrilateral.

    Args: four points and optional keyword bools:
        pt0 (ndarray): The first point.
        pt1 (ndarray): The second point.
        pt2 (ndarray): The third point.
        pt3 (ndarray): The fourth point.

        selected_metrics  (optional bool):  False if all non-verdict qualities desired else True
        quality_alpha_min (optional bool):  True if desired else False
        quality_lmax_lmin (optional bool):  True if desired else False
        quality_als       (optional bool):  True if desired else False
        quality_rr        (optional bool):  True if desired else False
        quality_lr        (optional bool):  True if desired else False
        quality_lh        (optional bool):  True if desired else False
        quality_condition (optional bool):  True if desired else False
        quality_shear     (optional bool):  True if desired else False

        Ideally, all these should be passed in as a single "desired_metrics" set, with default
        "{XM_QUALITY_END}", and then run selected measurements only if "desired_metrics"
        doesn't have "XM_QUALITY_END" as a value ... but the Numba optimizing compiler doesn't
        handle optional sets very well.

    Returns:
        (q_alpha_min, q_ll, q_als, q_rr, q_lr, q_lh, q_condition, q_shear): The XMS and Verdict quality metrics.
    """
    # Precalculate some constants related to quadrilateral

    # Find the four angles between the legs of this element
    theta0 = angle_between_edges(pt1, pt0, pt3)
    theta1 = angle_between_edges(pt2, pt1, pt0)
    theta2 = angle_between_edges(pt3, pt2, pt1)
    theta3 = angle_between_edges(pt0, pt3, pt2)
    delta, alpha = minmax(List([theta0, theta1, theta2, theta3]))

    # Compute the four edge lengths
    l01_sq = ((pt0[0] - pt1[0]) ** 2) + ((pt0[1] - pt1[1]) ** 2)
    l12_sq = ((pt1[0] - pt2[0]) ** 2) + ((pt1[1] - pt2[1]) ** 2)
    l23_sq = ((pt2[0] - pt3[0]) ** 2) + ((pt2[1] - pt3[1]) ** 2)
    l30_sq = ((pt3[0] - pt0[0]) ** 2) + ((pt3[1] - pt0[1]) ** 2)

    l01 = np.sqrt(l01_sq)  # l0 - an alias used in Verdict documentation
    l12 = np.sqrt(l12_sq)  # l1
    l23 = np.sqrt(l23_sq)  # l2
    l30 = np.sqrt(l30_sq)  # l3

    l_min, l_max = minmax(List([l01, l12, l23, l30]))

    # Compute the four midpoints
    mpt01 = ((pt0[0] + pt1[0]) / 2.0, (pt0[1] + pt1[1]) / 2.0,)
    mpt12 = ((pt1[0] + pt2[0]) / 2.0, (pt1[1] + pt2[1]) / 2.0,)
    mpt23 = ((pt2[0] + pt3[0]) / 2.0, (pt2[1] + pt3[1]) / 2.0,)
    mpt30 = ((pt3[0] + pt0[0]) / 2.0, (pt3[1] + pt0[1]) / 2.0,)
    # Compute the length of the two bisectors
    lbisect1 = np.sqrt(((mpt01[0] - mpt23[0]) ** 2) + ((mpt01[1] - mpt23[1]) ** 2))
    lbisect2 = np.sqrt(((mpt12[0] - mpt30[0]) ** 2) + ((mpt12[1] - mpt30[1]) ** 2))
    # Compute the lengths of the two diagonals
    ldiag1 = np.sqrt(((pt0[0] - pt2[0]) ** 2) + ((pt0[1] - pt2[1]) ** 2))
    ldiag2 = np.sqrt(((pt1[0] - pt3[0]) ** 2) + ((pt1[1] - pt3[1]) ** 2))

    # Compute the area
    s = (l01 + l12 + l23 + l30) / 2.0
    area = np.sqrt((s - l01) * (s - l12) * (s - l23) * (s - l30) - l01 * l12 * l23 * l30 * (
                   (np.cos((theta0 + theta2) / 2.0)) ** 2))

    # Inner diameter (the shortest bisector == 2 * r (inner radius))
    inner_diameter = min(lbisect1, lbisect2)
    # Outer diameter (the largest diagonal == 2 * R (outer radius))
    outer_diameter = max(ldiag1, ldiag2)

    # Compute the quality measures for this element
    if selected_metrics:
        if quality_condition or quality_shear:
            # Calculate "quadrant" signed areas of the quadrilateral
            l30xl01 = l30 * l01
            l01xl12 = l01 * l12
            l12xl23 = l12 * l23
            l23xl30 = l23 * l30

            area0 = l30xl01 * np.sin(theta0)
            area1 = l01xl12 * np.sin(theta1)
            area2 = l12xl23 * np.sin(theta2)
            area3 = l23xl30 * np.sin(theta3)

            min_area = min(area0, area1, area2, area3)

            # Calculate condition or shear
            if quality_condition:
                if min_area < VERDICT_DBL_MIN:
                    condition = VERDICT_DBL_MAX
                else:
                    # note that
                    # \[    (l_ij * l_jk) * np.sin(theta_j) = area_j    \]
                    condition = 0.5 * max((l01_sq + l30_sq) / area0,
                                          (l12_sq + l01_sq) / area1,
                                          (l23_sq + l12_sq) / area2,
                                          (l30_sq + l23_sq) / area3, )

            if quality_shear:
                if (min_area < VERDICT_DBL_MIN) or (min(l01, l12, l23, l30) < VERDICT_DBL_MIN):
                    shear = 0.0
                else:
                    shear = min(area0 / l30xl01,
                                area1 / l01xl12,
                                area2 / l12xl23,
                                area3 / l23xl30, )

        q_alpha_min = delta * FOUR_OVER_TAU if quality_alpha_min else XM_QUALITY_UNDEFINED
        q_ll = l_min / l_max if quality_lmax_lmin else XM_QUALITY_UNDEFINED
        q_als = 4.0 * area / (l01_sq + l12_sq + l23_sq + l30_sq) if quality_als else XM_QUALITY_UNDEFINED
        q_rr = ROOT_TWO * inner_diameter / outer_diameter if quality_rr else XM_QUALITY_UNDEFINED
        q_lr = inner_diameter / l_max if quality_lr else XM_QUALITY_UNDEFINED
        q_lh = l_min * np.sin(np.pi - alpha) / l_max if quality_lh else XM_QUALITY_UNDEFINED
        q_condition = condition if quality_condition else XM_QUALITY_UNDEFINED
        q_shear = shear if quality_shear else XM_QUALITY_UNDEFINED
    else:
        q_alpha_min = delta * FOUR_OVER_TAU
        q_ll = l_min / l_max
        q_als = 4.0 * area / (l01_sq + l12_sq + l23_sq + l30_sq)
        q_rr = ROOT_TWO * inner_diameter / outer_diameter
        q_lr = inner_diameter / l_max
        q_lh = l_min * np.sin(np.pi - alpha) / l_max
        q_condition = XM_QUALITY_UNDEFINED
        q_shear = XM_QUALITY_UNDEFINED

    return q_alpha_min, q_ll, q_als, q_rr, q_lr, q_lh, q_condition, q_shear
