"""ImportProjectTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import Argument, IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.schism import read_horizontal_grid_file


class ImportHorizontalGridTool(Tool):
    """Tool to import a SCHISM horizontal grid file."""
    ARG_INPUT_FILE = 0
    ARG_NEW_GRID = 1

    def __init__(self, name='Import Horizontal Grid as UGrid'):
        """Initializes the class."""
        super().__init__(name=name)
        self._force_ugrid = True
        self._geom_txt = 'UGrid'

    def initial_arguments(self) -> list[Argument]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            A list of the initial tool arguments.
        """
        arguments = [
            self.file_argument(name='input_file', description='Input file', file_filter='Horizontal Grid (*.gr3)'),
            self.grid_argument(name='new_grid', description=f'New {self._geom_txt}', io_direction=IoDirection.OUTPUT,
                               optional=True),
        ]
        return arguments

    def validate_arguments(self, arguments: list[Argument]) -> dict[str, str]:
        """Validate the tool arguments.

        Args:
            arguments (list): The tool arguments.

        Returns:
            A dictionary of argument errors.
        """
        return {}

    def run(self, arguments: list[Argument]) -> None:
        """Run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        input_file = Path(arguments[self.ARG_INPUT_FILE].text_value)
        with open(input_file, 'r') as file:
            grid = read_horizontal_grid_file(file, self.logger)
            if not arguments[self.ARG_NEW_GRID].text_value:
                arguments[self.ARG_NEW_GRID].value = Path(input_file).stem
            self.set_output_grid(grid, arguments[self.ARG_NEW_GRID], force_ugrid=self._force_ugrid)
