"""Xmstool Argument class."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import ABC, abstractmethod
import copy
import enum
import os
from typing import List, Optional, Union

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class IoDirection(enum.IntEnum):
    """IO Direction of argument data (input or output)."""
    INPUT = 1
    OUTPUT = 2


IoDirectionArg = Union[IoDirection, int]


class Argument(ABC):
    """Abstract base class for tool arguments."""

    NONE_SELECTED = '-- None Selected --'

    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: IoDirectionArg = IoDirection.INPUT,
        optional: bool = False,
        value: object = None,
        hide: bool = False
    ):
        """Construct a tool argument.

        Args:
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (Optional[object]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
        """
        self.original_precedence = None
        self.name = name
        self.description = description
        self.io_direction = int(io_direction)
        self.optional = optional
        self.hide = hide
        self._value = value

    def log_arguments(self, lines: List[str]):
        """Append argument to log when running tool.

        Args:
            lines (List[str]): The logged lines.
        """
        lines.append(f'\'{self.name}\': {self.text_value}')

    @abstractmethod
    def _get_type(self) -> str:
        """Get a string representing the argument type. Should be overridden.

        Returns:
            (str): The argument type.
        """
        return ''

    @abstractmethod
    def _set_value(self, value: object) -> None:
        """Set the argument value.

        Args:
            value (object): The new argument value.
        """
        self._value = value

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {'name': self.name, 'description': self.description, 'value': copy.deepcopy(self.value)}

        if not self.optional:
            # If a value is a "DataFrame", it's assumed to be satisfied if it exists, even if it's empty.
            #
            # If a value is an "integer" (ie "Integer") or "float" (ie "Number"), then this *ought* to be
            # overridden on the specific argument level, but unfortunately, integer and float arguments
            # only check for "max" and "min", which ensure that each value satisfies "min <= value <= max",
            # and some arguments require something like "min < value < max".  (The "proper" way to fix this
            # would be to add "min_eq=True" and "max_eq=True" flags to Integer and Float.)  For now, we
            # just assume that all Integer and Float numbers are automatically set to 0, so "required"
            # information isn't provided.
            #
            # Also, the existence of 'requirement_satisfied' in "interface_info" implies that the argument
            # is required; its absence implies that the argument is optional
            is_required_satisfied = self._is_required_satisfied()
            if is_required_satisfied is not None:
                interface_info['requirement_satisfied'] = is_required_satisfied

        return interface_info

    def _is_required_satisfied(self) -> bool | None:
        """Is the "required" value satisfied?

        Returns:
              True/False if the required value is satisfied. None if not required.
        """
        return bool(self.value)

    @property
    def type(self) -> str:
        """Get the argument type.

        Returns:
            (str): The argument type.
        """
        return self._get_type()

    @property
    def show(self) -> bool:
        """Convenience getter to make more readable (hide/show).

        Returns:
            (bool): Should the argument value be shown (not hidden).
        """
        return not self.hide

    @show.setter
    def show(self, show: bool) -> None:
        """Convenience setter to make more readable (hide/show).

        Args:
            show (bool): Should argument value be shown (not hidden).
        """
        self.hide = not show

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The argument values as a dictionary.
        """
        values = {}
        self._add_key_value('name', self.name, values)
        self._add_key_value('description', self.description, values)
        self._add_key_value('io_direction', int(self.io_direction), values)
        self._add_key_value('optional', self.optional, values)
        self._add_key_value('hide', self.hide, values)
        self._add_key_value('value', self._value, values)
        return values

    def _add_key_value(self, key: str, value: object, values: dict):
        """Add the value to the dictionary if not None.

        Args:
            key (str): Dictionary key to set value to.
            value (obj): Dictionary value.
            values (dict): Dictionary to add the value to.
        """
        if value is not None:
            values[key] = value

    def _get_text_value(self) -> str:
        """Get text value of the argument. May override if str() doesn't give correct value.

        Returns:
            (str): The text value of the argument.
        """
        if self.value is None:
            return ''
        return str(self.value)

    @property
    def value(self) -> object:
        """Get value of the argument.

        Returns:
            (object): The value of the argument.
        """
        return self._value

    @value.setter
    def value(self, value: object):
        """Set value of the argument.

        Args:
            value (obj): The value of the argument.
        """
        self._set_value(value)

    def value_equals(self, value: object) -> bool:
        """Check whether the argument value is equal to the given value.

        Args:
            value: The value to check.

        Returns:
            If the argument value is equal to the given value.
        """
        return self.value == value

    @property
    def text_value(self) -> str:
        """Get text value of the argument.

        Returns:
            (str): The text value of the argument.
        """
        return self._get_text_value()

    def validate(self) -> Optional[str]:
        """Validate the argument.

        Returns:
            (Optional[str]): An error string if invalid or None.
        """
        if not self.optional:
            msg = 'Argument must be specified.'
            if self.value is None:
                return msg
            elif type(self.value) is str and not self.value:
                return msg
        return None

    def equivalent_to(self, argument) -> bool:
        """Check if this argument is equivalent to another for use from history.

        Args:
            argument (Argument): The argument to check against.

        Returns:
            (bool): True if the arguments are equivalent.
        """
        if self.name != argument.name:
            return False
        if self.io_direction != argument.io_direction:
            return False
        if self.optional != argument.optional:
            return False
        if self.type != argument.type:
            return False
        return True

    def adjust_value_from_results(self) -> None:
        """Set the value for running from previous results."""
        if self.io_direction == IoDirection.OUTPUT:
            self.value = None


def contains_invalid_characters(
    argument: Argument,
    allow_forward_slash: bool = False,
    only_basename: bool = False,
    only_check_output: bool = True
) -> str | None:
    """Check if an argument value contains illegal tree item name/filename characters.

    Args:
        argument (Argument): The tool argument
        allow_forward_slash (bool): If True will allways allow '/' character even when not using a file-structured
            data handler. Really just DatasetArgument that allows this.
        only_basename (bool): If True will only validate the portion of the string returned by os.path.basename().
            Really only for RasterArgument.
        only_check_output (bool): If True will only validate if the argument is IoDirection.OUTPUT. We want to
            validate all output coverage, dataset, grid, and raster arguments. Some input string arguments also
            need this validation, e.g. output tree item name prefixes.

    Returns:
        (str): Error message if invalid characters detected or None if valid
    """
    if argument.value is None or (only_check_output and argument.io_direction == IoDirection.INPUT):
        return None  # Don't check if input or no value has been specified yet.

    # Disallow names containing the following characters:
    # < (less than)
    # > (greater than)
    # " (double quote)
    # | (vertical bar or pipe)
    # ? (question mark)
    # * (asterisk)
    # ! (exclamation - valid for filenames but not XMS tree item names)
    # : (colon - sometimes works, but is actually NTFS Alternate Data Streams)
    # \ (backslash)
    # / (forward slash - Sometimes)
    # Also disallow output arguments beginning with a "." - This is legal for XMS tree item names, but the name "."
    # causes problems if it is a dataset because the name is used in the H5 file path. Just use the same restriction
    # raster filenames were enforcing.
    invalid_chars = ['<', '>', '"', '|', '?', '*', '!', ':', '\\']

    # But allow forward slashes with datasets so the user can specify an output tree folder.
    if not allow_forward_slash:
        invalid_chars.append('/')

    # Only check the basename if the value is expected to be an absolute file path (rasters, testing, running
    # outside XMS).
    value = argument.value
    if only_basename or (hasattr(argument, 'data_handler') and argument.data_handler.uses_file_system):
        value = os.path.basename(value)

    if any(invalid_char in value for invalid_char in invalid_chars) or value.startswith('.'):
        return ('Argument must not begin with "." or contain any of the following characters: '
                f'{" ".join(invalid_chars)}')
    return None
