"""DatasetArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import List, Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules

ALLOW_ONLY_TRANSIENT = 'ONLY_TRANSIENT'
ALLOW_ONLY_SCALARS = 'ONLY_SCALARS'
ALLOW_ONLY_VECTORS = 'ONLY_VECTORS'
ALLOW_ONLY_CELL_MAPPED = 'ONLY_CELL_MAPPED'
ALLOW_ONLY_POINT_MAPPED = 'ONLY_POINT_MAPPED'


def dataset_included(dataset: Optional[DatasetReader], filters: Optional[List[str]]) -> bool:
    """Determine if a dataset should be included based on the filters.

    Args:
        dataset (Optional[DatasetReader]): The dataset.
        filters (Optional[List[str]]): The filters.

    Returns:
        (bool): Should the dataset be included?
    """
    included = dataset is not None
    if included and filters is not None:
        if ALLOW_ONLY_TRANSIENT in filters and dataset.num_times < 2:
            included = False
        elif ALLOW_ONLY_SCALARS in filters and dataset.num_components != 1:
            included = False
        elif ALLOW_ONLY_VECTORS in filters and dataset.num_components < 2:
            included = False
        elif ALLOW_ONLY_CELL_MAPPED in filters and dataset.location != 'cells':
            if isinstance(dataset.location, str) and dataset.location != 'cells':
                included = False
            elif isinstance(dataset.location, tuple) and dataset.location[0] != 'cells':
                included = False
        elif ALLOW_ONLY_POINT_MAPPED in filters and dataset.location != 'points':
            if isinstance(dataset.location, str) and dataset.location != 'points':
                included = False
            elif isinstance(dataset.location, tuple) and dataset.location[0] != 'points':
                included = False
    return included
