"""OutputPointsComponentDisplay class."""
# 1. Standard python modules
import os
import shutil
import uuid

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.data import output_points_data as opd
from xms.tuflowfv.gui.output_points_dialog import OutputPointsDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


OUTPUT_POINT_JSON = 'output_pt_display.json'
OUTPUT_INITIAL_ATT_ID_FILE = 'initial_output_pt.attids'
OUTPUT_INITIAL_COMP_ID_FILE = 'initial_output_pt.compids'


class OutputPointsComponentDisplay:
    """Helper class for Output Points component display options."""

    def __init__(self, output_comp, query=None):
        """Create a helper class.

        Args:
            output_comp (OutputPointsComponent): The component this helper should help
            query (Query): The XMS inter-process communicator
        """
        self.comp = output_comp
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''
        self._query = query

    def _unpack_xms_data(self, param_map):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (dict): The ActionRequest parameter map

        Returns:
            (int): component id of atts to display
        """
        # Get the XMS attribute ids of the selected features (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected features (if any).
        comp_id = UNINITIALIZED_COMP_ID
        id_files = param_map.get('id_files', [])
        if id_files and id_files[0]:
            files_dict = {'POINT': (id_files[0], id_files[1])}
            self.comp.load_coverage_component_id_map(files_dict)
            comp_id = self._check_selected_types()
        # Clean up temp files dumped by SMS.
        shutil.rmtree(os.path.join(os.path.dirname(self.comp.main_file), 'temp'), ignore_errors=True)
        return comp_id

    def _check_selected_types(self):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Returns:
            (int): component id of attributes to display
        """
        num_features = len(self.selected_att_ids)
        if num_features == 1:  # 1 feature selected, use those atts
            comp_id = self.comp.get_comp_id(TargetType.point, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one feature selected, check types
            try:  # Get the component ids of all the selected arcs
                self.selected_comp_ids = list(self.comp.comp_to_xms[self.comp.cov_uuid][TargetType.point].keys())
            except KeyError:
                return UNINITIALIZED_COMP_ID  # No component ids assigned for any of the selected arcs
            self.dlg_message = 'Multiple points selected. Changes will apply to all selected points.'
            # If there are multiple entities selected with the same component id, display those attributes. Otherwise
            # display an empty, default dialog.
            return UNINITIALIZED_COMP_ID if len(self.selected_comp_ids) != 1 else self.selected_comp_ids[0]

    def _update_component_ids_from_files(self):
        """Read att and comp ids from files and update in XMS.

        Called in get_initial_display_options() after a model native import.
        """
        comp_folder = os.path.dirname(self.comp.main_file)
        initial_att_file = os.path.join(comp_folder, OUTPUT_INITIAL_ATT_ID_FILE)
        initial_comp_file = os.path.join(comp_folder, OUTPUT_INITIAL_COMP_ID_FILE)
        att_ids = read_display_option_ids(initial_att_file)
        comp_ids = read_display_option_ids(initial_comp_file)
        io_util.removefile(initial_att_file)
        io_util.removefile(initial_comp_file)
        for att_id, comp_id in zip(att_ids, comp_ids):
            self.comp.update_component_id(TargetType.point, att_id, comp_id)

    def update_display_options_file(self, new_main_file, new_path):
        """Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file (str): Path to the new component's main file
            new_path (str): The new component's directory.
        """
        new_data = opd.OutputPointsData(new_main_file)
        # If duplicating, clear the coverage UUID. Will query XMS for the new coverage's UUID on the create event.
        new_data.info.attrs['cov_uuid'] = ''
        # Set component UUID in point display options file.
        new_comp_uuid = os.path.basename(new_path)
        basename = os.path.basename(self.comp.disp_opts_files[0])
        fname = os.path.join(new_path, basename)
        json_dict = read_display_options_from_json(fname)
        json_dict['uuid'] = str(uuid.uuid4())  # Generate a new point display list UUID.
        json_dict['comp_uuid'] = new_comp_uuid
        categories = CategoryDisplayOptionList()
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
        new_data.info.attrs['display_uuid'] = json_dict['uuid']  # Store point display list UUID in component data.
        new_data.commit()

    def refresh_component_ids(self, new_comp_id):
        """Load all the currently used component ids and clean out the unused ones.

        Called on OK of the assign BC feature dialogs. Keeps the files fresh and the data has been small enough so far.

        Args:
            new_comp_id (int): Component id of the feature that was just assigned
        """
        # Load all currently used component ids
        self._query.load_component_ids(self.comp, points=True)
        # Drop unused component ids from the xarray datasets
        point_comp_ids = list(self.comp.comp_to_xms.get(self.comp.cov_uuid, {}).get(TargetType.point, {}).keys())
        if new_comp_id > UNINITIALIZED_COMP_ID:
            point_comp_ids.append(new_comp_id)
        mask = self.comp.data.points.comp_id.isin(point_comp_ids)
        self.comp.data.points = self.comp.data.points.where(mask, drop=True)
        # Rewrite the display option id files now to clear out non-existent component ids.
        self.update_id_files()
        self.comp.data.vacuum()
        # Send back updated display lists to XMS after ActionRequest
        self.comp.display_option_list = [
            XmsDisplayMessage(file=self.comp.disp_opts_files[0], edit_uuid=self.comp.cov_uuid),
        ]

    def update_id_files(self):
        """Writes the display id files."""
        comp_path = os.path.dirname(self.comp.main_file)
        id_file = os.path.join(comp_path, 'output_pts.display_ids')
        if len(self.comp.data.points.comp_id) > 0:
            write_display_option_ids(id_file, self.comp.data.points.comp_id.data.astype(int))
        else:  # If we don't have any of this type, just delete the id file.
            io_util.removefile(id_file)

    def assign_attributes(self, params, parent):
        """Display the Assign Output Points Attributes and persist data if accepted.

        Args:
            params (dict): The ActionRequest parameter map
            parent (QWidget): The parent window

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        old_comp_id = self._unpack_xms_data(params[0])
        if not self.selected_att_ids:
            return [('INFO', 'No points selected. Select one or more points to assign output attributes.')], []

        increment = True
        comp_id = old_comp_id
        if comp_id == UNINITIALIZED_COMP_ID:  # Create some default atts if this is the initial assignment.
            comp_id = self.comp.data.add_output_points()
            increment = False
        point_data = self.comp.data.points.where(self.comp.data.points.comp_id == comp_id, drop=True)
        if increment:
            # Increment the component id anytime dialog triggered (won't commit unless user accepts). We don't want to
            # just update the existing attributes because they may have been assigned with a multi-select.
            comp_id = self.comp.data.add_output_points(dset=point_data)

        dlg = OutputPointsDialog(parent=parent, point_data=point_data,
                                 export_format=self.comp.data.info.attrs['export_format'])
        if dlg.exec():
            # Save the coverage level attribute
            self.comp.data.info.attrs['export_format'] = dlg.export_format
            # Update the attribute datasets
            self.comp.data.update_point(comp_id=comp_id, new_atts=dlg.point_data)
            # Associate all selected points with the new component id.
            for feature_id in self.selected_att_ids:
                self.comp.update_component_id(TargetType.point, feature_id, comp_id)
            self.refresh_component_ids(comp_id)
        return [], []

    def display_options(self, parent, cov_uuid):
        """Shows the display options dialog for a display list.

        Args:
            parent (QWidget): The parent window
            cov_uuid (str): UUID of the coverage geometry associated with the display list, if any.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Get the point display options.
        point_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[0])
        point_categories.from_dict(json_dict)
        categories_list = [point_categories]

        dlg = CategoryDisplayOptionsDialog(categories_list, parent, package_name='xmstuflowfv')
        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for idx, category_list in enumerate(category_lists):
                write_display_options_to_json(self.comp.disp_opts_files[idx], category_list)
                self.comp.display_option_list.append(
                    XmsDisplayMessage(
                        file=self.comp.disp_opts_files[idx], edit_uuid=cov_uuid, draw_type=DrawType.draw_at_ids
                    )
                )
        return [], []

    def ensure_display_option_files_exist(self):
        """Copies default BC arc and point display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists. Should only be called by the unmapped BC coverage on
        creation.
        """
        if not os.path.exists(self.comp.disp_opts_files[0]):  # Only points in this coverage
            # Read the default arc display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                        'default_data', OUTPUT_POINT_JSON)
            json_dict = read_display_options_from_json(default_file)
            categories.from_dict(json_dict)
            categories.comp_uuid = self.comp.uuid
            write_display_options_to_json(self.comp.disp_opts_files[0], categories)
            # Save our display list UUID to the main file
            self.comp.data.info.attrs['display_uuid'] = categories.uuid
            self.comp.data.commit()

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (Optional[dict]): Generic map of parameters, unused

        Returns:
            Empty message and ActionRequest lists
        """
        # Query XMS for parent coverage's UUID if we don't know it yet.
        if not self.comp.cov_uuid:
            self.comp.cov_uuid = query.parent_item_uuid()
            self.comp.data.info.attrs['cov_uuid'] = self.comp.cov_uuid
            self.comp.data.commit()
        if not self.comp.cov_uuid:
            return [('ERROR', 'Could not get the TUFLOWFV Output Points coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.comp.main_file), OUTPUT_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            self._update_component_ids_from_files()
        # Send the display options json files to XMS.
        self.comp.display_option_list = [
            XmsDisplayMessage(file=self.comp.disp_opts_files[0], edit_uuid=self.comp.cov_uuid),  # points
        ]
