"""Worker thread for the solution importer."""
# 1. Standard python modules
import logging

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.tuflowfv.dmi.xms_data import XmsData
from xms.tuflowfv.file_io.solution_reader import TuflowfvSolutionReader


class SolutionLoadWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing TUFLOWFV solutions."""
    processing_finished = Signal()

    def __init__(self, parent, fvc_filename, xms_data):
        """Construct the worker.

        Args:
            parent (QWidget): The Qt parent
            fvc_filename (str): Path to the control file
            xms_data (XmsData): Simulation data retrieved from SMS
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.tuflowfv')
        self._fvc_filename = fvc_filename
        self._xms_data = xms_data
        self.datasets = []

    def _do_work(self):
        """Thread runner that exports the fort.14."""
        # Look for a mapped BC coverage first.
        self._logger.info('Loading TUFLOWFV simulation run solution files...')
        try:
            reader = TuflowfvSolutionReader(sim_data=self._xms_data.sim_data, fvc_filename=self._fvc_filename,
                                            xms_data=self._xms_data)
            self.datasets = reader.read()
        except Exception as e:
            self._logger.error(f'Error(s) encountered while loading TUFLOWFV solution files {str(e)}.')

    def add_datasets(self, query):
        """Send imported data to XMS.

        Args:
            query (Query): XMS interprocess communication object
        """
        for dataset in self.datasets:
            query.add_dataset(dataset)


def import_solution_with_feedback(fvc_filename, query, parent):
    """Entry point to read a TUFLOWFV solution.

    Args:
        fvc_filename (str): Path to the control file
        query (Query): XMS interprocess communication object
        parent (QWidget): The Qt parent
    """
    # The Context for this ActionRequest is at some temporary model run component, we want the parent simulation.
    xms_data = XmsData(query, at_sim=False)
    worker = SolutionLoadWorkerThread(parent, fvc_filename, xms_data)
    display_text = {
        'title': 'Loading TUFLOWFV Solution Files',
        'working_prompt': 'Reading TUFLOWFV solution files. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
        'success_prompt': 'Successfully read TUFLOWFV solution',
        'note': '',
        'auto_load': 'Auto load solution datasets into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():  # Send even if errors because we may have a partial solution
        worker.add_datasets(query)
