"""Class to write TUFLOWFV Holland Wind boundaries."""
# 1. Standard python modules
import datetime
import logging

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation, julian_to_datetime

# 4. Local modules


class HollandWindWriter:
    """Class to write a TUFLOWFV Holland Wind boundaries."""
    METERS_PER_NAUTICAL_MILE = 1852

    def __init__(self, reftime, uses_isodate):
        """Constructor.

        Args:
            reftime (datetime.datetime): The simulation reference date
            uses_isodate (bool): True if the simulation uses isodate, False if uses hours time format
        """
        self._reftime = reftime
        self._uses_isodate = uses_isodate
        self._logger = logging.getLogger('xms.tuflowfv')
        self._xcoords = None
        self._ycoords = None

    def _get_time_from_julian(self, julian):
        """Convert a nodal time stored as a Julian date to either a Python datetime or float hour offset.

        Args:
            julian (float): The nodal time as a Julian date

        Returns:
            Union[datetime.datetime, float]: The nodal time as a float offset if simulation time format is hours or a
                Python datetime if ISODATE
        """
        dt = julian_to_datetime(julian)
        if self._uses_isodate:  # If using ISODATE time format, return the Python datetime and allow pandas to format
            return dt
        # If using hours time format, get a timedelta from the reftime to this nodal time and convert to total hours.
        delta = dt - self._reftime
        return delta / datetime.timedelta(hours=1)

    def _build_csv_dataframe(self, time, p0, pa, rmax, holland, rhoa, km, thetamax, deltafm, wbgx, wbgy):
        """Combine columns stored in geometry and attribute dumps and default constant columns.

        Args:
            time (np.array): The nodal times array (hour floats or datetime objects)
            p0 (np.array): The central pressure array
            pa (np.array): The ambient pressure array
            rmax (np.array): The maximum wind radius array
            holland (np.array): The Holland peakedness array
            rhoa (np.array): The air density array
            km (np.array): The KM array, whatever that is (mean near surface wind speed / gradient level wind speed)
            thetamax (np.array): The maximum wind direction array
            deltafm (np.array): The asymmetry factor array
            wbgx (np.array): The wind background x component array
            wbgy (np.array): The wind background y component array

        Returns:
            pd.DataFrame: DataFrame of the extracted nodal values
        """
        df_data = {
            'Time': time,
            'X': self._xcoords,
            'Y': self._ycoords,
            'P0': p0,
            'PA': pa,
            'RMAX': rmax,
            'B': holland,
            'RHOA': rhoa,  # Air density
            'KM': km,  # Mean near surface wind speed / gradient level wind speed
            'THETMAX': thetamax,
            'DELTAFM': deltafm,  # Asymmetry factor
            'Latitude': self._ycoords,  # Not supporting Cartesian space with wind boundaries for now
            'WBGX': wbgx,  # Background wind X component
            'WBGY': wbgy,  # Background wind Y component
        }
        return pd.DataFrame(df_data)

    def _extract_coordinates(self, wind_dump):
        """Extract the wind node coordinates from the coverage dump geometry.

        Args:
            wind_dump (WindCoverage): Attribute/geometry dump of the coverage to write

        Returns:
            tuple(np.array, np.array): The x and y coordinates of the wind nodes
        """
        cov_pts = wind_dump.m_cov.get_points(FilterLocation.PT_LOC_ALL)
        self._xcoords = np.array([pt.x for pt in cov_pts])
        self._ycoords = np.array([pt.y for pt in cov_pts])

    def _extract_nodal_values(self, wind_dump):
        """Extract the values stored on each wind node.

        Args:
            wind_dump (WindCoverage): Attribute/geometry dump of the coverage to write

        Returns:
            pd.DataFrame: DataFrame of the extracted nodal values
        """
        time = []
        p0 = []
        pa = []
        rmax = []
        holland = []
        rhoa = []
        km = []
        thetamax = []
        deltafm = []
        wbgx = []
        wbgy = []
        for node in wind_dump.m_nodeWind:
            time.append(self._get_time_from_julian(node.m_date))
            p0.append(node.m_iMinSeaLevelPressure)
            pa.append(node.m_iPressure)
            rmax.append(int(node.m_iMaxWindRadius * self.METERS_PER_NAUTICAL_MILE))
            holland.append(node.m_hollandB)
            rhoa.append(node.rhoa)
            km.append(node.km)
            thetamax.append(node.m_iStormDirection)
            deltafm.append(node.deltafm)
            wbgx.append(node.wbgx)
            wbgy.append(node.wbgy)
        time = np.array(time)  # Allow numpy/pandas determine dtype - either float or datetime
        p0 = np.array(p0, dtype=np.int32)  # We think the rest of these need to be integers
        pa = np.array(pa, dtype=np.int32)
        rmax = np.array(rmax, dtype=np.int32)
        holland = np.array(holland, dtype=np.float32)
        rhoa = np.array(rhoa, dtype=np.float32)
        km = np.array(km, dtype=np.float32)
        thetamax = np.array(thetamax, dtype=np.int32)
        deltafm = np.array(deltafm, dtype=np.int32)
        wbgx = np.array(wbgx, dtype=np.float32)
        wbgy = np.array(wbgy, dtype=np.float32)
        return self._build_csv_dataframe(time, p0, pa, rmax, holland, rhoa, km, thetamax, deltafm, wbgx, wbgy)

    def write(self, filename, wind_dump):
        """Write a Holland wind boundary coverage to a CSV file.

        Args:
            filename (str): The path to the CSV file to write
            wind_dump (WindCoverage): Attribute/geometry dump of the coverage to write
        """
        try:
            self._logger.info('Exporting Holland wind boundary from CSV file...')
            self._extract_coordinates(wind_dump)
            df = self._extract_nodal_values(wind_dump)
            df.sort_values(by=['Time'], inplace=True)
            df.to_csv(filename, index=False, date_format='%d/%m/%Y %H:%M:%S' if self._uses_isodate else None)
        except Exception as e:
            self._logger.error(f'Error exporting Holland wind boundary to CSV file: {str(e)}')
