"""Class to read a TUFLOWFV shapefile and convert it to a data_objects coverage."""
# 1. Standard python modules
import logging
import os
import uuid

# 2. Third party modules
import shapefile
from shapely.geometry import shape

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon

# 4. Local modules
from xms.tuflowfv.file_io.output_component_builder import OutputComponentBuilder


class ShapefileConverter:
    """Class to read a TUFLOWFV shapefile and convert it to a data_objects coverage."""

    def __init__(self, filename, manager=None):
        """Constructor.

        Args:
            filename (str): Path to the polygon shapefile
            manager (StructureArcManager): Handles determining if an arc is a structure
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._filename = filename
        self._next_point_id = 1
        self._next_arc_id = 1
        self._next_poly_id = 1
        self._pt_hash = {}  # {location_hash: Point}
        self._arc_hash = {}  # {arc_hash: Arc}
        self._manager = manager
        # This is the mapping of XMS map module feature id to the attribute (BC or material) id in the file.
        self.feature_id_to_att_id = {}  # {feature_id: bc_id/mat_id}
        self.feature_id_to_feature_name = {}  # {feature_id: feature name label}

    def _find_field_name(self, shp_f):
        """Find the first field name in a shapefile's records.

        TUFLOWFV currently just assumes the required field is the first one as there is currently only one field
        per shapefile read. In the future this might change.

        Args:
            shp_f (shapefile.Reader): The feature object as read from the shapefile by pyshp

        Returns:
            str: The ID field name,
        """
        try:
            # First field is always the hidden 'DeletionFlag', skip it. First item in field list is the field name.
            return shp_f.fields[1][0]
        except Exception:
            raise ValueError(f'Could not find field name in shapefile: {self._filename}')

    def _find_output_field_names(self, shp_f):
        """Find the field name's in an output point shapefile's records.

        We are assuming the field names are in the following order: 'Type' (ignored), 'Label', 'Comment, 'Vert_min',
        'Vert_max'

        Args:
            shp_f (shapefile.Reader): The feature object as read from the shapefile by pyshp

        Returns:
            tuple(str): The 'Label' field name, the 'Comment' field name, the 'Vert_min' field name, and the 'Vert_max'
                field name
        """
        try:
            # First field is always the hidden 'DeletionFlag', skip it.
            return shp_f.fields[2][0], shp_f.fields[3][0], shp_f.fields[4][0], shp_f.fields[5][0]
        except Exception:
            raise ValueError(f'Could not find field name in shapefile: {self._filename}')

    def _create_do_coverage(self):
        """Get an empty data_objects Coverage.

        Returns:
            data_objects.parameters.Coverage: See description
        """
        # Using the file's basename as the coverage name.
        cov_name = f'{os.path.splitext(os.path.basename(self._filename))[0]}'
        cov = Coverage(name=cov_name, uuid=str(uuid.uuid4()))
        cov.complete()
        return cov

    def _get_hashed_point(self, x, y):
        """Get an existing data_objects point using a location hash (creates the point if it doesn't exist).

        Args:
            x (float): The x-coordinate of the point

        Returns:
            data_objects.parameters.Point: The existing or newly created point associated with the passed in location
        """
        point_hash = hash((x, y))
        do_point = self._pt_hash.get(point_hash)

        if not do_point:
            # When dealing with structures, we get the next id from the manager. This allows us to avoid
            # resetting between files.
            next_point_id = self._next_point_id
            if self._manager:
                next_point_id = self._manager.next_point_id
                self._manager.next_point_id += 1
            else:
                self._next_point_id += 1
            do_point = Point(x=x, y=y, feature_id=next_point_id)
            self._pt_hash[point_hash] = do_point
        return do_point

    def _get_hashed_arc(self, coords):
        """Create a data_objects arc from a sequence of point coordinates, or retrieve a previously created one.

        Args:
            coords (Sequence): The x,y coords of the arc points

        Returns:
            data_objects.parameters.Arc: The existing or newly created arc associated with the passed in locations
        """
        points = [self._get_hashed_point(x, y) for x, y in coords]
        # Check if we have already created this arc
        point_ids = [point.id for point in points]
        point_ids.sort()
        arc_hash = hash(tuple(point_ids))  # Create a hash using all the sorted point ids of the arc (ignore direction)
        do_arc = self._arc_hash.get(arc_hash)
        if not do_arc:
            vertices = points[1:-1] if len(points) > 2 else None
            do_arc = Arc(start_node=points[0], end_node=points[-1], vertices=vertices, feature_id=self._next_arc_id)
            self._next_arc_id += 1
            self._arc_hash[arc_hash] = do_arc
        return do_arc

    def convert_polygons(self, create_coverage=True):
        """Convert a polygon shapefile to a data_object Coverage.

        Args:
            create_coverage: If True will return a data_objects coverage. If False returns a list of the polygons.

        Returns:
             Union[Coverage, list[Polygon]: The data_objects Coverage geometry or polygons
        """
        try:
            do_polygons = []
            with shapefile.Reader(self._filename) as f:
                field_name = self._find_field_name(f)
                for polygon in f:
                    geometry = shape(polygon.shape)
                    if geometry.geom_type not in ['Polygon', 'MultiPolygon']:
                        continue
                    exterior = self._get_hashed_arc(geometry.exterior.coords)
                    holes = []
                    for interior in geometry.interiors:
                        holes.append([self._get_hashed_arc(interior.coords)])
                    do_polygon = Polygon(feature_id=self._next_poly_id)
                    identifier = polygon.record[field_name]
                    try:  # Material GIS shapefile
                        self.feature_id_to_att_id[self._next_poly_id] = int(identifier)
                    except ValueError:  # New style string identifiers for BC polygons
                        self.feature_id_to_att_id[do_polygon.id] = do_polygon.id
                        self.feature_id_to_feature_name[self._next_poly_id] = identifier
                    self._next_poly_id += 1
                    do_polygon.set_arcs([exterior])
                    if holes:
                        do_polygon.set_interior_arcs(holes)
                    do_polygons.append(do_polygon)
            if not do_polygons:
                return None
            if create_coverage:
                do_cov = self._create_do_coverage()
                do_cov.polygons = do_polygons
                return do_cov
            else:
                return do_polygons
        except Exception as e:
            self._logger.error(f"Error reading shapefile: {str(e)}")
        return None

    def convert_lines(self, create_coverage=True):
        """Convert a line shapefile to a data_object Coverage.

        Args:
            create_coverage (bool): If True will return a data_objects coverage. If False returns a list of the
                polygons.

        Returns:
             Union[Coverage, list[Polygon]: The data_objects Coverage geometry or polygons
        """
        try:
            do_arcs = []
            with shapefile.Reader(self._filename) as f:
                field_name = self._find_field_name(f)
                for line in f:
                    geometry = shape(line.shape)
                    if geometry.geom_type not in ['LineString', 'LinearRing']:
                        self._logger.error(f'Unsupported feature found in {self._filename}')
                        continue
                    do_arc = self._get_hashed_arc(geometry.coords)
                    identifier = line.record[field_name]
                    if self._manager is not None and self._manager.is_arc_structure(identifier):
                        self._manager.add_arc_structure_location(identifier, do_arc)
                        continue  # This arc is already being used as a structure

                    # ID fields can be integers or strings, try both.
                    try:  # BC GIS shapefile
                        self.feature_id_to_att_id[do_arc.id] = int(identifier)
                    except ValueError:  # New style string identifiers for BC polygons
                        self.feature_id_to_att_id[do_arc.id] = identifier
                        self.feature_id_to_feature_name[do_arc.id] = identifier
                    do_arcs.append(do_arc)
            if not do_arcs:
                return None
            if create_coverage:
                do_cov = self._create_do_coverage()
                do_cov.arcs = do_arcs
                return do_cov
            else:
                return do_arcs
        except Exception as e:
            self._logger.error(f"Error reading shapefile: {str(e)}")
        return None

    def convert_points(self):
        """Convert a point shapefile to a data_object Coverage.

        Returns:
             Coverage: The data_objects Coverage geometry
        """
        do_points = []
        try:
            with shapefile.Reader(self._filename) as f:
                field_name = self._find_field_name(f)
                for point in f:
                    geometry = shape(point.shape)
                    if geometry.geom_type not in ['Point', 'MultiPoint']:
                        continue
                    # For some reason, *geometry.coords wouldn't work
                    do_point = self._get_hashed_point(geometry.coords[0][0], geometry.coords[0][1])
                    identifier = point.record[field_name]
                    try:  # Old style int ids
                        self.feature_id_to_att_id[do_point.id] = int(identifier)
                    except ValueError:  # New style string identifiers for points
                        self.feature_id_to_att_id[do_point.id] = do_point.id
                        self.feature_id_to_feature_name[do_point.id] = identifier
                    do_points.append(do_point)
        except Exception as e:
            self._logger.error(f"Error reading shapefile: {str(e)}")
        if not do_points:
            return None
        do_cov = self._create_do_coverage()
        do_cov.set_points(do_points)
        return do_cov

    def convert_output_points(self):
        """Convert an output point shapefile to a data_object Coverage.

        Returns:
             Coverage: The data_objects Coverage geometry
        """
        try:
            do_points = []
            with shapefile.Reader(self._filename) as f:
                label_field, comment_field, vert_min_field, vert_max_field = self._find_output_field_names(f)
                data = {}
                for line in f:
                    geometry = shape(line.shape)
                    if geometry.geom_type not in ['Point', 'MultiPoint']:
                        continue
                    # For some reason, *geometry.coords wouldn't work
                    do_point = self._get_hashed_point(geometry.coords[0][0], geometry.coords[0][1])
                    self.feature_id_to_att_id[do_point.id] = do_point.id
                    self.feature_id_to_feature_name[self._next_poly_id] = line.record[label_field]
                    data[do_point.id] = {
                        'label': line.record[label_field] if line.record[label_field] is not None else str(do_point.id),
                        'comment': line.record[comment_field] if line.record[comment_field] is not None else '',
                        'vert_min': line.record[vert_min_field] if line.record[vert_min_field] is not None else 0.0,
                        'vert_max': line.record[vert_max_field] if line.record[vert_max_field] is not None else 0.0,
                    }
                    do_points.append(do_point)
            if not do_points:
                return None, None
            do_cov = self._create_do_coverage()
            do_cov.set_points(do_points)
            # Build the component data
            builder = OutputComponentBuilder(cov_uuid=do_cov.uuid, from_csv=False, atts=data,
                                             point_id_to_feature_id=self.feature_id_to_att_id,
                                             point_id_to_feature_name=self.feature_id_to_feature_name)
            do_comp = builder.build_output_component()
            return do_cov, do_comp
        except Exception as e:
            self._logger.error(f'Error reading shapefile: {str(e)}')
        return None, None
