"""TUFLOWFV hydraulic structures' writer."""
import logging
import os.path
import shutil
from pathlib import Path

# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.tuflowfv.data.structure_data import get_default_structure_dataset
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.file_io.culvert_csv_io import write_culvert_csv

NUM_LINK_FEATURES = 2


class StructureWriter:
    """Writes hydraulic structures to the control file."""
    def __init__(self, pycomp, cov, fvc_file, ss):
        """Constructor.

        Args:
            pycomp (StructureComponent): The simulation's structure component
            cov (Coverage): The data objects coverage
            fvc_file (str): The path to the control file
            ss (StringIO): Stream to write structures to
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._pycomp = pycomp
        self._cov = cov
        self._data = pycomp.data
        self._ss = ss
        self._poly_arcs = set()
        self._processed_polys = set()
        self._processed_arcs = set()
        self._fvc_file = Path(fvc_file)
        self._csv_dir = Path(fvc_file).parent.parent / "model" / "csv"

    def _get_name(self, feature_atts):
        """Check for a name in the attributes and log a warning if it is missing.

        Args:
            feature_atts (xr.Dataset): The structure's attributes
        """
        name = str(feature_atts.name.item())
        if not name:
            self._logger.warning(f'No name found for structure feature {feature_atts.comp_id.item()}. Using ID as name.')
            name = str(feature_atts.comp_id.item())
        return name

    def _write_structure_header(self, comp_id, feature_atts, target_type):
        """Write header line for a single structure.

        Args:
            comp_id (int): index of the structure
            feature_atts (xr.Dataset): The structure's attributes
            target_type (TargetType): The feature type of the structure

        Returns:
            bool: False on failure
        """
        struct_type = feature_atts.struct_type.item()
        header = f'Structure == {struct_type.title()}, '
        if struct_type.startswith('linked'):  # Structure set, need to write both names
            partner_id, swap_comps = self._data.find_partner(comp_id=comp_id)
            dset = self._data.arcs if target_type == TargetType.arc else self._data.arcs
            partner_atts = dset.where(dset.comp_id == partner_id, drop=True)

            if partner_atts.sizes['comp_id'] != 1:  # Couldn't find the partner's attributes
                feature_type = 'arc' if feature_atts == TargetType.arc else 'polygon'
                att_ids = self._pycomp.get_xms_ids(comp_id=partner_id, entity_type=feature_type)
                # Dumb function returns -1 instead of an empty list on failure.
                att_id = att_ids[0] if att_ids is list else att_ids
                self._logger.warning(f'Invalid link found for {feature_type} feature {att_id}.')
                return False

            name1 = self._get_name(feature_atts)
            name2 = self._get_name(partner_atts)

            if feature_atts.upstream[0].item() == 1:  # This arc is the upstream connection
                name = f'{name1}, {name2}'
            else:  # The other arc is the upstream connection
                name = f'{name2}, {name1}'
        else:
            name = feature_atts.name.item()
        self._ss.write(f'{header}{name}\n')
        return True

    def _write_polygons(self):
        """Write the structure polygon structures to the file."""
        for poly in self._cov.polygons:
            # Find the comp_id of this polygon feature.
            comp_id = self._pycomp.get_comp_id(TargetType.polygon, poly.id)
            if comp_id == UNINITIALIZED_COMP_ID:
                self._logger.warning(f'No structure attributes found for feature polygon {poly.id}.')
                continue
            elif comp_id in self._processed_polys:
                continue  # Already wrote this structure via the other feature in the set. Avoid duplicates.
            # Keep track of the comp ids of the structures we have written in case they are part of a set.
            self._processed_polys.add(comp_id)

            # Get the attributes for this feature.
            poly_atts = self._data.polygons.where(self._data.polygons == comp_id, drop=True)
            if poly_atts.sizes['comp_id'] != 1:
                self._logger.warning(f'Invalid structure found for polygons feature {poly.id}.')
                continue

            # Write the structure attributes to the control file.
            if not self._write_structure_header(comp_id=comp_id, feature_atts=poly_atts,
                                                target_type=TargetType.polygon):
                continue  # Problem finding partner in a set. Warning already logged.
            self._write_structure_atts(poly_atts)

            # Keep track of the arcs used to build the polygon, so we can filter it out of arc structures later.
            poly_arcs = poly.arcs
            for poly_arc in poly_arcs:
                self._poly_arcs.add(poly_arc.id)

            self._ss.write('End Structure\n\n')

    def _write_arcs(self):
        """Write the structure arc structures to the file."""
        for arc in self._cov.arcs:
            if arc.id in self._poly_arcs:
                continue  # Arc is used to build a polygon
            comp_id = self._pycomp.get_comp_id(TargetType.arc, arc.id)
            if comp_id != UNINITIALIZED_COMP_ID:
                arc_atts = self._data.arcs.where(self._data.arcs.comp_id == comp_id, drop=True)
            else:
                self._logger.info(f'Using default values for arc {arc.id}.')
                name = f'arc_{arc.id}'
                arc_atts = get_default_structure_dataset(comp_id=arc.id, fill=True)
                arc_atts['name'][0] = name


            if arc_atts.name[0].item() in self._processed_arcs:
                continue  # Already wrote this structure via the other feature in the set. Avoid duplicates.
            if not self._write_structure_header(comp_id, arc_atts, TargetType.arc):
                continue  # Problem finding partner in a set. Warning already logged.
            self._write_structure_atts(arc_atts)
            # Mark both arcs as processed so that we don't write them again.
            self._processed_arcs.add(arc_atts.name[0].item())
            if arc_atts.connection:
                self._processed_arcs.add(arc_atts.connection[0].item())
            else:  # Double-check our sets list to make sure we add both members.
                feature_set = self._data.sets.where(self._data.sets.comp1_id == comp_id, drop=True)
                if feature_set.sizes['set_id'] == 1:
                    partner_atts = self._data.arcs.where(self._data.arcs.comp_id == feature_set.comp2_id.item(), drop=True)
                    self._processed_arcs.add(partner_atts.name[0].item())

            self._ss.write('End Structure\n\n')

    def _write_structure_atts(self, feature_atts):
        """Write the structure attributes to the control file.

        Args:
            feature_atts (xr.Dataset): The structure's attributes
        """
        # When calculating the relative path to the control file, we need to make sure to not include the file name
        # since os.path.relpath will treat the file as a directory, and we will end up with an extra set of "../"
        fvc_dir = self._fvc_file.parent

        # Write Weir Structure
        flux_function = feature_atts.flux_function[0].item().lower()
        if flux_function == 'weir':
            if feature_atts.elevation_is_dz[0].item() != 0:
                self._ss.write('  Flux Function == Weir_dz \n')
            else:
                self._ss.write('  Flux Function == Weir  \n')

            # Write Weir Properties
            # Properties == z, C, Ex, a, b, Csf_min
            properties = f'  Properties == {feature_atts.weir_z[0].item()}, {feature_atts.weir_cw[0].item()}'
            if feature_atts.define_weir[0].item() != 0:
                properties += (f', {feature_atts.weir_ex[0].item()}, {feature_atts.weir_a[0].item()}, '
                               f'{feature_atts.weir_b[0].item()}, {feature_atts.weir_csfm[0].item()}')
            self._ss.write(properties + '\n')

        # Write Culvert Structure
        elif flux_function == 'culvert':
            self._ss.write('  Flux Function == Culvert \n')
            # Culvert files are written relative to the control file.
            relative_file = str(os.path.relpath(self._csv_dir.resolve() / "culverts.csv", fvc_dir))
            self._ss.write(f'  Culvert File == {relative_file}, {feature_atts.comp_id.item()}\n')

        # Write Bridge Structure
        elif flux_function == 'nlswe':
            self._ss.write('  Flux Function == nlswe \n')
            self._ss.write(f'  Energy Loss Function == {feature_atts.energy_loss_function[0].item()} \n')
            if feature_atts.energy_loss_function[0].item().lower() == 'coefficient':
                self._ss.write(f'  Form Loss Coefficient == {feature_atts.form_loss_coefficient[0].item()} \n')
            else:
                self._ss.write(f'  Energy Loss File == {feature_atts.energy_loss_file[0].item()} \n')
            if feature_atts.blockage_file[0].item():
                copied_blockage_file = self._csv_dir / f"blockage-{feature_atts.comp_id.item()}.csv"
                shutil.copy(feature_atts.blockage_file[0].item(), copied_blockage_file)
                relative_file = str(os.path.relpath(copied_blockage_file, fvc_dir))
                self._ss.write(f'  Blockage File == {relative_file} \n')
            if feature_atts.width_file[0].item():
                copied_width_file = self._csv_dir / f"width-{feature_atts.comp_id.item()}.csv"
                shutil.copy(feature_atts.width_file[0].item(), copied_width_file)
                relative_file = str(os.path.relpath(copied_width_file, fvc_dir))
                self._ss.write(f'  Width File == {relative_file} \n')

    def write(self):
        """Write the hydraulic structures to the control file."""
        self._write_polygons()
        self._write_arcs()

        # Write global culvert CSV
        write_culvert_csv(str(self._csv_dir / "culverts.csv"), self._data)
