"""Qt delegate that displays a button for bringing up the Assign BC dialog."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate
import xarray as xr

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui.add_gridded_bc_dialog import AddGriddedBcDialog


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def run_add_gridded_bcs_dialog(sim_data, grid_id, parent, time_formats):
    """Run the BC grid definition dialog for a global BC.

    Args:
        sim_data (SimData): The simulation data
        grid_id (int): Grid id of the row to to add/edit gridded BCs
        parent (QWidget): The Qt parent dialog
        time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
            specifiers for (strftime, qt)
    """
    # Filter out BCs from other grids
    gridded_bcs = sim_data.gridded_bcs.bcs.where(sim_data.gridded_bcs.bcs.grid_id == grid_id, drop=True)
    orig_bcs = sim_data.gridded_bcs.bcs.copy(deep=True)  # Make a copy of the original data in case user cancels
    sim_data.gridded_bcs.bcs = gridded_bcs
    dlg = AddGriddedBcDialog(bc_data=sim_data.gridded_bcs, parent=parent, grid_id=grid_id, time_formats=time_formats)
    if dlg.exec():
        other_grid_bcs = orig_bcs.where(orig_bcs.grid_id != grid_id, drop=True)
        sim_data.gridded_bcs.bcs = xr.concat([other_grid_bcs, dlg.gridded_bcs_table.bc_data.bcs], 'comp_id')
    else:  # Restore original Dataset if user cancels
        sim_data.gridded_bcs.bcs = orig_bcs


class AddGriddedBcButtonDelegate(QStyledItemDelegate):
    """Qt delegate that displays a button for bringing up the dialog to add gridded datasets to a file."""

    def __init__(self, parent, sim_data, time_formats):
        """Initializes the class.

        Args:
            parent (QObject): The parent object
            sim_data (SimData): The simulation data
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        super().__init__(parent)
        self._parent_dlg = parent
        self._sim_data = sim_data
        self._time_formats = time_formats

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Define BCs...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            bool: True if the event is handled
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                row = index.row()
                grid_id = self._sim_data.grid_definitions.coords['grid_id'][row].item()
                run_add_gridded_bcs_dialog(sim_data=self._sim_data, grid_id=grid_id, parent=self._parent_dlg,
                                           time_formats=self._time_formats)
                return True
        return super().editorEvent(event, model, option, index)
