"""Dialog for assigning Material and Sediment Material coverage properties."""
# 1. Standard python modules
import os

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Qt
from PySide2.QtWidgets import (QComboBox, QGroupBox, QHBoxLayout, QHeaderView, QLabel, QMessageBox, QPushButton,
                               QSizePolicy, QSpacerItem, QStyle, QToolBar, QVBoxLayout)

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.guipy.data.polygon_texture import PolygonOptions, PolygonTexture
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.texture_button_delegate import TextureButtonDelegate
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename, get_save_filename
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.resources.resources_util import get_resource_path
from xms.guipy.validators.number_corrector import NumberCorrector
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.guipy.widgets.widget_builder import setup_toolbar

# 4. Local modules
from xms.tuflowfv.data.material_data import DEFAULT_MATERIAL_COLORS, MaterialData
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui.advanced_material_button_delegate import AdvancedMaterialButtonDelegate
from xms.tuflowfv.gui.material_filter_model import MaterialFilterModel


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


CBX_OPT_2DM_FORMAT = 0
CBX_OPT_SHP_FORMAT = 1


class MaterialDialog(XmsDlg):
    """A dialog to define materials and their properties."""

    def __init__(self, title, parent, material_data, display_projection):
        """Initializes the material list and properties dialog.

        Args:
            title (str): Title of the dialog
            parent (QWidget): Parent Qt dialog
            material_data (MaterialData): The material data to edit
            display_projection (Projection): The current display projection. Used for switching unit labels between SI
                and Imperial.
        """
        super().__init__(parent, 'xms.tuflowfv.gui.material_dialog')
        self.parent = parent
        self.widgets = {}
        self.display_projection = display_projection
        self.material_model = None  # QxPandasTableModel for materials spreadsheet
        self.actions = None  # Disable delete of unassigned material
        self.material_data = material_data
        self.mat_att_material_id = -1  # init to -1 so it will be set to UNASSIGNED when the dialog starts
        self.deleted_material_ids = set()
        self.change_all_textures = PolygonOptions()
        self.delegates = dict()
        self.number_corrector = NumberCorrector(self)
        self.delegates['advanced_delegate'] = None
        self.delegates['check_delegate'] = CheckBoxNoTextDelegate(self)
        self.delegates['check_delegate'].state_changed.connect(self.on_toggle_changed)
        self.delegates['material_display'] = TextureButtonDelegate(self)
        self.add_icon = get_resource_path(':/resources/icons/add.svg')
        self.delete_icon = get_resource_path(':/resources/icons/delete.svg')
        self.change_all_icon = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'change-all-mat.svg'
        )

        # Setup the dialog
        self.setWindowTitle(title)
        self.setup_ui()
        self.adjustSize()

        # Calculate a reasonable width for the dialog.
        self.dlg_width = self.widgets['table_view'].horizontalHeader().length()
        self.dlg_width += self.widgets['table_view'].style().pixelMetric(QStyle.PM_ScrollBarExtent)
        self.dlg_width += self.widgets['table_view'].frameWidth() * 2
        self.dlg_width += 20
        self.resize(self.dlg_width, self.size().height())

    def setup_ui(self):
        """Setup dialog widgets."""
        # Add a main vertical layout.
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])

        # set up the materials table view
        self._setup_ui_materials_view()

        # add buttons to add, delete, change all textures for materials
        self._setup_ui_material_buttons()

        # Connect a slots to the material list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        selection_model.selectionChanged.connect(self.on_material_changed)
        if self.parent.selected_material is not None:  # Assign Material command
            current_row = self.parent.selected_material
        else:  # Material List and Properties command
            current_row = MaterialData.UNASSIGNED_MAT
        current_index = self.widgets['table_view'].model().index(current_row, MaterialData.COL_NAME)
        selection_model.setCurrentIndex(current_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)

        # Add widgets for selecting the export format
        self._setup_export_format()

        # add the ok, cancel, help... buttons at the bottom of the dialog
        self._setup_ui_bottom_button_box()

    def _setup_ui_materials_view(self):
        """Sets up the table veiw for the materials."""
        self.widgets['table_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['table_horiz_layout'])

        # Add the material list table view.
        self.widgets['table_view'] = QxTableView()
        df = self.material_data.materials.to_dataframe()
        if not df.index.empty and df.index[0] == 0:
            df.index = df.index + 1  # Start index at 1, not 0
        self.material_model = QxPandasTableModel(df)
        self.widgets['table_horiz_layout'].addWidget(self.widgets['table_view'])

        # Override filter model to enable editing of description.
        self.widgets['table_view'].filter_model = MaterialFilterModel(self)
        self.widgets['table_view'].filter_model.setSourceModel(self.material_model)
        self.widgets['table_view'].setModel(self.widgets['table_view'].filter_model)

        # Set delegate for columns that are toggles.
        toggle_columns = [
            MaterialData.COL_ACTIVE,
            MaterialData.COL_OVERRIDE_BOTTOM_ROUGHNESS,
        ]
        self.material_model.set_checkbox_columns(toggle_columns)
        for toggle_column in toggle_columns:
            self.widgets['table_view'].setItemDelegateForColumn(toggle_column, self.delegates['check_delegate'])

        # Set up the color and texture column to be a button.
        self.widgets['table_view'].setItemDelegateForColumn(MaterialData.COL_COLOR,
                                                            self.delegates['material_display'])

        # Set up delegate for the advanced button column.
        self.delegates['advanced_delegate'] = AdvancedMaterialButtonDelegate(parent=self,
                                                                             display_projection=self.display_projection,
                                                                             set_mat=False)
        self.widgets['table_view'].setItemDelegateForColumn(MaterialData.COL_ADVANCED,
                                                            self.delegates['advanced_delegate'])

        # Hide the ID column.
        self.widgets['table_view'].setColumnHidden(MaterialData.COL_ID, True)
        # Hide the columns we want to handle in the advanced dialog.
        for col in range(MaterialData.COL_ADVANCED + 1, self.material_model.columnCount()):
            self.widgets['table_view'].setColumnHidden(col, True)

        self.widgets['table_view'].horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.widgets['table_view'].horizontalHeader().setStretchLastSection(True)
        self.widgets['table_view'].resizeColumnsToContents()

        # Set the name for the unassigned material to be read only.
        self.material_model.read_only_cells.add((0, MaterialData.COL_NAME))

    def _setup_ui_material_buttons(self):
        """Adds toolbar below the materials table view to add, delete, change all textures."""
        self.widgets['tool_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['tool_horiz_layout'])
        # Add the "+", "-", and change all buttons.
        self.widgets['toolbar'] = QToolBar()
        button_list = [
            [self.add_icon, 'Add Material', self.on_btn_add_material],
            [self.delete_icon, 'Delete Material', self.on_btn_delete_material],
            [self.change_all_icon, 'Change all material textures', self.on_btn_change_all]
        ]
        self.actions = setup_toolbar(self.widgets['toolbar'], button_list)
        change_all_btn = self.widgets['toolbar'].widgetForAction(self.actions[self.change_all_icon])
        texture_icon = DisplayOptionIconFactory.get_icon(self.change_all_textures, change_all_btn.size().width())
        change_all_btn.setIcon(texture_icon)
        self.widgets['tool_horiz_layout'].addWidget(self.widgets['toolbar'])

    def _setup_export_format(self):
        """Adds a combobox for selecting the simulation export format."""
        # Build a layout for the widgets
        layout = QHBoxLayout()
        self.widgets['lbl_global_roughness'] = QLabel('TUFLOWFV simulation export file format:')
        layout.addWidget(self.widgets['lbl_global_roughness'])
        self.widgets['cbx_export_format'] = QComboBox()
        self.widgets['cbx_export_format'].addItem('2dm', '2dm')
        self.widgets['cbx_export_format'].addItem('Shapefile', 'Shapefile')
        self.widgets['cbx_export_format'].currentIndexChanged.connect(self.on_export_format_changed)
        layout.addWidget(self.widgets['cbx_export_format'])
        spacer_item = QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        layout.addSpacerItem(spacer_item)
        # Put the layout in a group box
        self.widgets['grp_export_format'] = QGroupBox()
        self.widgets['grp_export_format'].setTitle('Global export format')
        self.widgets['grp_export_format'].setLayout(layout)
        # Add group box to the dialog layout
        self.widgets['main_vert_layout'].addWidget(self.widgets['grp_export_format'])
        # Initialize value of the combobox
        option_idx = self.widgets['cbx_export_format'].findText(
            str(self.material_data.info.attrs['export_format'])
        )
        self.widgets['cbx_export_format'].setCurrentIndex(option_idx if option_idx > -1 else CBX_OPT_2DM_FORMAT)
        if option_idx == CBX_OPT_SHP_FORMAT:  # Hide Inactive column if shapefile export format
            self.widgets['table_view'].setColumnHidden(MaterialData.COL_ACTIVE, True)

    def _setup_ui_bottom_button_box(self):
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_import'] = QPushButton('Import Material List...')
        self.widgets['btn_import'].clicked.connect(self.on_btn_import)
        self.widgets['btn_export'] = QPushButton('Export Material List...')
        self.widgets['btn_export'].clicked.connect(self.on_btn_export)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_import'])
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_export'])
        self.widgets['btn_box'] = gui_util.build_ok_cancel_buttons(self)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])
        self.widgets['main_vert_layout'].addLayout(self.widgets['btn_horiz_layout'])

    def _get_new_material_name(self):
        """Get a unique name for a new material.

        Returns:
            (str): See description
        """
        mat_name = 'new material'
        unique = False
        i = 1
        while not unique:
            unique = True
            for row in self.material_model.data_frame['name']:
                if str(row) == mat_name:
                    unique = False
                    mat_name = f'new material ({i})'
                    i += 1
                    break
        return mat_name

    def on_btn_change_all(self):
        """Called when the change all material texture/style button is clicked."""
        dlg = PolygonDisplayOptionsDialog(self.change_all_textures, self)
        dlg.ui.lab_line_color.setVisible(False)
        dlg.ui.color_btn.setVisible(False)
        if dlg and dlg.exec():
            self.change_all_textures = dlg.get_options()
            new_opts = self.delegates['material_display'].poly_opts_to_str(self.change_all_textures)
            texture = new_opts.split()[-1]
            for row in range(self.material_model.rowCount()):
                data = self.material_model.data(self.material_model.index(row, MaterialData.COL_COLOR), Qt.EditRole)
                lst = data.split()
                data = f'{lst[0]} {lst[1]} {lst[2]} {texture}'
                self.material_model.setData(self.material_model.index(row, MaterialData.COL_COLOR), data, Qt.EditRole)

    def on_btn_add_material(self):
        """Called when the add material button is clicked."""
        mat_id = max(self.material_model.data_frame['id']) + 1
        # Get a unique name
        description = self._get_new_material_name()
        default_data = self.material_data.default_data_dict()
        # Randomize the color
        row_idx = self.material_model.rowCount()
        mat_color = DEFAULT_MATERIAL_COLORS[row_idx % len(DEFAULT_MATERIAL_COLORS)]
        # We decided we want all materials (except unassigned) to have a solid pattern by default.
        default_data['color'] = [f'{mat_color[0]} {mat_color[1]} {mat_color[2]} {int(PolygonTexture.null_pattern)}']
        default_data['id'] = mat_id
        default_data['name'] = description
        self.material_model.set_default_values(default_data)
        row_idx = self.material_model.rowCount()
        self.material_model.insertRows(row_idx, 1)

        # Connect a slots to the material list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        new_mat_index = self.widgets['table_view'].model().index(row_idx, MaterialData.COL_NAME)
        selection_model.setCurrentIndex(new_mat_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)
        self.widgets['table_view'].resizeColumnsToContents()
        self.widgets['table_view'].filter_model.headerDataChanged.emit(Qt.Horizontal, 0, 0)

    def on_btn_delete_material(self):
        """Called when the delete material button is clicked."""
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        if len(indices) <= 0:
            return
        rows = dict()
        for index in indices:
            if index.row() != MaterialData.UNASSIGNED_MAT:
                rows[index.row()] = index

        for row in sorted(rows.keys(), reverse=True):
            index = rows[row]
            # Keep track of deleted material ids so we can reassign polygons in XMS.
            delete_id_index = self.material_model.index(index.row(), MaterialData.COL_ID)
            mat_id = int(float(self.material_model.data(delete_id_index, Qt.DisplayRole)))
            self.deleted_material_ids.add(mat_id)
            self.material_model.removeRows(index.row(), 1)

        # select the row above the first currently selected row
        row = sorted(rows.keys())[0] - 1
        select_index = self.widgets['table_view'].model().index(row, MaterialData.COL_NAME)
        self.widgets['table_view'].selectionModel().setCurrentIndex(
            select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
        )

    def on_toggle_changed(self, index):
        """Called when the use default checkbox is toggled for a row.

        Args:
            index (QModelIndex): Model index of the toggle whose state changed
        """
        if not index.isValid():
            return
        row = index.row()
        col = index.column()

        update_indices = []
        table_view = self.widgets['table_view']
        if col == MaterialData.COL_OVERRIDE_BOTTOM_ROUGHNESS:
            update_indices.append(table_view.filter_model.index(row, MaterialData.COL_BOTTOM_ROUGHNESS))
        for update_idx in update_indices:
            table_view.update(update_idx)

    def on_material_changed(self):
        """Called when the material row selection changes."""
        # Disable the delete button if the unassigned material currently selected.
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        enable_delete = True
        # if the 'unassigned' material is select then the user can not delete
        for index in indices:
            if index.row() == MaterialData.UNASSIGNED_MAT:
                enable_delete = False
        self.widgets['toolbar'].widgetForAction(self.actions[self.delete_icon]).setEnabled(enable_delete)

    def on_export_format_changed(self, index):
        """Called when the export format combobox option is changed.

        Args:
            index (int): Combobox option index of the new BC type
        """
        # Inactive option only works with materials specified in a .2dm file.
        self.widgets['table_view'].setColumnHidden(MaterialData.COL_ACTIVE, index == CBX_OPT_SHP_FORMAT)

    def on_btn_import(self):
        """Imports TUFLOWFV material properties."""
        if self.parent is None:
            msg_box = QMessageBox(self)
            msg_box.setWindowTitle('Information')
            msg_box.setText('Dialog must be used inside of parent dialog for import to function.')
            msg_box.exec()
            return

        file_filter = 'TUFLOWFV materials (*.tuflowfv_mat);;All files (*.*)'
        filename = get_open_filename(parent=self, caption='Open', file_filter=file_filter)
        if filename:
            file_data = XarrayBase(filename)
            file_type = file_data.info.attrs.get('FILE_TYPE', '')
            if file_type == self.material_data.info.attrs['FILE_TYPE']:
                material_data = MaterialData(filename)
                self.parent.import_file(material_data)
            else:
                msg_box = QMessageBox(self)
                msg_box.setWindowTitle('Import Materials')
                msg_box.setText(f'File failed to import file.\n{filename}')
                msg_box.exec()

    def on_btn_export(self):
        """Exports TUFLOWFV material properties to a file."""
        selected_filter = 'TUFLOWFV materials (*.tuflowfv_mat)'
        file_filter = f'{selected_filter};;All files (*.*)'
        filename = get_save_filename(parent=self, selected_filter=selected_filter, file_filters=file_filter)
        if filename:
            self._export_materials_to_file(filename)

    def _export_materials_to_file(self, filename):
        """Writes materials to a file.

        Args:
            filename (str): file name
        """
        mat_data = MaterialData(filename)
        if self.widgets['cbx_export_format'].currentIndex() == CBX_OPT_SHP_FORMAT:  # Clear Inactive flags if not .2dm
            self.material_model.data_frame = self.material_model.data_frame.assign(inactive=0)
        mat_data.materials = self.material_model.data_frame.to_xarray()
        if os.path.exists(filename) and os.path.isfile(filename):
            os.remove(filename)
        mat_data.commit()

    def accept(self):
        """Save material properties and block accepted signal if material names are no longer unique."""
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        mat_names = self.material_model.data_frame['name'].to_list()
        set_names = set()
        for name in mat_names:
            num_names = len(set_names)
            set_names.add(name)
            if num_names == len(set_names):
                msg = f'Material: "{name}" is not a unique name.\n' \
                      f'Material names must be unique. Edit the names to ensure that they are unique.'
                message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
                return

        # If we came from the Assign Material command, make sure we have a single material row selected.
        if self.parent.selected_material is not None:
            indices = self.widgets['table_view'].selectionModel().selectedIndexes()
            selected_rows = {index.row() for index in indices if index.isValid()}
            if len(selected_rows) != 1:
                msg = 'Please select a single material row to update the assignment of the selected polygon(s).'
                message_box.message_with_ok(parent=self, message=msg, app_name=app_name, icon='Critical',
                                            win_icon=self.windowIcon())
                return
            self.parent.selected_material = indices[0].row()

        # Save the material list and global attributes.
        format_idx = self.widgets['cbx_export_format'].currentIndex()
        if format_idx == CBX_OPT_SHP_FORMAT:  # Clear Inactive flags if not .2dm
            self.material_model.data_frame = self.material_model.data_frame.assign(inactive=0)
        self.material_data.materials = self.material_model.data_frame.to_xarray()
        self.material_data.info.attrs['export_format'] = self.widgets['cbx_export_format'].itemData(format_idx)
        if self.parent is not None:
            self.parent.accept()
        else:  # pragma: no cover
            super().accept()

    def reject(self):
        """Call reject slot on parent if we have one."""
        if self.parent is not None:
            self.parent.reject()
        else:  # pragma: no cover
            super().reject()
