"""Qt delegate for a tree item selector button."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSize, Qt, Signal
from PySide2.QtGui import QBrush, QColor, QFontMetrics, QPixmap
from PySide2.QtWidgets import QApplication, QPushButton, QStyle, QStyledItemDelegate, QStyleOptionToolButton

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.tuflowfv.gui import gui_util

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def filter_simulation_from_tree(tree_node, sim_uuid):
    """Filters a specific TUFLOWFV simulation from the tree so it is not a candidate for selection.

    Args:
        sim_uuid (str): UUID of the parent simulation - don't want to link ourselves
        tree_node (TreeNode): The root project explorer tree

    Returns:
        TreeNode: The project tree without the specified simulation.
    """
    petc = tree_util.ProjectExplorerTreeCreator()
    tree_copy = petc.copy(tree_node)

    def selectable(t):
        return t.item_typename != 'TI_DYN_SIM' or t.model_name != 'TUFLOWFV' or t.uuid != sim_uuid
    tree_util.filter_project_explorer(tree_copy, selectable)
    return tree_copy


def select_child_simulation(sim_uuid, tree_node, previous_selection, parent):
    """Display a dialog to select a linked child simulation.

    Args:
        sim_uuid (str): UUID of the parent simulation - don't want to link ourselves
        tree_node (TreeNode): The root project explorer tree
        previous_selection (str): UUID of the previous selection, if any
        parent (QDialog): The parent Qt dialog. Needs to be something that won't blow if windowIcon() is called.

    Returns:
        Union[str, None]: The UUID of the selected shapefile. None if the user canceled. Empty string if the user
            cleared the selection.
    """
    item_uuid = None
    filtered_tree = filter_simulation_from_tree(tree_node, sim_uuid)
    dlg = TreeItemSelectorDlg(title='Select Child TUFLOWFV Simulation', target_type=type(None),
                              previous_selection=previous_selection, pe_tree=filtered_tree, show_root=False,
                              parent=parent, selectable_xms_types=['TI_DYN_SIM'])
    if not dlg.has_selectable_items():
        msg = 'No other TUFLOWFV simulations exist. Define a separate TUFLOWFV simulation to link it as a child.'
        message_with_ok(parent=parent, message=msg, app_name='SMS', win_icon=parent.windowIcon())
        item_uuid = ''  # Clear any previous selection, it doesn't exist anymore
    elif dlg.exec():
        item_uuid = dlg.get_selected_item_uuid()
        item_uuid = item_uuid if item_uuid else ''  # Return empty string if user OKs and cleared a previous selection
    return item_uuid


def select_shapefile_or_coverage(tree_node, previous_selection, tree_type, layer_type, parent):
    """Display a dialog to select a Z line shapefile in SMS.

    Args:
        tree_node (TreeNode): The root project explorer tree
        previous_selection (str): UUID of the previous selection, if any
        tree_type (str): The tree item type: TI_GENERIC_ARC=line shapefiles, TI_GENERIC_POINT=point shapefiles
        layer_type (str): The type of layer being selected, e.g. 'Z', 'Output'
        parent (QDialog): The parent Qt dialog. Needs to be something that won't blow if windowIcon() is called.

    Returns:
        Union[str, None]: The UUID of the selected shapefile. None if the user canceled. Empty string if the user
            cleared the selection.
    """
    shape_type = 'Line' if tree_type == 'TI_GENERIC_ARC' else 'Point'
    item_uuid = None
    dlg = TreeItemSelectorDlg(title=f'Select {layer_type} {shape_type} Layer', target_type=type(None),
                              previous_selection=previous_selection, pe_tree=tree_node, show_root=False,
                              parent=parent, selectable_xms_types=[tree_type, 'TI_COVER'])
    if not dlg.has_selectable_items():
        shape_type = shape_type.lower()
        msg = f'No {shape_type} shapefiles or coverages exist. Load a {shape_type} shapefile/coverage into SMS to ' \
              f'select it as a {layer_type} {shape_type} input.'
        message_with_ok(parent=parent, message=msg, app_name='SMS', win_icon=parent.windowIcon())
        item_uuid = ''  # Clear any previous selection, it doesn't exist anymore
    elif dlg.exec():
        item_uuid = dlg.get_selected_item_uuid()
        item_uuid = item_uuid if item_uuid else ''  # Return empty string if user OKs and cleared a previous selection
    return item_uuid


def select_output_points_cov(tree_node, previous_selection, parent):
    """Display a dialog to select an output points feature coverage geometry of any coverage type.

    Args:
        tree_node (TreeNode): The root project explorer tree
        previous_selection (str): UUID of the previous selection, if any
        parent (QDialog): The parent Qt dialog. Needs to be something that won't blow if windowIcon() is called.

    Returns:
        Union[str, None]: The UUID of the selected shapefile. None if the user canceled. Empty string if the user
            cleared the selection.
    """
    item_uuid = None
    dlg = TreeItemSelectorDlg(title='Select Output Points Coverage', target_type=type(None),
                              previous_selection=previous_selection, pe_tree=tree_node, show_root=False,
                              parent=parent, selectable_xms_types=['TI_COVER'])
    if not dlg.has_selectable_items():
        msg = 'No coverages exist. Load a coverage into SMS to use it as the locations for this output points block.'
        message_with_ok(parent=parent, message=msg, app_name='SMS', win_icon=parent.windowIcon())
        item_uuid = ''  # Clear any previous selection, it doesn't exist anymore
    elif dlg.exec():
        item_uuid = dlg.get_selected_item_uuid()
        item_uuid = item_uuid if item_uuid else ''  # Return empty string if user OKs and cleared a previous selection
    return item_uuid


class TreeItemSelectorDelegate(QStyledItemDelegate):
    """Qt delegate for a tree item selector button."""
    state_changed = Signal()

    def __init__(self, tree_type, tree_node, parent, sim_uuid=None, layer_type=''):
        """Initializes the class.

        Args:
            tree_type (str): The tree item type: TI_GENERIC_ARC=line Z shapefiles, TI_GENERIC_POINT=point Z shapefiles,
                TI_DYN_SIM=child simulations
            tree_node (TreeNode): The root project explorer tree
            parent (QObject): The parent dialog
            sim_uuid (Optional[str]): UUID of the simulation selecting data. Only need to provide if
                tree_type==TI_DYN_SIM (selecting a linked child simulation).
            layer_type (Optional[str]): The type of the layer to select, e.g. 'Z', 'Output'
        """
        super().__init__(parent)
        self._uuid_to_tree_name = {}
        self._tree_node = tree_node
        self._tree_type = tree_type
        self._sim_uuid = sim_uuid
        self._layer_type = layer_type
        self._fill_uuid_to_tree_name(self._tree_node)

    def _fill_uuid_to_tree_name(self, tree_node):
        """Fills in a dictionary of UUIDs to names in the project explorer tree.

        Args:
            tree_node (TreeNode): The current node in the project explorer tree.
        """
        if not tree_node:
            return
        if tree_node.uuid:
            self._uuid_to_tree_name[tree_node.uuid] = tree_node.name
        for child in tree_node.children:
            self._fill_uuid_to_tree_name(child)

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            item_uuid = index.data()
            if item_uuid and item_uuid in self._uuid_to_tree_name:
                gui_util.set_widget_text_to_tree_path(tree_node=self._tree_node, item_uuid=item_uuid, widget=btn)
            else:
                btn.setText(gui_util.NULL_SELECTION)
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the XY series editor button is clicked.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                item_uuid = index.data()
                if self._tree_type == 'TI_DYN_SIM':  # Select a child simulation
                    item_uuid = select_child_simulation(sim_uuid=self._sim_uuid, tree_node=self._tree_node,
                                                        previous_selection=item_uuid, parent=self.parent())
                else:  # Select a Z line/point shapefile
                    item_uuid = select_shapefile_or_coverage(tree_node=self._tree_node, previous_selection=item_uuid,
                                                             tree_type=self._tree_type, layer_type=self._layer_type,
                                                             parent=self.parent())
                if item_uuid is not None:  # Only update model if user accepted dialog
                    model.setData(index, item_uuid)
                    self.state_changed.emit()
                return True
        return super().editorEvent(event, model, option, index)

    def sizeHint(self, option, index):  # noqa: N802
        """Help keep the size adjusted for custom painted combobox.

        Args:
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            (QSize): An appropriate size hint
        """
        hint = super().sizeHint(option, index)
        item_uuid = index.data(Qt.EditRole)
        item = tree_util.find_tree_node_by_uuid(self._tree_node, item_uuid)
        if item:
            btn_text = item.name
        else:
            btn_text = gui_util.NULL_SELECTION

        fm = QFontMetrics(option.font)
        btn_opt = QStyleOptionToolButton()
        btn_opt.rect = option.rect
        btn_opt.state = option.state | QStyle.State_Enabled
        btn_opt.text = btn_text
        return QApplication.style().sizeFromContents(QStyle.CT_ToolButton, btn_opt,
                                                     QSize(fm.boundingRect(btn_text).width(), hint.height()))
