"""Qt table view for specifying Z modifications."""
# 1. Standard python modules

# 2. Third party modules
import numpy as np
from PySide2.QtGui import QDoubleValidator
from PySide2.QtWidgets import QAbstractItemView, QHeaderView

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate
from xms.guipy.delegates.user_data_combobox_delegate import UserDataComboboxDelegate
from xms.guipy.validators.number_corrector import NumberCorrector

# 4. Local modules
from xms.tuflowfv.data import sim_data as smd
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui import z_modifications_filter_model as zfm
from xms.tuflowfv.gui.tree_item_selector_delegate import TreeItemSelectorDelegate
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget
from xms.tuflowfv.gui.zpoints_button_delegate import ZPointsButtonDelegate


class ZModificationsTableWidget(TuflowfvTableWidget):
    """The z modifications table widget."""

    def __init__(self, tree_node, data_frame, sim_data, parent):
        """Construct the widget.

        Args:
            tree_node (TreeNode): The root project explorer tree node
            data_frame (pd.DataFrame): The model data
            sim_data (SimData): SimData object. Needed for resolving relative paths
            parent (QDialog): The parent dialog
        """
        if len(data_frame.index) > 0:
            data_frame.index = np.arange(1, len(data_frame) + 1)
        super().__init__(parent=parent, data_frame=data_frame, select_col=2, default_values=self._get_default_values())
        self._data = sim_data
        self._tree_node = tree_node
        self.setup_ui()

    def _get_default_values(self):
        """Returns a dict of default values."""
        return {
            smd.ELEV_MOD_TYPE_VAR: smd.ELEV_TYPE_SET_ZPTS,
            smd.ELEV_SET_ZPTS_VAR: 0.0,
            smd.ELEV_GRID_ZPTS_VAR: '',
            smd.ELEV_CSV_FILE_VAR: '',
            smd.ELEV_CSV_TYPE_VAR: smd.CELL_ELEV_CSV_TYPE_CELL,
            smd.ELEV_ZLINE_UUID_VAR: '',
            smd.ELEV_ZPOINT1_UUID_VAR: '',
            smd.ELEV_ZPOINT2_UUID_VAR: '',
            smd.ELEV_ZPOINT3_UUID_VAR: '',
            smd.ELEV_ZPOINT4_UUID_VAR: '',
            smd.ELEV_ZPOINT5_UUID_VAR: '',
            smd.ELEV_ZPOINT6_UUID_VAR: '',
            smd.ELEV_ZPOINT7_UUID_VAR: '',
            smd.ELEV_ZPOINT8_UUID_VAR: '',
            smd.ELEV_ZPOINT9_UUID_VAR: '',
        }

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        mod_types = {
            zfm.COL_TEXT_SET_ZPTS: smd.ELEV_TYPE_SET_ZPTS,
            zfm.COL_TEXT_GRID_ZPTS: smd.ELEV_TYPE_GRID_ZPTS,
            zfm.COL_TEXT_CSV_FILE: smd.ELEV_TYPE_CELL_CSV,
            zfm.COL_TEXT_LINE_SHAPE: smd.ELEV_TYPE_ZLINE,
        }
        type_cbx_delegate = UserDataComboboxDelegate(strings=mod_types, data_type=int, parent=self)
        dbl_validator = QDoubleValidator(self)
        dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        set_zpts_delegate = EditFieldValidator(dbl_validator, self)
        grid_zpts_file_delegate = FileSelectorButtonDelegate(
            proj_dir=self._data.info.attrs['proj_dir'],
            caption='Select a GRID Zpts Raster File',
            parent=self.parent(),
            file_filters=gui_util.GRID_ZPTS_FILE_FILTER
        )
        csv_file_delegate = FileSelectorButtonDelegate(
            proj_dir=self._data.info.attrs['proj_dir'],
            caption='Select a Cell Elevation CSV File',
            parent=self.parent(),
            file_filters=gui_util.CSV_FILE_FILTER
        )
        strings = {'Cell ID': smd.CELL_ELEV_CSV_TYPE_CELL, 'Coordinate': smd.CELL_ELEV_CSV_TYPE_COORD}
        csv_cbx_delegate = UserDataComboboxDelegate(strings=strings, parent=self)
        zlines_delegate = TreeItemSelectorDelegate(tree_type='TI_GENERIC_ARC', tree_node=self._tree_node,
                                                   layer_type='Z', parent=self.parent())
        zpoints_delegate = ZPointsButtonDelegate(parent=self.parent(), tree_node=self._tree_node)
        delegates = {
            zfm.COL_MODIFICATION_TYPE: type_cbx_delegate,
            zfm.COL_SET_ZPTS: set_zpts_delegate,
            zfm.COL_GRID_ZPTS: grid_zpts_file_delegate,
            zfm.COL_CSV_FILE: csv_file_delegate,
            zfm.COL_CSV_TYPE: csv_cbx_delegate,
            zfm.COL_LINE_SHAPE: zlines_delegate,
            zfm.COL_POINT_SHAPES: zpoints_delegate
        }
        filter_model = zfm.ZModificationsFilterModel(self)
        self.set_cbx_columns_in_model = False
        super()._setup_ui(column_delegates=delegates, stretch_last_section=True, fixed_size=False,
                          filter_model=filter_model, movable_rows=True)

        # Connect slots/signals to resize column width when user selects a new file or tree item
        grid_zpts_file_delegate.state_changed.connect(self.resize_columns)
        csv_file_delegate.state_changed.connect(self.resize_columns)
        zlines_delegate.state_changed.connect(self.resize_columns)

        self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table_view.horizontalHeader().setStretchLastSection(True)
        # Hide the columns we want to handle in the z points dialog.
        for col in range(zfm.COL_POINT_SHAPES + 1, self.model.columnCount()):
            self.table_view.setColumnHidden(col, True)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def resize_columns(self):
        """Calls resizeColumnsToContents() and then resets the last column to stretch."""
        self.table_view.resizeColumnsToContents()
        # resizeColumnsToContents() will turn off setStretchLastSection, so we need to explicitly reset it on the last
        # visible column
        self.table_view.horizontalHeader().setSectionResizeMode(zfm.COL_POINT_SHAPES, QHeaderView.Stretch)

    def on_btn_add_row(self):
        """Overload to resize columns to contents when a row is added."""
        super().on_btn_add_row()
        self.resize_columns()
