"""Module for the get_model function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['get_model']

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import DateTimeColumnType, FloatColumnType, TableDefinition

# 4. Local modules


def get_model() -> GenericModel:
    """Create the hard bottom data definition."""
    gm = GenericModel(exclusive_point_conditions=True, exclusive_arc_conditions=True)

    pp = gm.point_parameters

    test = pp.add_group(group_name='output_point', label='Output Point')
    test.add_text('name', 'Name')

    ap = gm.arc_parameters
    ap.add_group(group_name='input', label='Input')
    ap.add_group(group_name='lateral', label='Lateral')

    params = gm.global_parameters.add_group('parameters', 'Parameters', is_active=True)

    # Time Step Information
    params.add_float('maxGlobalDt', 'Maximum Global Timestep (seconds)', 0.0)
    params.add_float('maxCFLXY', 'Maximum CFL Time Step for x-y (seconds)', 0.0)
    params.add_float('maxCFLKTheta', 'Maximum CDF Time Step for k-theta (seconds)', 0.0)
    params.add_float('minSourceDt', 'Minimum Source Term Time Step (seconds)', 0.0)

    # Spectral Parameterization
    params.add_float('FREQ1', 'First frequency (Hz) [FREQ1]', 0.0)
    params.add_integer('NK', 'Number of frequencies [NK]', 0)
    params.add_integer('NTH', 'Number of direction bins [NTH]', 0)
    params.add_float('THOFF', 'Relative offset of first direction [THOFF]', 0.0, high=0.5, low=-0.5)
    params.add_float('XFR', 'Frequency increment [XFR]', 0.0)

    # Grid Limits
    params.add_float('ZLIM', 'Coastline limit depth (m) [ZLIM]', -0.1)
    params.add_float('DMIN', 'Abs. minimum water depth (m) [DMIN]', 0.3)

    consts = gm.global_parameters.add_group('consts', 'Constants in source terms')

    # Exponential Input
    # Only allow SIN3 at the moment
    consts.add_option(
        'exponential_input',
        'Exponential Input',
        default='WAM4 and Variants [SIN3]',
        options=[
            # 'WAM-3 [SIN1]',
            # 'Tolman and Chalikov [SIN2]',
            'WAM4 and Variants [SIN3]',
            # 'BYDRZ input [SIN6]'
        ]
    )

    consts.add_float('ZWND', 'Height of Wind (m) [ZWND]', 0.0)
    consts.add_float('ALPHA0', 'Minimum Value of Charnock Coefficient [ALPHA0', 0.0)
    consts.add_float('Z0MAX', 'Maximum Value of Air-Side Roughness z0 [Z0MAX]', 0.0)
    consts.add_float('BETAMAX', 'Maximum Value of Wind-Wave Coupling [BETAMAX]', 1.43)
    consts.add_float('SINTHP', 'Power of Cosine in Wind Input [SINTHP]', 0.0)
    consts.add_float('ZALP', 'Wave Age Shift to Account for Gustiness [ZALP]', 0.0)
    consts.add_float('TAUWSHELTER', 'Sheltering of Short Waves to Reduce u_star [TAUWSHELTER]', 0.3)
    desc = 'Choice of Swell Attenuation Formulation [SWELLFPAR]'
    swellfpar = consts.add_option('SWELLFPAR', desc, default='TC 1996', options=['TC 1996', 'ACC 2008'])
    consts.add_float('SWELLF', 'Swell Attenuation Factor [SWELLF]', 0.0)

    # Extra Parameters for ACC 2008
    flags = {'TC 1996': False, 'ACC 2008': True}
    consts.add_float('SWELLF2', 'Swell Attenuation Factor [SWELLF2]', 0.0).add_dependency(swellfpar, flags)
    consts.add_float('SWELLF3', 'Swell Attenuation Factor [SWELLF3]', 0.0).add_dependency(swellfpar, flags)
    consts.add_float('SWELLF4', 'Threshold Reynolds Number for ACC2008 [SWELLF4]', 0.0).add_dependency(swellfpar, flags)
    desc = 'Relative Viscous Decay Below Threshold [SWELLF5]'
    consts.add_float('SWELLF5', desc, 0.0).add_dependency(swellfpar, flags)
    consts.add_float('Z0RAT', 'Roughness for Oscil. Flow / Mean Flow [Z0RAT]', 0.0).add_dependency(swellfpar, flags)

    # Bottom Friction - JONSWAP [SBT1]
    consts.add_float('GAMMA', 'Bottom Friction Empirical Constant [GAMMA]', 0.0)

    hidden = {'TC 1996': False, 'ACC 2008': False}
    # FLDRY is hidden in the model control
    consts.add_integer('FLDRY', 'FLDRY', 0).add_dependency(swellfpar, hidden)

    # FLCX is hidden in the model control
    consts.add_integer('FLCX', 'FLCX', 1).add_dependency(swellfpar, hidden)

    # FLCY is hidden in the model control
    consts.add_integer('FLCY', 'FLCY', 1).add_dependency(swellfpar, hidden)

    # FLCTH is hidden in the model control
    consts.add_integer('FLCTH', 'FLCTH', 1).add_dependency(swellfpar, hidden)

    # FLCK is hidden in the model control
    consts.add_integer('FLCK', 'FLCK', 1).add_dependency(swellfpar, hidden)

    # FLSOU is hidden in the model control
    consts.add_integer('FLSOU', 'FLSOU', 1).add_dependency(swellfpar, hidden)

    # Source Terms (all hidden in the model control)
    consts.add_integer('stresses', 'stresses', 0).add_dependency(swellfpar, hidden)
    consts.add_float('FLX3CDMAX', 'FLX3CDMAX', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('FLX3CTYPE', 'FLX3CTYPE', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('FLX4CDFAC', 'FLX4CDFAC', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SLN1CLIN', 'SLN1CLIN', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SLN1RFPM', 'SLN1RFPM', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SLN1RFHF', 'SLN1RFHF', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN1CINP', 'SIN1CINP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2SWELLF', 'SIN2SWELLF', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2STABSH', 'SIN2STABSH', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2STABOF', 'SIN2STABOF', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2CNEG', 'SIN2CNEG', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2CPOS', 'SIN2CPOS', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN2FNEG', 'SIN2FNEG', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SIN6SINA0', 'SIN6SINA0', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIN6SINU10', 'SIN6SINU10', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SNL1LAMBDA', 'SNL1LAMBDA', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1NLPROP', 'SNL1NLPROP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1KDCONV', 'SNL1KDCONV', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1KDMIN', 'SNL1KDMIN', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1SNLCS1', 'SNL1SNLCS1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1SNLCS2', 'SNL1SNLCS2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL1SNLCS3', 'SNL1SNLCS3', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('non_linear_interactions', 'non_linear_interactions', 0).add_dependency(swellfpar, hidden)

    consts.add_float('SNL2IQTYPE', 'SNL2IQTYPE', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL2TAILNL', 'SNL2TAILNL', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SNL3MSC', 'SNL3MSC', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL3NSC', 'SNL3NSC', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL3KDFD', 'SNL3KDFD', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNL3KDFS', 'SNL3KDFS', 0.0).add_dependency(swellfpar, hidden)

    consts.add_integer('SNL4INDTSA', 'SNL4INDTSA', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('SNL4ALTLP', 'SNL4ALTLP', 0).add_dependency(swellfpar, hidden)

    consts.add_float('SNLSA34', 'SNLSA34', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNLSFHFC', 'SNLSFHFC', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNLSDMN', 'SNLSDMN', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SNLSFC13', 'SNLSFC13', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SDS1CDIS', 'SDS1CDIS', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS1APM', 'SDS1APM', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('whitecapping_dissipation', 'whitecapping_dissipation', 0).add_dependency(swellfpar, hidden)

    consts.add_float('SDS2SDSA0', 'SDS2SDSA0', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS2SDSA1', 'SDS2SDSA1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS2SDSA2', 'SDS2SDSA2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS2SDSB0', 'SDS2SDSB0', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS2SDSB1', 'SDS2SDSB1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS2PHIMIN', 'SDS2PHIMIN', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SDS3SDSC1', 'SDS3SDSC1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3MNMEANP', 'SDS3MNMEANP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3WNMEANPTAIL', 'SDS3WNMEANPTAIL', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSDELTA1', 'SDS3SDSDELTA1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSDELTA2', 'SDS3SDSDELTA2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSLF', 'SDS3SDSLF', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSHF', 'SDS3SDSHF', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSC2', 'SDS3SDSC2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSC4', 'SDS3SDSC4', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSBR', 'SDS3SDSBR', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSP', 'SDS3SDSP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSBR2', 'SDS3SDSBR2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSC5', 'SDS3SDSC5', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSC6', 'SDS3SDSC6', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSDTH', 'SDS3SDSDTH', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SDS6SDSET', 'SDS6SDSET', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS6SDSA1', 'SDS6SDSA1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS6SDSA2', 'SDS6SDSA2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS6SDSP1', 'SDS6SDSP1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS6SDSP2', 'SDS6SDSP2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SDS3SDSDTH', 'SDS3SDSDTH', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SWL6SWLB1', 'SWL6SWLB1', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SBT1GAMMA', 'SBT1GAMMA', 0.0).add_dependency(swellfpar, hidden)

    consts.add_integer('SIC2IC2DISPER', 'SIC2IC2DISPER', 0).add_dependency(swellfpar, hidden)
    consts.add_float('SIC2IC2TURB', 'SIC2IC2TURB', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIC2IC2ROUGH', 'SIC2IC2ROUGH', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIC2IC2REYNOLDS', 'SIC2IC2REYNOLDS', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIC2IC2SMOOTH', 'SIC2IC2SMOOTH', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIC2IC2VISC', 'SIC2IC2VISC', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('SIS2ISC1', 'SIS2ISC1', 1.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2BACKSCAT', 'SIS2IS2BACKSCAT', 1.0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIS2IS2BREAK', 'SIS2IS2BREAK', 0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2C1', 'SIS2IS2C1', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2C2', 'SIS2IS2C2', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2C3', 'SIS2IS2C3', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2ISBACKSCAT', 'SIS2ISBACKSCAT', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIS2IS2DISP', 'SIS2IS2DISP', 0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2FRAGILITY', 'SIS2FRAGILITY', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2DMIN', 'SIS2IS2DMIN', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIS2IS2DAMP', 'SIS2IS2DAMP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIS2', 'SIS2', 0).add_dependency(swellfpar, hidden)

    consts.add_float('STR1PTRIAD1', 'STR1PTRIAD1', 0.05).add_dependency(swellfpar, hidden)
    consts.add_float('STR1PTRIAD2', 'STR1PTRIAD2', 2.5).add_dependency(swellfpar, hidden)
    consts.add_float('STR1PTRIAD3', 'STR1PTRIAD3', 10.0).add_dependency(swellfpar, hidden)
    consts.add_float('STR1PTRIAD4', 'STR1PTRIAD4', 0.2).add_dependency(swellfpar, hidden)
    consts.add_float('STR1PTRIAD5', 'STR1PTRIAD5', 0.01).add_dependency(swellfpar, hidden)

    consts.add_float('REF1REFCOAST', 'REF1REFCOAST', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFFREQ', 'REF1REFFREQ', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFMAP', 'REF1REFMAP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFRMAX', 'REF1REFRMAX', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFFREQPOW', 'REF1REFFREQPOW', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFICEBERG', 'REF1REFICEBERG', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFSUBGRID', 'REF1REFSUBGRID', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('REF1REFCOSP_STRAIGHT', 'REF1REFCOSP_STRAIGHT', 0.0).add_dependency(swellfpar, hidden)

    consts.add_integer('SIG1IGMETHOD', 'SIG1IGMETHOD', 0).add_dependency(swellfpar, hidden)
    consts.add_float('SIG1IGADDOUTP', 'SIG1IGADDOUTP', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIG1IGSOURCE', 'SIG1IGSOURCE', 1).add_dependency(swellfpar, hidden)
    consts.add_float('SIG1IGSTERMS', 'SIG1IGSTERMS', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIG1IGMAXFREQ', 'SIG1IGMAXFREQ', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('SIG1IGEMPIRICAL', 'SIG1IGEMPIRICAL', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIG1IGBCOVERWRITE', 'SIG1IGBCOVERWRITE', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('SIG1IGSWELLMAX', 'SIG1IGSWELLMAX', 0).add_dependency(swellfpar, hidden)

    consts.add_float('PROCFLTM', 'PROCFLTM', 0.0).add_dependency(swellfpar, hidden)
    consts.add_integer('propagation_schemes', 'propagation_schemes', 0).add_dependency(swellfpar, hidden)

    consts.add_float('PRO2DTIME', 'PRO2DTIME', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('PRO2LATMIN', 'PRO2LATMIN', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('PRO3WDTHCG', 'PRO3WDTHCG', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('PRO3WDTHTH', 'PRO3WDTHTH', 0.0).add_dependency(swellfpar, hidden)

    consts.add_float('PSMCCFLTM', 'PSMCCFLTM', 0.7).add_dependency(swellfpar, hidden)
    consts.add_float('PSMCDTIME', 'PSMCDTIME', 0.0).add_dependency(swellfpar, hidden)
    consts.add_float('PSMCLATMIN', 'PSMCLATMIN', 86.0).add_dependency(swellfpar, hidden)
    consts.add_float('PSMCRFMAXD', 'PSMCRFMAXD', 80.0).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCLvSMC', 'PSMCLvSMC', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCISHFT', 'PSMCISHFT', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCJEQT', 'PSMCJEQT', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCNBISMC', 'PSMCNBISMC', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCUNO3', 'PSMCUNO3', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('PSMCUNO3Add', 'PSMCUNO3Add', 0).add_dependency(swellfpar, hidden)

    consts.add_integer('E3D', 'E3D', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('I1E3D', 'I1E3D', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('I2E3D', 'I2E3D', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('TH1MF', 'TH1MF', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('I1TH1MF', 'I1TH1MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('I2TH1MF', 'I2TH1MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('STH1MF', 'STH1MF', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('I1STH1MF', 'I1STH1MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('I2STH1MF', 'I2STH1MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('TH2MF', 'TH2MF', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('I1TH2MF', 'I1TH2MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('I2TH2MF', 'I2TH2MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('STH2MF', 'STH2MF', 0).add_dependency(swellfpar, hidden)
    consts.add_integer('I1STH2MF', 'I1STH2MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('I2STH2MF', 'I2STH2MF', 1).add_dependency(swellfpar, hidden)
    consts.add_integer('P2SFOut', 'P2SFOut', 0).add_dependency(swellfpar, hidden)

    # Surf Breaking (advanced)
    surf_breaking = consts.add_boolean('surf_breaking', 'Surf Breaking (advanced)', default=False)
    flags = {True: True, False: False}
    consts.add_float('BJALFA', 'Dissipation Constant [BJALFA]', default=1.0).add_dependency(surf_breaking, flags)
    consts.add_float('BJGAM', 'Breaking Threshold [BJGAM]', default=0.73).add_dependency(surf_breaking, flags)

    default = 'Use Hmax/d Ratio Only'
    options = ['Use Hmax/d Ratio Only', 'Use Hmax/d in Miche Formulation']
    consts.add_option('BJFLAG', 'Method [BJFLAG]', default, options).add_dependency(surf_breaking, flags)

    grids = gm.global_parameters.add_group('grids', 'Grids')

    # Only allow UNST for the moment
    grids.add_option(
        'UNST',
        '',
        'UNSPT parameters [UNST]',
        [
            'UNSPT parameters [UNST]',
            # SMC Grid Propagation [PSMC]
        ]
    )

    # Boundary Spectra
    options = ['Listed in ww3_grid.nml', 'Taken From Type 15 Elements']
    ugobcauto = grids.add_option('UGOBCAUTO', 'Source of OBC Points [UGOBCAUTO]', 'Listed in ww3_grid.nml', options)

    desc = 'Threshold ( < 0) depth for OBC points [UGOBCDEPTH]'
    flags = {'Listed in ww3_grid.nml': False, 'Taken From Type 15 Elements': True}
    grids.add_float('UGOBCDEPTH', desc, 0.0, high=0.0).add_dependency(ugobcauto, flags)

    # Numerical Method
    method = grids.add_option('method', 'Explicit or Implicit', 'Explicit', ['Explicit', 'Implicit'])

    # Iteration Criteria
    flags = {'Explicit': False, 'Implicit': True}
    desc = 'Terminate based on max. number of iterations [JGS_TERMINATE_MAXITER]'
    grids.add_boolean('JGS_TERMINATE_MAXITER', desc, default=True).add_dependency(method, flags)

    desc = 'Max. number of solver iterations [JGS_MAXITER]'
    grids.add_integer('JGS_MAXITER', desc, default=100, low=0).add_dependency(method, flags)

    desc = '% of grid points that do not need to converge [JGS_PMIN]'
    grids.add_float('JGS_PMIN', desc, default=1.0).add_dependency(method, flags)

    desc = 'Implicit solver threshold for JGS_TERMINATE_DIFFERENCE [JGS_DIFF_THR]'
    grids.add_float('JGS_DIFF_THR', desc, 1e-08).add_dependency(method, flags)

    desc = 'Implicit solver threshold for JGS_TERMINATE_NORM [JGS_NORM_THR]'
    grids.add_float('JGS_NORM_THR', desc, 0.0).add_dependency(method, flags)

    hidden = {'Explicit': False, 'Implicit': False}

    grids.add_integer('UNSTJGS_USE_JACOBI', 'UNSTJGS_USE_JACOBI', 1).add_dependency(method, hidden)
    grids.add_integer('UNSTJGS_BLOCK_GAUSS_SEIDEL', 'UNSTJGS_BLOCK_GAUSS_SEIDEL', 1).add_dependency(method, hidden)
    grids.add_integer('JGS_TERMINATE_DIFFERENCE', 'JGS_TERMINATE_DIFFERENCE', 1).add_dependency(method, hidden)
    grids.add_float('JGS_TERMINATE_NORM', 'JGS_TERMINATE_NORM', 0.0).add_dependency(method, hidden)
    grids.add_integer('UNSTSETUP_APPLY_WLV', 'UNSTSETUP_APPLY_WLV', 1).add_dependency(method, hidden)
    grids.add_float('UNSTSOLVERTHR_SETUP', 'UNSTSOLVERTHR_SETUP', 1.0E-6).add_dependency(method, hidden)
    grids.add_float('UNSTCRIT_DEP_SETUP', 'UNSTCRIT_DEP_SETUP', 0.1).add_dependency(method, hidden)
    grids.add_integer('unstructured_grids', 'unstructured_grids', 0).add_dependency(method, hidden)

    run_control = gm.global_parameters.add_group('run_control', 'Run control')

    # Executables
    run_control.add_integer('executables', 'Number of processors for running WW3 executables', default=1, low=1)

    options = ['F: no forcing', 'T: external forcing file', 'H: homogeneous forcing input', 'C: coupled forcing field']

    # Define water levels
    water_levels = run_control.add_option('water_levels', 'Define water levels', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    run_control.add_dataset('water_level_dataset', 'Dataset').add_dependency(water_levels, flags)
    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Height')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    item_id = 'water_level_homogeneous'
    run_control.add_table(item_id, 'Define Homogeneous', '', table_def).add_dependency(water_levels, flags)

    # Define currents
    currents = run_control.add_option('currents', 'Define currents', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    run_control.add_dataset('currents_dataset', 'Dataset').add_dependency(currents, flags)

    definition = [
        DateTimeColumnType(header='Time'),
        FloatColumnType(header='Speed'),
        FloatColumnType(header='Direction')
    ]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    run_control.add_table('currents_homogeneous', 'Define Homogeneous', '', table_def).add_dependency(currents, flags)

    # Define winds
    winds = run_control.add_option('winds', 'Define winds', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    run_control.add_dataset('winds_dataset', 'Dataset').add_dependency(winds, flags)

    definition = [
        DateTimeColumnType(header='Time'),
        FloatColumnType(header='Speed'),
        FloatColumnType(header='Direction'),
        FloatColumnType(header='AirSeaTemp')
    ]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    run_control.add_table('winds_homogeneous', 'Define Homogeneous', '', table_def).add_dependency(winds, flags)

    # Define air density
    air_density = run_control.add_option('air_density', 'Define air density', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    run_control.add_dataset('air_density_dataset', 'Dataset').add_dependency(air_density, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Thickness')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    item_id = 'air_density_homogeneous'
    run_control.add_table(item_id, 'Define Homogeneous', '', table_def).add_dependency(air_density, flags)

    run_control.add_date_time('starting_date', 'Starting Date', default=datetime.datetime(1950, 1, 1, 0, 0, 0))

    run_control.add_date_time('end_date', 'End Date', default=datetime.datetime(1950, 1, 1, 0, 0, 0))

    # Spectral Boundary Conditions
    run_control.add_option('interpolation_method', 'Interpolation Method', 'Linear', ['Linear', 'Nearest Point'])

    verbose = run_control.add_boolean('verbose_mode', 'Verbose mode', default=True)

    # atm_momentum is hidden in the model control
    hidden = {True: False, False: False}
    run_control.add_option('define_atm_momentum', '', 'F: no forcing', options).add_dependency(verbose, hidden)
    run_control.add_dataset('atm_momentum_dataset', 'Dataset').add_dependency(verbose, hidden)

    # bound_mode is hidden in the model control
    run_control.add_text('bound_mode', '', 'WRITE').add_dependency(verbose, hidden)

    # bound_interp is hidden in the model control
    run_control.add_integer('bound_interp', '', 2).add_dependency(verbose, hidden)

    # bound_verbose is hidden in the model control
    run_control.add_integer('bound_verbose', '', 1).add_dependency(verbose, hidden)

    # bound_file is hidden in the model control
    run_control.add_text('bound_file', '', 'spec.list').add_dependency(verbose, hidden)

    # num_proc is hidden in the model control
    run_control.add_integer('num_proc', '', 1).add_dependency(verbose, hidden)

    # Dataset UUID's
    run_control.add_text('forcing_water_levels_uuid', '', '').add_dependency(verbose, hidden)
    run_control.add_text('forcing_currents_uuid', '', '').add_dependency(verbose, hidden)
    run_control.add_text('forcing_winds_levels_uuid', '', '').add_dependency(verbose, hidden)
    run_control.add_text('forcing_atm_momentum_uuid', '', '').add_dependency(verbose, hidden)
    run_control.add_text('forcing_air_density_uuid', '', '').add_dependency(verbose, hidden)

    # Output
    output = gm.global_parameters.add_group('output', 'Output')

    # Output file
    output.add_text('file_name', 'Prefix for output file name', default='ww3.')

    default = datetime.datetime(1950, 1, 1, 0, 0, 0)
    flags = {True: True, False: False}

    point_output = output.add_boolean('point_output', 'Point output', default=False)
    output.add_date_time('output_start_date_point', 'Start Date', default).add_dependency(point_output, flags)
    output.add_integer('output_second_interval_point', 'Interval Seconds', 0).add_dependency(point_output, flags)
    output.add_date_time('output_end_date_point', 'End Date', default).add_dependency(point_output, flags)

    along_track = output.add_boolean('along_track', 'Output along track', default=False)
    output.add_boolean('input_file_along_track', 'Input File Along Track', False).add_dependency(along_track, flags)
    output.add_date_time('output_start_date_track', 'Start Date', default).add_dependency(along_track, flags)
    output.add_integer('output_second_interval_track', 'Interval Seconds', default=0).add_dependency(along_track, flags)
    output.add_date_time('output_end_date_track', 'End Date', default).add_dependency(along_track, flags)

    restart_files = output.add_boolean('restart_files', 'Restart files', default=False)
    output.add_date_time('output_start_date_restart', 'Start Date', default).add_dependency(restart_files, flags)
    output.add_integer('output_second_interval_restart', 'Interval Seconds', 0).add_dependency(restart_files, flags)
    output.add_date_time('output_end_date_restart', 'End Date', default).add_dependency(restart_files, flags)

    boundary_data = output.add_boolean('boundary_data', 'Boundary data', default=False)
    output.add_date_time('output_start_date_boundary', 'Start Date', default).add_dependency(boundary_data, flags)
    output.add_integer('output_second_interval_boundary', 'Interval Seconds', 0).add_dependency(boundary_data, flags)
    output.add_date_time('output_end_date_boundary', 'End Date', default).add_dependency(boundary_data, flags)

    wave_field = output.add_boolean('wave_field', 'Separated wave field data', default=False)
    output.add_date_time('output_start_date_separated', 'Start Date', default).add_dependency(wave_field, flags)
    output.add_integer('output_second_interval_separated', 'Interval Seconds', 0).add_dependency(wave_field, flags)
    output.add_date_time('output_end_date_separated', 'End Date', default).add_dependency(wave_field, flags)

    coupling = output.add_boolean('coupling', 'Coupling', default=False)
    output.add_date_time('output_start_date_coupling', 'Start Date', default).add_dependency(coupling, flags)
    output.add_integer('output_second_interval_coupling', 'Interval Seconds', default=0).add_dependency(coupling, flags)
    output.add_date_time('output_end_date_coupling', 'End Date', default).add_dependency(coupling, flags)

    # IOSType is always hidden in the model control
    hidden = {True: False, False: False}
    output.add_integer('IOSType', 'IOSType', default=0).add_dependency(coupling, hidden)
    output.add_integer('OType1', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType2', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType3', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType4', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType5', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType6', '', default=0).add_dependency(coupling, hidden)  # hidden
    output.add_integer('OType7', '', default=0).add_dependency(coupling, hidden)  # hidden
    # Ice and mud (advanced)
    ice_and_mud = gm.global_parameters.add_group('ice_and_mud', 'Ice and mud (advanced)')

    options = ['F: no forcing', 'T: external forcing file', 'C: coupled forcing field']
    concentration = ice_and_mud.add_option('concentration', 'Define ice concentration', 'F: no forcing', options)
    flags = {'F: no forcing': False, 'T: external forcing file': True, 'C: coupled forcing field': False}
    ice_and_mud.add_dataset('ice_concentration', 'Dataset').add_dependency(concentration, flags)

    # Ice parameter 1
    options = ['F: no forcing', 'T: external forcing file', 'H: homogeneous forcing input', 'C: coupled forcing field']
    param_1 = ice_and_mud.add_option('param_1', 'Define ice parameter 1', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('ice_param_1_dataset', 'Dataset').add_dependency(param_1, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Thickness')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('ice_param_1', 'Define Homogeneous', '', table_def).add_dependency(param_1, flags)

    # Ice parameter 2
    param_2 = ice_and_mud.add_option('param_2', 'Define ice parameter 2', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('ice_param_2_dataset', 'Dataset').add_dependency(param_2, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Viscosity')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('ice_param_2', 'Define Homogeneous', '', table_def).add_dependency(param_2, flags)

    # Ice parameter 3
    param_3 = ice_and_mud.add_option('param_3', 'Define ice parameter 3', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('ice_param_3_dataset', 'Dataset').add_dependency(param_3, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Density')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('ice_param_3', 'Define Homogeneous', '', table_def).add_dependency(param_3, flags)

    # Ice parameter 4
    param_4 = ice_and_mud.add_option('param_4', 'Define ice parameter 4', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('ice_param_4_dataset', 'Dataset').add_dependency(param_4, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Modulus')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('ice_param_4', 'Define Homogeneous', '', table_def).add_dependency(param_4, flags)

    # Ice parameter 5
    param_5 = ice_and_mud.add_option('param_5', 'Define ice parameter 5', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('ice_param_5_dataset', 'Dataset').add_dependency(param_5, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Floe Diameter')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('ice_param_5', 'Define Homogeneous', '', table_def).add_dependency(param_5, flags)

    # Mud Density
    mud_density = ice_and_mud.add_option('mud_density', 'Define mud density', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('mud_density_dataset', 'Dataset').add_dependency(mud_density, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Density')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_table('mud_density_table', 'Define Homogeneous', '', table_def).add_dependency(mud_density, flags)

    # Mud thickness
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    mud_thickness = ice_and_mud.add_option('mud_thickness', 'Define mud thickness', 'F: no forcing', options)
    ice_and_mud.add_dataset('mud_thickness_dataset', 'Dataset').add_dependency(mud_thickness, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Thickness')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    item_id = 'mud_thickness_table'
    ice_and_mud.add_table(item_id, 'Define Homogeneous', '', table_def).add_dependency(mud_thickness, flags)

    # Mud viscosity
    mud_viscosity = ice_and_mud.add_option('mud_viscosity', 'Define mud viscosity', 'F: no forcing', options)
    flags = {
        'F: no forcing': False,
        'T: external forcing file': True,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }
    ice_and_mud.add_dataset('mud_viscosity_dataset', 'Dataset').add_dependency(mud_viscosity, flags)

    definition = [DateTimeColumnType(header='Time'), FloatColumnType(header='Viscosity')]
    table_def = TableDefinition(definition)

    flags = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': True,
        'C: coupled forcing field': False
    }
    item_id = 'mud_viscosity_table'
    ice_and_mud.add_table(item_id, 'Define Homogeneous', '', table_def).add_dependency(mud_viscosity, flags)

    hidden = {
        'F: no forcing': False,
        'T: external forcing file': False,
        'H: homogeneous forcing input': False,
        'C: coupled forcing field': False
    }

    # assimilation_data_mean_para is hidden in the model control
    desc = 'assimilation_data_mean_para'
    ice_and_mud.add_option(desc, '', 'F: no forcing', options).add_dependency(mud_viscosity, hidden)

    # assimilation_data_1D_spectra is hidden in the model control'
    desc = 'assimilation_data_1D_spectra'
    ice_and_mud.add_option(desc, '', 'F: no forcing', options).add_dependency(mud_viscosity, hidden)

    # assimilation_data_2D_spectra is hidden in the model control
    desc = 'assimilation_data_2D_spectra'
    ice_and_mud.add_option(desc, '', 'F: no forcing', options).add_dependency(mud_viscosity, hidden)

    # Sea state stress (advanced)
    sea_state_stress = gm.global_parameters.add_group('sea_state_stress', 'Sea state stress (advanced)')

    desc = 'Diagnostic sea-state dependent stress method'
    options = ['Reichl et al. 2014', 'Donelan et al. 2012']
    sea_state_stress.add_option('sea_state_method', desc, 'Reichl et al. 2014', options)

    desc = 'Diagnostic sea-state dependent stress type'
    options = ['Constant value', 'Wind speed dependent']
    stress_type = sea_state_stress.add_option('stress_type', desc, 'Constant value', options)
    flags = {'Constant value': True, 'Wind speed dependent': False}
    sea_state_stress.add_float('TAILLEV', 'TAILLEV', 0.01, high=0.02, low=0.001).add_dependency(stress_type, flags)
    sea_state_stress.add_float('TAILT1', 'TAILT1', default=1.25)
    sea_state_stress.add_float('TAILT2', 'TAILT2', default=3.0)

    # These are hidden in the model control
    hidden = {'Constant value': False, 'Wind speed dependent': False}
    sea_state_stress.add_integer('FLDMethod', 'FLDMethod', 0).add_dependency(stress_type, hidden)
    sea_state_stress.add_integer('TAILTYPE', '', 0).add_dependency(stress_type, hidden)

    # Miscellaneous (advanced)
    miscellaneous = gm.global_parameters.add_group('miscellaneous', 'Miscellaneous (advanced)')

    miscellaneous.add_float('PMOVE', 'Power p for Moving Grids [PMOVE]', default=0.0)
    miscellaneous.add_float('XSEED', 'Xseed [XSEED]', default=0.0)
    options = [
        'No subgrid', 'Transparent at cell boundaries', 'Transparent at cell centers',
        'Tranparent at cell boundaries with continuous ice', 'Transparent at cell centers with continuous ice'
    ]
    flagtr = miscellaneous.add_option('FLAGTR', 'Subgrid Type [FLAGTR]', default='No subgrid', options=options)
    miscellaneous.add_float('XP', 'Xp [XP]', default=0.0)
    miscellaneous.add_float('XR', 'Xr [XR]', default=0.0)
    miscellaneous.add_float('XFILT', 'Xf [XFILT]', default=0.0)
    miscellaneous.add_integer('XIHMAX', 'Number of Discrete Levels in Part [XIHMAX]', default=0)
    miscellaneous.add_float('HSPMIN', 'Minimum Hs in Partitioning [HSPMIN]', default=0.0)
    miscellaneous.add_float('WSM', 'Wind Speed Multiplier in Part [WSM]', default=0.0)
    miscellaneous.add_float('WSC', 'Cut of Wind Sea Fraction for Identifying Wind Sea in Part [WSC]', default=0.0)
    miscellaneous.add_float('FLC', 'Combining Wind Seas in Partitioning Flag [FLC]', default=0.0)
    miscellaneous.add_integer('NOSW', 'Number of Partitioned Swell Fields in Field Output [NOSW]', default=0)
    miscellaneous.add_float('FMICHE', 'Constant in Miche Limiter [FMICHE]', default=1.6)
    miscellaneous.add_float('STDX', 'Space-Time Extremes X-Length [STDX]', default=0.0)
    miscellaneous.add_float('STDY', 'Space-Time Extremes Y-Length [STDY]', default=0.0)
    miscellaneous.add_float('STDT', 'Space-Time Extremes Duration [STDT]', default=0.0)
    miscellaneous.add_integer('P2SF', 'Second-Order Pressure at K=0" [P2SF]', default=0)
    miscellaneous.add_integer('I1P2SF', 'First Frequency Index [I1P2SF]', default=1, low=0)
    miscellaneous.add_integer('I2P2SF', 'Last Frequency Index [I2P2SF]', default=1, low=0)

    # These options are hidden in the model control
    hidden = {
        'No subgrid': False,
        'Transparent at cell boundaries': False,
        'Transparent at cell centers': False,
        'Tranparent at cell boundaries with continuous ice': False,
        'Transparent at cell centers with continuous ice': False
    }
    miscellaneous.add_float('CICE0', '', 0.0).add_dependency(flagtr, hidden)
    miscellaneous.add_float('CICEN', '', 0.0).add_dependency(flagtr, hidden)

    # We can remove this once we update the executable calls to use the new data manager
    miscellaneous.add_text('domain_uuid', '', default='').add_dependency(flagtr, hidden)

    return gm
