"""Class to read a WaveWatch3 prnc (NetCDF input field preprocessor) namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import logging
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.wavewatch3.data.model import get_model
from xms.wavewatch3.file_io.io_util import READ_BUFFER_SIZE
from xms.wavewatch3.file_io.netcdf_dataset_reader import NetcdfDatasetReader

RUN_CONTROL_DATASETS = [
    'water_level_dataset', 'currents_dataset', 'winds_dataset', 'air_density_dataset', 'atm_momentum_dataset'
]
ICE_AND_MUD_DATASETS = [
    'ice_concentration', 'ice_param_1_dataset', 'ice_param_2_dataset', 'ice_param_3_dataset', 'ice_param_4_dataset',
    'ice_param_5_dataset', 'mud_density_dataset', 'mud_thickness_dataset', 'mud_viscosity_dataset'
]


def _get_true_false_as_int(value):
    """Gets the various formats of true/false in as an integer.

    Args:
        value (:obj:`str`):  The string from the file.

    Returns:
        (:obj:`int`):  1 if some form of true, 0 if false
    """
    if value.upper() == '.FALSE.' or value.upper() == 'F' or value.upper() == "'F'" or value == '0':
        return 0
    elif value.upper() == '.TRUE.' or value.upper() == 'T' or value.upper() == "'T'" or value == '1':
        return 1
    else:
        raise ValueError(f"Unable to determine value of {value}.")


def _get_longest_common_substring(s1: str, s2: str) -> str:
    """Finds the longest common substring between two strings."""
    longest_substring = ""
    # Iterate through all possible substrings of the first string
    for i in range(len(s1)):
        for j in range(i + 1, len(s1) + 1):
            substring = s1[i:j]
            # Check if the substring exists in the second string
            # and is longer than the one we already found
            if substring in s2 and len(substring) > len(longest_substring):
                longest_substring = substring
    return longest_substring


class PrncNmlReader:
    """Class to read a WaveWatch3 prnc nml file."""
    def __init__(self, filename='', sim_data=None):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the nml file. If not provided (not testing or control file read),
                will retrieve from Query.
            sim_data(:obj:`xms.wavewatch3.data.SimData`):  The simulation data to edit.
        """
        self._filename = filename
        self._query = None
        self._sim_data = sim_data
        self._sim_comp_uuid = str(uuid.uuid4())
        self._setup_query()
        self._lines = []
        self._current_line = 0
        self._logger = logging.getLogger('xms.wavewatch3')
        self._forcing_field = ''
        self._forcing_file = ''
        self._longitude_field = ''
        self._latitude_field = ''
        self._forcing_var_1 = ''
        self._forcing_var_2 = ''
        self._grid_asis = False
        self._grid_latlon = False
        self._global_values = get_model().global_parameters
        self._global_values.restore_values(self._sim_data.global_values)

    def _setup_query(self):
        """Setup the xmsapi Query for sending data to SMS and get the import filename."""
        if not self._filename:  # pragma: no cover - slow to setup Query for the filename
            self._query = Query()
            self._filename = self._query.read_file

    def _parse_next_line(self, shell=False):
        """Parse the next line of text from the file.

        Skips empty and comment lines.

        Args:
            shell (:obj:`bool`): If True will parse line using shlex. Slower but convenient for quoted tokens.

        Returns:
            (:obj:`list[str]`: The next line of text, split on whitespace
        """
        line = None
        while not line or line.startswith('!'):  # blank lines and control file identifier
            if self._current_line >= len(self._lines):
                # raise RuntimeError('Unexpected end of file.')
                return None
            line = self._lines[self._current_line].strip()
            self._current_line += 1
        return line.split()

    def _get_namelist_cards_and_values(self, firstline_data):
        """Reads the entire namelist until the closing / is found.  Stores cards and values.

        Handles cases where the data is on multiple lines, or on a single line.

        Args:
            firstline_data (:obj:`list[str]`):  List of data parsed on the opening line.

        Returns:
            (:obj:`dict`):  Dictionary with the keys being the cards read, and values of corresponding data.
        """
        namelist_list = firstline_data
        if '/' not in namelist_list:
            # We haven't found the end of the namelist yet.  Read more lines as necessary.
            end_not_found = True
            while end_not_found:
                # Grab the next line of data
                data = self._parse_next_line()
                # Extend the list of all data by the current line read
                namelist_list.extend(data)
                # Check if we've got the end of the namelist yet
                if '/' in data:
                    end_not_found = False

        # Now, we have the entire namelist as a list of values.  Parse into cards/commands and values
        cards = []
        values = []
        for i in range(len(namelist_list)):
            # Get the list elements on either side of each =
            if namelist_list[i] == '=':
                if 0 < i < len(namelist_list):
                    cards.append(namelist_list[i - 1].rstrip(','))
                    str = namelist_list[i + 1].rstrip(',')
                    if str[0] == "'" and str[-1] != "'":
                        # We have a string to read... get it until we find a closing single quote
                        # This could be something like a date:  '20150227 000000'
                        closing_found = False
                        while not closing_found:
                            i += 1
                            str += " " + namelist_list[i + 1].rstrip(',')
                            if str[-1] == "'":
                                closing_found = True
                    values.append(str)

        # Return the cards and values found throughout the namelist read
        return cards, values

    def _read_prnc_nml_file(self):
        """Read the PRNC nml file."""
        reading = True
        while reading:
            data = self._parse_next_line()
            if data:
                if '&FORCING_NML' in data[0].strip():
                    self._read_forcing_nml_namelist(data)
                elif '&FILE_NML' in data[0].strip():
                    self._read_file_nml_namelist(data)
                else:
                    raise ValueError(f'Unrecognized namelist {data}')
            else:
                reading = False

    def _read_forcing_nml_namelist(self, data):
        """Read the FORCING_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading FORCING_NML namelist...')
        model_parameters = self._global_values
        run_control = model_parameters.group('run_control')
        ice_and_mud = model_parameters.group('ice_and_mud')

        cards, values = self._get_namelist_cards_and_values(data)
        if 'FORCING%FIELD%WATER_LEVELS' in cards:
            if values[cards.index('FORCING%FIELD%WATER_LEVELS')] == 'T':
                run_control.parameter('water_levels').value = 'T: external forcing file'
                self._forcing_field = 'water_level_dataset'
            else:
                run_control.parameter('water_levels').value = 'F: no forcing'
        if 'FORCING%FIELD%CURRENTS' in cards:
            if values[cards.index('FORCING%FIELD%CURRENTS')] == 'T':
                run_control.parameter('currents').value = 'T: external forcing file'
                self._forcing_field = 'currents_dataset'
            else:
                run_control.parameter('currents').value = 'F: no forcing'
        if 'FORCING%FIELD%WINDS' in cards:
            if values[cards.index('FORCING%FIELD%WINDS')] == 'T':
                run_control.parameter('winds').value = 'T: external forcing file'
                self._forcing_field = 'winds_dataset'
            else:
                run_control.parameter('winds').value = 'F: no forcing'
        if 'FORCING%FIELD%ATM_MOMENTUM' in cards:
            if values[cards.index('FORCING%FIELD%ATM_MOMENTUM')] == 'T':
                run_control.parameter('define_atm_momentum').value = 'T: external forcing file'
                self._forcing_field = 'atm_momentum_dataset'
            else:
                run_control.parameter('define_atm_momentum').value = 'F: no forcing'
        if 'FORCING%FIELD%AIR_DENSITY' in cards:
            if values[cards.index('FORCING%FIELD%AIR_DENSITY')] == 'T':
                run_control.parameter('air_density').value = 'T: external forcing file'
                self._forcing_field = 'air_density_dataset'
            else:
                run_control.parameter('air_density').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_CONC' in cards:
            if values[cards.index('FORCING%FIELD%ICE_CONC')] == 'T':
                ice_and_mud.parameter('concentration').value = 'T: external forcing file'
                self._forcing_field = 'ice_concentration'
            else:
                ice_and_mud.parameter('concentration').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_PARAM1' in cards:
            if values[cards.index('FORCING%FIELD%ICE_PARAM1')] == 'T':
                ice_and_mud.parameter('param_1').value = 'T: external forcing file'
                self._forcing_field = 'ice_param_1_dataset'
            else:
                ice_and_mud.parameter('param_1').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_PARAM2' in cards:
            if values[cards.index('FORCING%FIELD%ICE_PARAM2')] == 'T':
                ice_and_mud.parameter('param_2').value = 'T: external forcing file'
                self._forcing_field = 'ice_param_2_dataset'
            else:
                ice_and_mud.parameter('param_2').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_PARAM3' in cards:
            if values[cards.index('FORCING%FIELD%ICE_PARAM3')] == 'T':
                ice_and_mud.parameter('param_3').value = 'T: external forcing file'
                self._forcing_field = 'ice_param_3_dataset'
            else:
                ice_and_mud.parameter('param_3').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_PARAM4' in cards:
            if values[cards.index('FORCING%FIELD%ICE_PARAM4')] == 'T':
                ice_and_mud.parameter('param_4').value = 'T: external forcing file'
                self._forcing_field = 'ice_param_4_dataset'
            else:
                ice_and_mud.parameter('param_4').value = 'F: no forcing'
        if 'FORCING%FIELD%ICE_PARAM5' in cards:
            if values[cards.index('FORCING%FIELD%ICE_PARAM5')] == 'T':
                ice_and_mud.parameter('param_5').value = 'T: external forcing file'
                self._forcing_field = 'ice_param_5_dataset'
            else:
                ice_and_mud.parameter('param_5').value = 'F: no forcing'
        if 'FORCING%FIELD%MUD_DENSITY' in cards:
            if values[cards.index('FORCING%FIELD%MUD_DENSITY')] == 'T':
                ice_and_mud.parameter('mud_density').value = 'T: external forcing file'
                self._forcing_field = 'mud_density_dataset'
            else:
                ice_and_mud.parameter('mud_density').value = 'F: no forcing'
        if 'FORCING%FIELD%MUD_THICKNESS' in cards:
            if values[cards.index('FORCING%FIELD%MUD_THICKNESS')] == 'T':
                ice_and_mud.parameter('mud_thickness').value = 'T: external forcing file'
                self._forcing_field = 'mud_thickness_dataset'
            else:
                ice_and_mud.parameter('mud_thickness').value = 'F: no forcing'
        if 'FORCING%FIELD%MUD_VISCOSITY' in cards:
            if values[cards.index('FORCING%FIELD%MUD_VISCOSITY')] == 'T':
                ice_and_mud.parameter('mud_viscosity').value = 'T: external forcing file'
                self._forcing_field = 'mud_viscosity_dataset'
            else:
                ice_and_mud.parameter('mud_viscosity').value = 'F: no forcing'
        if 'FORCING%GRID%ASIS' in cards:
            self._grid_asis = _get_true_false_as_int(values[cards.index('FORCING%GRID%ASIS')])
        if 'FORCING%GRID%LATLON' in cards:
            self._grid_latlon = _get_true_false_as_int(values[cards.index('FORCING%GRID%LATLON')])

    def _read_file_nml_namelist(self, data):
        """Read the FILE_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading FILE_NML namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'FILE%FILENAME' in cards:
            self._forcing_file = values[cards.index('FILE%FILENAME')].lstrip("'").rstrip("'")
        if 'FILE%LONGITUDE' in cards:
            self._longitude_field = values[cards.index('FILE%LONGITUDE')].lstrip("'").rstrip("'")
        if 'FILE%LATITUDE' in cards:
            self._latitude_field = values[cards.index('FILE%LATITUDE')].lstrip("'").rstrip("'")
        if 'FILE%VAR(1)' in cards:
            self._forcing_var_1 = values[cards.index('FILE%VAR(1)')].lstrip("'").rstrip("'")
        if 'FILE%VAR(2)' in cards:
            self._forcing_var_2 = values[cards.index('FILE%VAR(2)')].lstrip("'").rstrip("'")

    def read_netcdf_dataset(self, query, mesh_uuid):
        """Read the netCDF dataset file for the forcing field.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS.
            temp_dir (:obj:`str`): The temp directory for creating .h5 dataset files in.
            mesh_uuid (:obj:`str`): The uuid of the associated mesh geometry.
        """
        model_parameters = self._global_values
        run_control = model_parameters.group('run_control')
        ice_and_mud = model_parameters.group('ice_and_mud')
        if self._forcing_file:
            self._logger.debug(f'Using mesh uuid of {mesh_uuid}')
            self._query = query
            reftime = datetime.datetime(1990, 1, 1)
            reader = NetcdfDatasetReader(
                self._forcing_file, reftime, self._query, self._query.xms_temp_directory, [], mesh_uuid
            )

            if self._forcing_var_2:
                common_substring = _get_longest_common_substring(self._forcing_var_1, self._forcing_var_2)

                # Use the common substring if it's found, otherwise fall back to the second var name
                dataset_name = common_substring if common_substring else self._forcing_var_2

                dataset_uuid = reader.read_netcdf_vectors(
                    self._forcing_file, f'/{self._forcing_var_1}', f'/{self._forcing_var_2}', dataset_name
                )
                if dataset_uuid and self._forcing_field in RUN_CONTROL_DATASETS:
                    run_control.parameter(self._forcing_field).value = dataset_uuid
            else:
                dataset_uuid = reader.read_netcdf_scalars(
                    self._forcing_file, f'/{self._forcing_var_1}', self._forcing_var_1
                )
                if dataset_uuid and self._forcing_field in RUN_CONTROL_DATASETS:
                    run_control.parameter(self._forcing_field).value = dataset_uuid
                elif dataset_uuid and self._forcing_field in ICE_AND_MUD_DATASETS:
                    ice_and_mud.parameter(self._forcing_field).value = dataset_uuid

            self._sim_data.global_values = self._global_values.extract_values()
            self._sim_data.commit()

    def read(self):
        """Top-level entry point for the WaveWatch3 bounc nml input file reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()

            self._read_prnc_nml_file()
            self._logger.info('Committing changes....')
            self._sim_data.global_values = self._global_values.extract_values()
            self._sim_data.commit()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception(
                'Unexpected error in prnc nml preprocessor file '
                f'(line {self._current_line + 1}).'
            )
            raise
