"""This module defines the Profile settings."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.profiles.administrative_settings import AdministrationSettings
from xms.FhwaVariable.core_data.app_data.profiles.gradation import GradationSettings
from xms.FhwaVariable.core_data.app_data.profiles.mannings_n_tables import ManningsNTables
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.personal_details import PersonalDetails
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


class ProfileSettings(SettingGroup, CalcData):
    """A class that defines the program settings."""

    def __init__(self, app_name, version, agency, developed_by, app_data=None):
        """Initializes the Profile Interactor.

        Args:
            app_name (str): The name of the program
            version (str): The version of the program
            agency (str): The agency that developed the program
            developed_by (str): The developer of the program
            app_data (AppData): The application data
        """
        SettingGroup.__init__(self, app_name, version, agency, developed_by, app_data=app_data)
        CalcData.__init__(self, app_data=app_data, model_name=app_name,)

        self.name = 'Profile calc'
        self.type = 'ProfileCalc'
        self.uuid = uuid.uuid4()

        self.version = 1.0
        self.modification_date = datetime.today().date()

        self.calc_support_dual_input = True  # Displays a second column of input (HY-8 Style)
        self.calc_support_warnings = False  # Displays the warnings text box
        self.calc_support_results = False  # Displays the Results table box
        self.calc_support_plot = False  # Displays the plot

        self.dlg_width = 800
        self.dlg_height = 600

        # input
        self.input = {}
        # self.input_dual = {}

        self.unknown = None

        # Results
        self.clear_my_own_results = True
        self.results = {}

        self.warnings = []

        # plot

        # Help/Wiki
        # self.self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Channel_Calculator'

        # Settings Data
        self.input['Profile name'] = Variable('Profile name', 'string', 'New profile')
        self.input['Version'] = Variable('Profile version', 'float', self.version, precision=2, limits=(
            0.0, self.max_value))
        self.input['Modification date'] = Variable('Date of last modification', 'date', self.modification_date)
        self.input['Modification date'].read_only = True
        self.input['Organization details'] = Variable('Organization details', 'calc', PersonalDetails(
            app_name, version, agency, developed_by))
        # self.input['Organization details'].name = 'Organization details'
        # self.input['Organization details'].type = 'PersonalDetails'

        self.input['Administrative settings'] = Variable(
            'Administrative settings', 'class', AdministrationSettings(app_name, version, agency, developed_by))

        self.input['Gradation settings'] = Variable('Gradation settings', 'class',
                                                    GradationSettings(app_name, version, agency, developed_by))
        # Consider putting this in the GUI repo
        self.input['Mannings n tables'] = Variable("Manning's n tables", 'class',
                                                   ManningsNTables(app_name, version, agency, developed_by))

    def set_name(self, name):
        """Set the name of the profile."""
        self.name = name
        self.input['Profile name'].value = name

    def set_version(self, version):
        """Set the version of the profile."""
        self.version = version
        self.input['Version'].value = version

    def set_modification_date(self, date=None):
        """Set the modification date of the profile."""
        if date is None:
            date = datetime.today().date()
        else:
            if isinstance(date, str):
                # Convert string to date object
                date = datetime.strptime(date, '%Y-%m-%d').date()

        self.modification_date = date
        self.input['Modification date'].value = date
