"""PreferenceDefinitions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import os
from pathlib import Path
import platform

# 2. Third party modules
try:
    import winreg
except ImportError:
    winreg = None


# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


# @singleton
class PreferenceDefinitions(SettingGroup):
    """Provides definitions for the user preferences."""

    def __init__(self, name: str = 'HydraulicToolbox', version: str = '0.1', agency: str = 'FHWA',
                 developed_by: str = 'Aquaveo', doc_folder: Path = None):
        """Initializes the preference values."""
        super().__init__(name, version, agency, developed_by)

        self.name = 'PreferenceDefinitions'
        self.type = 'PreferenceDefinitions'

        complexity_options, default_complexity = self.check_file_keys()

        # "U.S. Customary Units", "SI Units (Metric)", or "Both"
        self.input['Selected unit system'] = Variable(
            'Selected unit system', 'list', 0, ['U.S. Customary Units', 'SI Units (Metric)', 'Both'])

        self.input['Show wiki panel'] = Variable('Show wiki panel', 'bool', True, [],
                                                 note='Show the wiki panel in dockable widget in the main application.')
        # self.input['Use only dockable wiki'] = Variable(
        #     'Use only dockable wiki panel to display help information.', 'bool', False, [],
        #     note='Dialogs will not have a help wiki panel and when help buttons are clicked in them, results will '
        #     'show in the dockable wiki panel.')

        self.input['Complexity'] = Variable('Complexity of interface', 'list', default_complexity, complexity_options,
                                            note='This changes the number of options available. Most only need simple')
        self.input['Dialog columns'] = Variable('Number of columns in a dialog', 'list', 1,
                                                ['one', 'two', 'three', 'four'],
                                                note='This changes the way data is shown on the screen.')
        self.input['Dual column input'] = Variable(
            'Allow dual input tables', 'bool', True, [],
            note='This allows some dialogs to display two input panels (HY-8 style).')

        self.input['Create plots with equal aspect'] = Variable(
            "Create plots with equal aspect (X & Y scale the same)", 'bool', False, [])

        self.input['Recent files list length'] = Variable(
            'Number of recent files preserved in file menus', 'int', 20, [], complexity=1)

        self.input['Undo list length'] = Variable(
            'Number of undo/redo operations kept in memory', 'int', 100, [], complexity=1)

        self.input['Undo list show length'] = Variable(
            'Number of undo/redo operations shown in menu history', 'int', 10, [], complexity=1)

        self.input['Use significant figures'] = Variable(
            'Use automatic significant figures', 'bool', True, [], complexity=0,
            note='The number of significant figures will be decided by the program.')

        self.input['Additional significant figures'] = Variable('Additional significant figures', 'int', 0, [],
                                                                complexity=2)
        self.input['Copy headers of table to clipboard'] = Variable('Copy headers of table to clipboard', 'bool',
                                                                    True, [], complexity=2)
        self.input['Settings location'] = Variable(
            "Settings and profile file location", "file", Path(''), '',
            note='This folder needs to be located where you have read/write access for internal file storage.',
            complexity=3)
        self.input['Settings location'].file_mode = 'folder'
        self.input['Document folder'] = Variable("Document folder location", "file", Path(''), '',
                                                 note='This specified folder has the reference pdf documents.',
                                                 complexity=2)
        self.input['Document folder'].file_mode = 'folder'
        self.input['Adobe Reader'] = Variable(
            "Adobe Acrobat PDF Reader", "file", Path(''), '',
            note='Specify the executable file of Adobe Acrobat reader for pdf reference documents.', complexity=2)
        self.input['Adobe Reader'].file_mode = 'existing file'
        self.initialize_file_locations(doc_folder)

        # self.input['scientific notation threshold'] = Variable("scientific notation threshold", 'int', 10000, [],
        #                                                         complexity=2)

    def check_file_keys(self):
        """Check if specific files exist and unlock specific settings."""
        complexity_options = ['simple', 'moderate', 'advanced']
        default_complexity = 0
        fhwa_level = False
        aquaveo_level = False
        fhwa_file_path = 'c:/temp/Hydraulic_Toolbox__turn_on_FHWA_complexity_level.txt'
        aquaveo_file_path = 'c:/temp/Hydraulic_Toolbox__turn_on_Aquaveo_complexity_level.txt'
        if 'Complexity' in self.input:
            default_complexity = self.input['Complexity'].value
        if os.path.exists(fhwa_file_path):
            fhwa_level = True
        if os.path.exists(aquaveo_file_path):
            fhwa_level = True
            aquaveo_level = True
        if fhwa_level:
            complexity_options.append('FHWA')
            default_complexity = 3
        if aquaveo_level:
            complexity_options.append('Aquaveo')
            default_complexity = 4
        if 'Complexity' in self.input:
            self.input['Complexity'].value_options = complexity_options
            self.input['Complexity'].value = default_complexity
        return complexity_options, default_complexity

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        input_vars = copy.deepcopy(self.input)

        if not input_vars['Use significant figures'].get_val():
            input_vars.pop('Additional significant figures')

        return input_vars

    def initialize_file_locations(self, doc_folder: Path = None):
        """This function will check the registry for a saved file locations and executable locations."""
        # Settings folder
        settings_folder = self.retrieve_persistent_app_data('settings_folder')
        save_to_persistant = False
        system = platform.system()
        if not settings_folder:
            save_to_persistant = True
            app_data_folder = None
            if system == 'Windows':
                if os.environ.get('APPDATA'):
                    app_data_folder = Path(os.environ.get('APPDATA'))
                else:
                    # Needed for CI/CD testing - may need something better...
                    app_data_folder = Path(__file__).parents[2] / 'test' / 'files'
            else:
                app_data_folder = Path('~').expanduser() / ".aquaveo"
            settings_folder = app_data_folder / self._get_app_folder_pattern()
            settings_folder.mkdir(parents=True, exist_ok=True)
        self.input['Settings location'].set_val(settings_folder)
        if save_to_persistant:
            self.save_persistant_app_data('settings_folder', str(settings_folder))

        # The Docs directory is three levels up from the current file
        if not doc_folder:
            doc_folder = Path(__file__).parents[2] / 'Docs'
        self.input['Document folder'].set_val(doc_folder)

        # Adobe Reader
        location = None
        if system == 'Windows':
            location = self._search_for_adobe_on_windows()
        elif system == 'Darwin':
            location = self._search_for_adobe_on_mac()

        if location:
            self.input['Adobe Reader'].set_val(location)

    def _search_for_adobe_on_mac(self):
        """Check the applications folder for something called Adobe Acrobat Reader."""
        location = None
        try:
            location = Path('/Applications/Adobe Acrobat Reader DC.app')
            if not location.exists():
                location = None
        except FileNotFoundError:
            pass
        return location

    def _search_for_adobe_on_windows(self):
        """Check various locations in the registry for the Adobe Acrobat Reader executable."""
        found = False
        try:
            # Open the key for Adobe Acrobat Reader
            key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE,
                                 r"SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\Acrobat.exe")
            # Read the value of the default key (usually contains the installation path)
            path = winreg.QueryValue(key, "")
            found = True
        except FileNotFoundError:
            pass
        if not found:
            try:
                # Open the key for Adobe Acrobat Reader
                key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE,
                                     r"SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\AcroRd32.exe")
                # Read the value of the default key (usually contains the installation path)
                path = winreg.QueryValue(key, "")
                found = True
            except FileNotFoundError:
                pass
        if not found:
            try:
                key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\Classes\acrobat\shell\open\command")
                path = winreg.QueryValue(key, "")
                found = True
            except FileNotFoundError:
                pass
        if found:
            return path
