"""PlotOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
# import random

# 2. Third party modules
import matplotlib.hatch
from matplotlib.path import Path
import numpy as np

# 3. Aquaveo modules

# 4. Local modules


def create_circle_path_c_r(center=(0, 0), radius=0.05, num_vertices=24):
    """
    Create a path of a small circle.

    Args:
        center (tuple): The (x, y) coordinates of the circle's center.
        radius (float): The radius of the circle.
        num_vertices (int): The number of vertices to approximate the circle.

    Returns:
        Path: A matplotlib.path.Path object representing the circle.
    """
    # Calculate the angle for each vertex
    angles = np.linspace(0, 2 * np.pi, num_vertices, endpoint=False)

    # Calculate the vertices of the circle
    vertices = [(center[0] + radius * np.cos(angle), center[1] + radius * np.sin(angle)) for angle in angles]

    # Add the first vertex at the end to close the circle
    vertices.append(vertices[0])

    # Define the path codes
    codes = [Path.MOVETO] + [Path.LINETO] * (num_vertices - 1) + [Path.CLOSEPOLY]

    # Create the path
    circle_path = Path(vertices, codes)

    return circle_path


def create_circle_path(x, y, radius, num_segments=24):
    """Create a path of a circle.

    Args:
        x (float): The x-coordinate of the circle's center.
        y (float): The y-coordinate of the circle's center.
        radius (float): The radius of the circle.
        num_segments (int): The number of segments to approximate the circle.

    Returns:
        tuple: A tuple containing the vertices and codes of the circle path.
    """
    vertices = []
    codes = []
    for i in range(num_segments):
        angle = 2 * np.pi * i / num_segments
        vertices.append((x + radius * np.cos(angle), y + radius * np.sin(angle)))
        if i == 0:
            codes.append(Path.MOVETO)
        else:
            codes.append(Path.LINETO)
    vertices.append(vertices[0])  # Close the circle
    codes.append(Path.CLOSEPOLY)
    return vertices, codes


def create_sand_particles():
    """
    Create a path of multiple small circles at random locations.

    Returns:
        Path: A matplotlib.path.Path object representing the combined circles.
    """
    radius = 0.004
    all_vertices = []
    all_codes = []
    all_centers = [
        (0.451012283103414, -0.165760270186294),
        (0.322120911383545, 0.137612289946655),
        (-0.197316367746893, -0.0340549331662775),
        (-0.126239507642999, 0.14146136223717),
        (0.0543033531247652, -0.271156198231818),
        (0.0719632065192686, 0.45029974364007763),
        (-0.42617992092773016, 0.36513754121488895),
        (-0.42303384760923035, -0.398573298592558)
    ]
    for center in all_centers:
        # Create the circle path
        circle_path = create_circle_path_c_r(center=center, radius=radius)

        # Append the vertices and codes
        all_vertices.extend(circle_path.vertices)
        all_codes.extend(circle_path.codes)

    # Create the combined path
    combined_path = Path(all_vertices, all_codes)

    return combined_path


# sand_path, sand_centers = create_random_circles_path(num_circles=8, radius=0.004, bounds=(-0.5, 0.5))
sand_path = create_sand_particles()

# Define the vertices for the triangles
concrete_vertices = [
    (0.14, 0.08), (0.145, 0.045), (0.11, 0.05), (0.14, 0.08),  # First triangle
    (-0.085, 0.395), (-0.12, 0.36), (-0.08, 0.35), (-0.085, 0.395),  # Second triangle
    (0.34, -0.21), (0.325, -0.235), (0.31, -0.22), (0.34, -0.21),  # Fourth triangle
    (-0.29, 0.16), (-0.26, 0.14), (-0.25, 0.17), (-0.29, 0.16),  # Sixth triangle
    (-0.45, 0.485), (-0.43, 0.49), (-0.435, 0.475), (-0.45, 0.485)  # eighth triangle
]
# # Define the path codes
concrete_codes = [
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # First triangle
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Second triangle
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Fourth triangle
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Sixth triangle
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY   # eighth triangle
]
# Create the path
concrete_path = Path(concrete_vertices, concrete_codes)

# Define the vertices for the lines
clay_vertices = [
    (-0.5, -0.3), (0.1, -0.3),  # First line
    (-0.5, -0.025), (0.5, -0.025),  # Second line
    (-0.5, 0.11), (0.5, 0.11),  # Third line
    (-0.5, 0.25), (0.5, 0.25)  # Fourth line
]

# Define the path codes
clay_codes = [
    Path.MOVETO, Path.LINETO,  # First line
    Path.MOVETO, Path.LINETO,  # Second line
    Path.MOVETO, Path.LINETO,  # Third line
    Path.MOVETO, Path.LINETO   # Fourth line
]

# Create the path
clay_path = Path(clay_vertices, clay_codes)

# # Create the path
# rock_path = Path(rock_vertices, rock_codes)

# def generate_random_circles(num_circles, min_radius, max_radius, box_size):
#     circles = []
#     while len(circles) < num_circles:
#         radius = random.uniform(min_radius, max_radius)
#         x = random.uniform(-box_size / 2 + radius, box_size / 2 - radius)
#         y = random.uniform(-box_size / 2 + radius, box_size / 2 - radius)
#         new_circle = (x, y, radius)

#         # Check for overlap
#         overlap = False
#         for circle in circles:
#             dist = np.sqrt((circle[0] - x) ** 2 + (circle[1] - y) ** 2)
#             if dist < circle[2] + radius:
#                 overlap = True
#                 break

#         if not overlap:
#             circles.append(new_circle)

#     return circles

# # Generate random circles
# num_circles = 17
# min_radius = 0.05
# max_radius = 0.15
# box_size = 1.0
# circles = generate_random_circles(num_circles, min_radius, max_radius, box_size)

# import csv
# with open('circles.csv', 'w', newline='') as csvfile:
#     circle_writer = csv.writer(csvfile)
#     circle_writer.writerow(['x', 'y', 'radius'])  # Write header
#     for circle in circles:
#         circle_writer.writerow(circle)


# circles = [
#     (-0.03541971446745368, 0.03530961524367926, 0.13485501681144432),
#     (0.39586739000795634, 0.1898114634099929, 0.10086421004495705),
#     (-0.24529724725743002, -0.236860394705399, 0.14015347816171053),
#     (0.2206617317717347, -0.27570087661830356, 0.12201057299066127),
#     (-0.2997170783902612, 0.2589607950720215, 0.10612104436327982),
#     (0.05674053324424355, 0.4054700966516354, 0.08974324657577253),
#     (-0.37369286125112045, 0.047441969575782506, 0.11411936421548455),
#     (0.1784327681049116, 0.07444548488184732, 0.05831758795450901),
#     (0.343150920106714, -0.11318018339665675, 0.06334496858254703),
#     (0.002887713307395401, -0.31816295017377216, 0.08729420279658025),
#     (-0.10417186127509104, 0.2443491741373156, 0.07082540843689435),
#     (-0.42898814506669325, -0.4268629390852136, 0.07043708712037416),
#     (-0.16880443141144647, 0.4024572152390359, 0.05380603235842349),
#     (0.14490697354161708, -0.43517065265474114, 0.050203099763803306),
#     (0.41901669332259844, -0.3361915487347224, 0.06053053864100717),
#     (0.21980273905818293, 0.26876447028629613, 0.06318104944238553),
#     (0.352733352893314, 0.3751283507409835, 0.0689769401831766)
# ]

# # Create the vertices and path codes for the circles
# rock_vertices = []
# rock_codes = []
# for circle in circles:
#     circle_vertices, circle_codes = create_circle_path(circle[0], circle[1], circle[2])
#     rock_vertices.extend(circle_vertices)
#     rock_codes.extend(circle_codes)

rock_1 = [
    (0.101823942, 0.035309615),
    (0.091670307, 0.068507265),
    (0.083786123, 0.095485825),
    (0.061040737, 0.124412963),
    (0.030190693, 0.147218313),
    (-0.00019322, 0.154319125),
    (-0.035193789, 0.175099671),
    (-0.066366543, 0.165656221),
    (-0.101649541, 0.153766267),
    (-0.124614939, 0.136813679),
    (-0.145205997, 0.103887273),
    (-0.157499338, 0.072909955),
    (-0.166240819, 0.035309615),
    (-0.163869312, 0.003423718),
    (-0.149358387, -0.026831379),
    (-0.122916368, -0.059912336),
    (-0.099025067, -0.073937276),
    (-0.066669342, -0.083329362),
    (-0.035641689, -0.081754524),
    (0.001907899, -0.07873006),
    (0.034924529, -0.064140345),
    (0.062158671, -0.050366064),
    (0.092171249, -0.020589508),
    (0.100043251, 0.005109042),
    (0.101823942, 0.035309615)
]

rock_2 = [
    (0.49, 0.189811463),
    (0.495, 0.2217578),
    (0.489993488, 0.248879219),
    (0.47319713, 0.272544186),
    (0.448778593, 0.290022539),
    (0.424629245, 0.308417407),
    (0.395292593, 0.300509653),
    (0.365428343, 0.290021634),
    (0.337449127, 0.280944343),
    (0.31565585, 0.263363406),
    (0.290943829, 0.244401079),
    (0.281008108, 0.216764168),
    (0.273540788, 0.189811463),
    (0.289080515, 0.164196158),
    (0.297846694, 0.141198222),
    (0.321452844, 0.12257161),
    (0.342352268, 0.109387168),
    (0.365952392, 0.095016201),
    (0.395279154, 0.08593888),
    (0.426985851, 0.090249538),
    (0.454079039, 0.1071686),
    (0.47909005, 0.121291066),
    (0.495546003, 0.139163674),
    (0.499, 0.165120141),
    (0.49, 0.189811463)
]

rock_3 = [
    (-0.078707637, -0.236860395),
    (-0.081288664, -0.199011225),
    (-0.096494803, -0.161938973),
    (-0.129851267, -0.133185001),
    (-0.1613403, -0.108664011),
    (-0.20305511, -0.087194941),
    (-0.245719064, -0.07980819),
    (-0.280889439, -0.078106317),
    (-0.319292525, -0.105088415),
    (-0.347438704, -0.128573383),
    (-0.365110445, -0.157794026),
    (-0.383095115, -0.196808979),
    (-0.387335828, -0.236860395),
    (-0.377278103, -0.276103831),
    (-0.369362228, -0.310713971),
    (-0.342523371, -0.341514827),
    (-0.314738144, -0.37037771),
    (-0.282662654, -0.38080607),
    (-0.245407333, -0.374733959),
    (-0.208400457, -0.366752321),
    (-0.176147476, -0.362777786),
    (-0.148005426, -0.340896013),
    (-0.122171758, -0.309056725),
    (-0.103646869, -0.274453427),
    (-0.078707637, -0.236860395)
]

rock_4 = [
    (0.349872627, -0.275700877),
    (0.352725847, -0.239756426),
    (0.332597189, -0.204642633),
    (0.316984115, -0.178982632),
    (0.284609021, -0.156770339),
    (0.255349279, -0.141818977),
    (0.220966552, -0.148383982),
    (0.190929777, -0.153573033),
    (0.166691975, -0.166437323),
    (0.139547527, -0.179812187),
    (0.125107649, -0.212393451),
    (0.107607539, -0.240918639),
    (0.104338609, -0.275700877),
    (0.097173576, -0.309536506),
    (0.110130243, -0.341778698),
    (0.135896362, -0.362802989),
    (0.16018494, -0.382954539),
    (0.188972739, -0.399163932),
    (0.220428768, -0.406725792),
    (0.253718345, -0.409615235),
    (0.285914338, -0.392353702),
    (0.308421581, -0.376069472),
    (0.332489851, -0.346819216),
    (0.341288087, -0.312136987),
    (0.349872627, -0.275700877)
]

rock_5 = [
    (-0.177065683, 0.258960795),
    (-0.187591648, 0.284693197),
    (-0.194560011, 0.309704639),
    (-0.217660322, 0.329027555),
    (-0.241160061, 0.343421427),
    (-0.269575251, 0.352545491),
    (-0.299536679, 0.368189084),
    (-0.327336667, 0.369427391),
    (-0.354899, 0.354171061),
    (-0.377774799, 0.337267463),
    (-0.38781819, 0.314346682),
    (-0.402026314, 0.287555603),
    (-0.406592471, 0.258960795),
    (-0.416690248, 0.231090753),
    (-0.397444237, 0.202468887),
    (-0.384237048, 0.181623753),
    (-0.358290177, 0.164206564),
    (-0.330493014, 0.158145952),
    (-0.300099885, 0.148067446),
    (-0.270134337, 0.148959314),
    (-0.241848907, 0.159219353),
    (-0.215662005, 0.179447169),
    (-0.197471541, 0.20376832),
    (-0.18841483, 0.228365499),
    (-0.177065683, 0.258960795)
]

rock_6 = [
    (0.141612208, 0.405470097),
    (0.140952743, 0.432646541),
    (0.130242097, 0.457923065),
    (0.121272646, 0.477772839),
    (0.099356615, 0.495883123),
    (0.08041404, 0.501835341),
    (0.056663667, 0.492672035),
    (0.03255492, 0.492490449),
    (0.009899303, 0.485897562),
    (-0.012869612, 0.466441868),
    (-0.022468859, 0.44993793),
    (-0.032973978, 0.428540598),
    (-0.040827022, 0.405470097),
    (-0.027429204, 0.380697508),
    (-0.023199464, 0.360163706),
    (-0.005274032, 0.338116122),
    (0.01324502, 0.32442022),
    (0.03520921, 0.311817695),
    (0.056403083, 0.314587325),
    (0.0811815, 0.313711958),
    (0.104857453, 0.328141716),
    (0.124426675, 0.340192277),
    (0.140990388, 0.36008882),
    (0.154423568, 0.381834911),
    (0.141612208, 0.405470097)
]

rock_7 = [
    (-0.258754943, 0.04744197),
    (-0.270760004, 0.07448074),
    (-0.279545385, 0.097244921),
    (-0.296200166, 0.121648499),
    (-0.318293305, 0.141633382),
    (-0.344831476, 0.142996099),
    (-0.373806382, 0.157757128),
    (-0.403844699, 0.158595533),
    (-0.430849496, 0.146135851),
    (-0.451056739, 0.130010838),
    (-0.473005966, 0.106250137),
    (-0.487182067, 0.077917175),
    (-0.488924997, 0.04744197),
    (-0.486575316, 0.015190167),
    (-0.479716414, -0.010586236),
    (-0.453286558, -0.040127787),
    (-0.429158524, -0.054476444),
    (-0.405061491, -0.071869464),
    (-0.373757739, -0.091311503),
    (-0.345966558, -0.088366369),
    (-0.317008343, -0.07312312),
    (-0.297176961, -0.051009523),
    (-0.278466266, -0.021731322),
    (-0.264759997, 0.01150145),
    (-0.258754943, 0.04744197)
]

rock_8 = [
    (0.264765926, 0.109556388),
    (0.263057895, 0.122193976),
    (0.251644674, 0.132616352),
    (0.244059115, 0.144470562),
    (0.228032737, 0.151918643),
    (0.210758666, 0.154396172),
    (0.193057625, 0.153334914),
    (0.174985443, 0.153083475),
    (0.157962331, 0.148275897),
    (0.145181143, 0.139584105),
    (0.131442991, 0.129828841),
    (0.12415434, 0.120932992),
    (0.118334314, 0.109556388),
    (0.13151963, 0.09736495),
    (0.13307908, 0.08559698),
    (0.144121377, 0.076207033),
    (0.160633629, 0.068105609),
    (0.176817308, 0.063156185),
    (0.192900125, 0.061120555),
    (0.212452735, 0.064262761),
    (0.229168615, 0.067181141),
    (0.245385523, 0.076681718),
    (0.260324252, 0.08526647),
    (0.269016058, 0.096758525),
    (0.264765926, 0.109556388)
]

rock_9 = [
    (0.410300755, -0.013180183),
    (0.408365967, 0.006462761),
    (0.400723141, 0.02348607),
    (0.389948575, 0.03935318),
    (0.375318382, 0.048682214),
    (0.359733101, 0.055921068),
    (0.343172602, 0.059131907),
    (0.327774772, 0.057327615),
    (0.313395341, 0.045322195),
    (0.301250591, 0.036032012),
    (0.289602264, 0.023170706),
    (0.28448071, 0.004652128),
    (0.279606018, -0.013180183),
    (0.288142595, -0.030686926),
    (0.292956984, -0.047585133),
    (0.302663338, -0.058702696),
    (0.312750738, -0.068284754),
    (0.328251679, -0.07970807),
    (0.342726534, -0.0793002),
    (0.361494816, -0.081776996),
    (0.378301269, -0.070077791),
    (0.392153304, -0.061471915),
    (0.405635325, -0.046159308),
    (0.414110097, -0.029979174),
    (0.410300755, -0.013180183)
]

rock_10 = [
    (0.085168438, -0.31816295),
    (0.080570757, -0.292289319),
    (0.073874925, -0.270407933),
    (0.055799217, -0.24712194),
    (0.038035087, -0.233054976),
    (0.012777374, -0.219420783),
    (-0.012144843, -0.235669087),
    (-0.036604074, -0.236305095),
    (-0.058303566, -0.242950893),
    (-0.075153083, -0.259138062),
    (-0.090414304, -0.276903839),
    (-0.100342188, -0.295853483),
    (-0.099683299, -0.31816295),
    (-0.083698367, -0.342610552),
    (-0.082065731, -0.369038111),
    (-0.06758737, -0.388498748),
    (-0.052667883, -0.40675215),
    (-0.032620174, -0.409707046),
    (-0.01241362, -0.42089025),
    (0.010278664, -0.417597881),
    (0.034483176, -0.413935443),
    (0.053573899, -0.395996743),
    (0.065892201, -0.372491506),
    (0.07162655, -0.346067959),
    (0.085168438, -0.31816295)
]

rock_11 = [
    (-0.01467598, 0.254349174),
    (-0.019445291, 0.274838347),
    (-0.023328438, 0.294988167),
    (-0.038188435, 0.310295168),
    (-0.050864239, 0.323258411),
    (-0.070567984, 0.336522577),
    (-0.089164091, 0.3305883),
    (-0.10782563, 0.328951497),
    (-0.124815842, 0.320841545),
    (-0.143145273, 0.313031178),
    (-0.15192669, 0.292885033),
    (-0.155986699, 0.275812363),
    (-0.16248557, 0.254349174),
    (-0.163486574, 0.237152513),
    (-0.154769536, 0.221013807),
    (-0.145350096, 0.207625967),
    (-0.127866084, 0.198854276),
    (-0.108863462, 0.190057995),
    (-0.089340477, 0.181950962),
    (-0.068683346, 0.184291892),
    (-0.050093365, 0.1928428),
    (-0.036538524, 0.203551331),
    (-0.022967832, 0.219744528),
    (-0.013618258, 0.236677842),
    (-0.01467598, 0.254349174)
]

rock_12 = [
    (-0.352329194, -0.401862939),
    (-0.352555626, -0.386602444),
    (-0.356382894, -0.372205268),
    (-0.371317378, -0.356485121),
    (-0.383602027, -0.350857679),
    (-0.400497466, -0.343556074),
    (-0.419164403, -0.334079591),
    (-0.438475526, -0.338355229),
    (-0.455106046, -0.346109149),
    (-0.471821943, -0.357068275),
    (-0.481319733, -0.370505197),
    (-0.489922099, -0.385591224),
    (-0.492166733, -0.401862939),
    (-0.49020838, -0.422875125),
    (-0.483985836, -0.440120888),
    (-0.471483239, -0.459566143),
    (-0.457447091, -0.472612933),
    (-0.438826604, -0.481147487),
    (-0.418910805, -0.480684112),
    (-0.400311776, -0.482199601),
    (-0.384781348, -0.473177239),
    (-0.369180591, -0.460794973),
    (-0.360580541, -0.443635997),
    (-0.348516466, -0.422552629),
    (-0.352329194, -0.401862939)
]

rock_13 = [
    (-0.113447583, 0.402457215),
    (-0.11092007, 0.41862837),
    (-0.120195806, 0.431828885),
    (-0.129552714, 0.446480344),
    (-0.14008254, 0.453369732),
    (-0.153706939, 0.460424211),
    (-0.168702426, 0.45349343),
    (-0.181933336, 0.451933019),
    (-0.195321121, 0.448629067),
    (-0.203746162, 0.439141447),
    (-0.212449773, 0.430484391),
    (-0.22015826, 0.416403385),
    (-0.222323492, 0.402457215),
    (-0.229844383, 0.388413502),
    (-0.221826145, 0.375778916),
    (-0.213149848, 0.365718924),
    (-0.198721753, 0.356443493),
    (-0.183956847, 0.348581385),
    (-0.169170905, 0.351021486),
    (-0.152636457, 0.351243088),
    (-0.137488338, 0.359353839),
    (-0.124467028, 0.365031679),
    (-0.116359611, 0.37730036),
    (-0.109095994, 0.389086308),
    (-0.113447583, 0.402457215)
]

rock_14 = [
    (0.177939622, -0.445170653),
    (0.178229533, -0.432555036),
    (0.173284036, -0.421295431),
    (0.164040977, -0.410215302),
    (0.153191911, -0.404000438),
    (0.138900962, -0.400858302),
    (0.125065244, -0.398741739),
    (0.112577817, -0.400091561),
    (0.100371384, -0.406521634),
    (0.091861654, -0.412736642),
    (0.08208882, -0.421855384),
    (0.076928321, -0.434306938),
    (0.078069195, -0.445170653),
    (0.075951071, -0.459194322),
    (0.078879211, -0.471445594),
    (0.089869964, -0.483910615),
    (0.099501449, -0.492013949),
    (0.111397406, -0.496212977),
    (0.124751408, -0.494742124),
    (0.137981698, -0.495287171),
    (0.150057453, -0.486405213),
    (0.161580196, -0.479866814),
    (0.170260534, -0.470584242),
    (0.174113077, -0.457726495),
    (0.177939622, -0.445170653)
]

rock_15 = [
    (0.485568086, -0.416191549),
    (0.483623046, -0.402497461),
    (0.474834565, -0.390199228),
    (0.466261594, -0.377212645),
    (0.450079966, -0.371024553),
    (0.435243751, -0.367349689),
    (0.418679072, -0.357575768),
    (0.402439746, -0.357518813),
    (0.384797337, -0.364514296),
    (0.369901817, -0.374147082),
    (0.361874298, -0.387305501),
    (0.356638932, -0.400323061),
    (0.349834073, -0.416191549),
    (0.346531527, -0.431338574),
    (0.35784272, -0.447300746),
    (0.369259333, -0.458755694),
    (0.382228931, -0.469873827),
    (0.399675518, -0.473334313),
    (0.418861643, -0.481473817),
    (0.435471954, -0.481447781),
    (0.451897435, -0.471531205),
    (0.461881171, -0.462172181),
    (0.471222139, -0.450048024),
    (0.482631596, -0.433778177),
    (0.485568086, -0.416191549)
]

rock_16 = [
    (0.276833225, 0.29376447),
    (0.275867329, 0.311341523),
    (0.265919402, 0.328308781),
    (0.25568184, 0.339858887),
    (0.24130028, 0.352017526),
    (0.225462052, 0.357192307),
    (0.207430612, 0.35843478),
    (0.191104368, 0.354569222),
    (0.178173145, 0.350773996),
    (0.166268171, 0.3370433),
    (0.152502656, 0.326555483),
    (0.147103207, 0.310217964),
    (0.145010584, 0.29376447),
    (0.143434048, 0.276959856),
    (0.146299823, 0.259237744),
    (0.159225863, 0.244967475),
    (0.171447559, 0.234980473),
    (0.188834884, 0.229119233),
    (0.207249231, 0.227052089),
    (0.223738972, 0.229031916),
    (0.238084318, 0.236590336),
    (0.250882037, 0.246797861),
    (0.263213029, 0.259473908),
    (0.270890926, 0.275523989),
    (0.276833225, 0.29376447)
]

rock_17 = [
    (0.415016605, 0.369856936),
    (0.415491404, 0.385494043),
    (0.407745752, 0.405128351),
    (0.39663222, 0.41),
    (0.384895367, 0.422283694),
    (0.369275744, 0.437089735),
    (0.352561959, 0.454933851),
    (0.334199963, 0.461893805),
    (0.318137115, 0.472739416),
    (0.299852504, 0.479670114),
    (0.264445466, 0.474959451),
    (0.256518044, 0.465579038),
    (0.252322979, 0.458085251),
    (0.253821712, 0.440474344),
    (0.252325036, 0.424164398),
    (0.255962953, 0.405128351),
    (0.268676715, 0.384064501),
    (0.285846635, 0.362428175),
    (0.302402151, 0.344752543),
    (0.37146208, 0.330954996),
    (0.390927717, 0.328704504),
    (0.407696103, 0.333046449),
    (0.417768072, 0.333896662),
    (0.426339543, 0.34355979),
    (0.415016605, 0.369856936)
]

rock_18 = [
    (0.235812741, -0.074445485),
    (0.234446316, -0.090242469),
    (0.225315739, -0.103270441),
    (0.219247292, -0.118088203),
    (0.20642619, -0.127398304),
    (0.192606933, -0.130495215),
    (0.1784461, -0.129168642),
    (0.163988354, -0.128854344),
    (0.150369865, -0.122844871),
    (0.140144914, -0.111980131),
    (0.129154393, -0.099786051),
    (0.123323472, -0.08866624),
    (0.118667451, -0.074445485),
    (0.129215704, -0.059206187),
    (0.130463264, -0.044496224),
    (0.139297101, -0.032758792),
    (0.152506903, -0.022632011),
    (0.165453847, -0.016445231),
    (0.1783201, -0.013900694),
    (0.193962188, -0.017828451),
    (0.207334892, -0.021476426),
    (0.220308419, -0.033352148),
    (0.232259402, -0.044083088),
    (0.239212846, -0.058448157),
    (0.235812741, -0.074445485)
]

rock_19 = [
    (0.116206318, 0.208517655),
    (0.114839893, 0.224514983),
    (0.105709316, 0.224514983),
    (0.099640869, 0.240311967),
    (0.086819767, 0.253339938),
    (0.07300051, 0.268157701),
    (0.058839677, 0.277467802),
    (0.044381931, 0.280564713),
    (0.030763442, 0.27923814),
    (0.020538491, 0.278923841),
    (0.00954797, 0.272914369),
    (0.003717049, 0.262049629),
    (-0.000938972, 0.249855549),
    (0.009609281, 0.238735737),
    (0.010856841, 0.224514983),
    (0.019690678, 0.209275685),
    (0.03290048, 0.194565722),
    (0.045847424, 0.182828289),
    (0.058713677, 0.172701509),
    (0.074355765, 0.166514729),
    (0.087728469, 0.163970191),
    (0.100701996, 0.167897949),
    (0.112652979, 0.171545924),
    (0.119606423, 0.183421645),
    (0.116206318, 0.208517655)
]

rock_20 = [
    (-0.342060378, 0.444829347),
    (-0.341770467, 0.457444964),
    (-0.346715964, 0.468704569),
    (-0.355959023, 0.479784698),
    (-0.366808089, 0.485999562),
    (-0.381099038, 0.489141698),
    (-0.394934756, 0.491258261),
    (-0.407422183, 0.489908439),
    (-0.419628616, 0.483478366),
    (-0.428138346, 0.477263358),
    (-0.43791118, 0.468144616),
    (-0.443071679, 0.455693062),
    (-0.441930805, 0.444829347),
    (-0.444048929, 0.430805678),
    (-0.441120789, 0.418554406),
    (-0.430130036, 0.406089385),
    (-0.420498551, 0.397986051),
    (-0.408602594, 0.393787023),
    (-0.395248592, 0.395257876),
    (-0.382018302, 0.394712829),
    (-0.369942547, 0.403594787),
    (-0.358419804, 0.410133186),
    (-0.349739466, 0.419415758),
    (-0.345886923, 0.432273505),
    (-0.342060378, 0.444829347)
]

rock_21 = [
    (-0.113447583, -0.437542785),
    (-0.11092007, -0.42137163),
    (-0.120195806, -0.408171115),
    (-0.129552714, -0.393519656),
    (-0.14008254, -0.386630268),
    (-0.153706939, -0.379575789),
    (-0.168702426, -0.38650657),
    (-0.181933336, -0.388066981),
    (-0.195321121, -0.391370933),
    (-0.203746162, -0.400858553),
    (-0.212449773, -0.409515609),
    (-0.22015826, -0.423596615),
    (-0.222323492, -0.437542785),
    (-0.229844383, -0.451586498),
    (-0.221826145, -0.464221084),
    (-0.213149848, -0.474281076),
    (-0.198721753, -0.483556507),
    (-0.183956847, -0.491418615),
    (-0.169170905, -0.488978514),
    (-0.152636457, -0.488756912),
    (-0.137488338, -0.480646161),
    (-0.124467028, -0.474968321),
    (-0.116359611, -0.46269964),
    (-0.109095994, -0.450913692),
    (-0.113447583, -0.437542785)
]

rock_22 = [
    (0.492246019, -0.186356719),
    (0.493654731, -0.171963333),
    (0.486544115, -0.159733273),
    (0.477244131, -0.146867479),
    (0.466554686, -0.140315353),
    (0.452597012, -0.135217046),
    (0.438181409, -0.137624155),
    (0.42532224, -0.139079271),
    (0.412525132, -0.143946283),
    (0.404057746, -0.151797597),
    (0.394819524, -0.160685497),
    (0.388385031, -0.173951776),
    (0.387872852, -0.186356719),
    (0.383053344, -0.20039041),
    (0.388526533, -0.212833339),
    (0.398360058, -0.224095846),
    (0.410389848, -0.232785228),
    (0.423720279, -0.238815796),
    (0.437790252, -0.236860319),
    (0.452672621, -0.237022041),
    (0.466284558, -0.228525687),
    (0.478556584, -0.222417567),
    (0.486950461, -0.211641941),
    (0.492508542, -0.199320093),
    (0.492246019, -0.186356719)
]

rock_23 = [
    (-0.42887699, -0.208178359),
    (-0.428172634, -0.200981666),
    (-0.431727942, -0.194866637),
    (-0.436377934, -0.188433739),
    (-0.441722657, -0.185157677),
    (-0.448701494, -0.182608523),
    (-0.455909295, -0.183812077),
    (-0.46233888, -0.184539636),
    (-0.468737434, -0.186973142),
    (-0.472971127, -0.190898799),
    (-0.477590238, -0.195342748),
    (-0.480807485, -0.201975888),
    (-0.481063574, -0.208178359),
    (-0.483473328, -0.215195205),
    (-0.480736734, -0.221416669),
    (-0.475819971, -0.227047923),
    (-0.469805076, -0.231392614),
    (-0.46313986, -0.234407898),
    (-0.456104874, -0.233430159),
    (-0.44866369, -0.233511021),
    (-0.441857721, -0.229262844),
    (-0.435721708, -0.226208784),
    (-0.431524769, -0.22082097),
    (-0.428745729, -0.214660047),
    (-0.42887699, -0.208178359)
]

rocks = [rock_1, rock_2, rock_3, rock_4, rock_5, rock_6, rock_7, rock_8, rock_9, rock_10, rock_11, rock_12, rock_13,
         rock_14, rock_15, rock_16, rock_17, rock_18, rock_19, rock_20, rock_21, rock_22, rock_23]

rock_vertices = []
rock_codes = []
for rock in rocks:
    rock_vertices += rock
    rock_codes += [Path.MOVETO] + [Path.LINETO] * (len(rock) - 2) + [Path.CLOSEPOLY]

# Create the path
rock_path = Path(rock_vertices, rock_codes)

earth_vertices = [
    (-0.5, -0.4375), (0, -0.4375),  # First line
    (-0.5, -0.25), (0, -0.25),  # Second line
    (-0.5, -0.0625), (0, -0.0625),  # Third line
    (-0.4375, 0), (-0.4375, 0.5),  # Fourth line
    (-0.25, 0), (-0.25, 0.5),  # Fifth line
    (-0.0625, 0), (-0.0625, 0.5),  # Sixth line
    (0, 0.4375), (0.5, 0.4375),  # Seventh line
    (0, 0.25), (0.5, 0.25),  # Eighth line
    (0, 0.0625), (0.5, 0.0625),  # Ninth line
    (0.4375, 0), (0.4375, -0.5),  # Tenth line
    (0.25, 0), (0.25, -0.5),  # Eleventh line
    (0.0625, 0), (0.0625, -0.5)  # Twelfth line
]

# Define the path codes
earth_codes = [
    Path.MOVETO, Path.LINETO,  # First line
    Path.MOVETO, Path.LINETO,  # Second line
    Path.MOVETO, Path.LINETO,  # Third line
    Path.MOVETO, Path.LINETO,  # Fourth line
    Path.MOVETO, Path.LINETO,  # Fifth line
    Path.MOVETO, Path.LINETO,  # Sixth line
    Path.MOVETO, Path.LINETO,  # Seventh line
    Path.MOVETO, Path.LINETO,  # Eighth line
    Path.MOVETO, Path.LINETO,  # Ninth line
    Path.MOVETO, Path.LINETO,  # Tenth line
    Path.MOVETO, Path.LINETO,  # Eleventh line
    Path.MOVETO, Path.LINETO   # Twelfth line
]

# Create the path
earth_path = Path(earth_vertices, earth_codes)

# Define the vertices for the lines
grass_vertices = [
    (-0.24, 0.08), (0, -0.24),  # First line
    (0, -0.24), (0.24, 0.08),  # Second line
    (0, -0.24), (0, 0.24),     # Vertical line
]

# Define the path codes
grass_codes = [
    Path.MOVETO, Path.LINETO,
    Path.MOVETO, Path.LINETO,
    Path.MOVETO, Path.LINETO,  # Vertical line
]

# Create the path
grass_path = Path(grass_vertices, grass_codes)

# Define the vertices for the lines
metal_vertices = [
    (-0.5, 0.35), (-0.35, 0.5),  # First line
    (-0.5, 0.2), (-0.2, 0.5),  # Second line
    (-0.5, 0), (0, 0.5),  # Third line
    (-0.5, -0.1), (0.1, 0.5),  # Fourth line
    (-0.5, -0.35), (0.35, 0.5),  # Fifth line
    (-0.5, -0.5), (0.5, 0.5),  # Sixth line
    (-0.35, -0.5), (0.5, 0.35),  # Seventh line
    (-0.2, -0.5), (0.5, 0.2),  # Eighth line
    (0, -0.5), (0.5, 0),  # Ninth line
    (0.1, -0.5), (0.5, -0.1),  # Tenth line
    (0.35, -0.5), (0.5, -0.35)  # Eleventh line
]

# Define the path codes
metal_codes = [
    Path.MOVETO, Path.LINETO,  # First line
    Path.MOVETO, Path.LINETO,  # Second line
    Path.MOVETO, Path.LINETO,  # Third line
    Path.MOVETO, Path.LINETO,  # Fourth line
    Path.MOVETO, Path.LINETO,  # Fifth line
    Path.MOVETO, Path.LINETO,  # Sixth line
    Path.MOVETO, Path.LINETO,  # Seventh line
    Path.MOVETO, Path.LINETO,  # Eighth line
    Path.MOVETO, Path.LINETO,  # Ninth line
    Path.MOVETO, Path.LINETO,  # Tenth line
    Path.MOVETO, Path.LINETO   # Eleventh line
]

# Create the path
metal_path = Path(metal_vertices, metal_codes)

# Define the vertices for the lines
zigzag_vertices = [
    # First segment (3)
    (-0.5, 0.45), (-0.45, 0.45), (-0.45, 0.5),

    # Second segment (7)
    (-0.5, 0.25), (-0.45, 0.25), (-0.45, 0.35), (-0.35, 0.35), (-0.35, 0.45), (-0.25, 0.45), (-0.25, 0.5),

    # Third segment (11)
    (-0.5, 0.05), (-0.45, 0.05), (-0.45, 0.15), (-0.35, 0.15), (-0.35, 0.25), (-0.25, 0.25), (-0.25, 0.35),
    (-0.15, 0.35), (-0.15, 0.45), (-0.05, 0.45), (-0.05, 0.5),

    # Fourth segment (15)
    (-0.5, -0.15), (-0.45, -0.15), (-0.45, -0.05), (-0.35, -0.05), (-0.35, 0.05), (-0.25, 0.05), (-0.25, 0.15),
    (-0.15, 0.15), (-0.15, 0.25), (-0.05, 0.25), (-0.05, 0.35), (0.05, 0.35), (0.05, 0.45), (0.15, 0.45),
    (0.15, 0.5),

    # Fifth segment (21)
    (-0.5, -0.35), (-0.45, -0.35), (-0.45, -0.25), (-0.35, -0.25), (-0.35, -0.15), (-0.25, -0.15), (-0.25, -0.05),
    (-0.15, -0.05), (-0.15, 0.05), (-0.05, 0.05), (-0.05, 0.15), (0.05, 0.15), (0.05, 0.25), (0.15, 0.25),
    (0.15, 0.35), (0.25, 0.35), (0.25, 0.45), (0.35, 0.45), (0.35, 0.55), (0.45, 0.55), (0.45, 0.5),

    # Sixth segment (22)
    (-0.45, -0.5), (-0.45, -0.45), (-0.35, -0.45), (-0.35, -0.35), (-0.25, -0.35), (-0.25, -0.25), (-0.15, -0.25),
    (-0.15, -0.15), (-0.05, -0.15), (-0.05, -0.05), (0.05, -0.05), (0.05, 0.05), (0.15, 0.05), (0.15, 0.15),
    (0.25, 0.15), (0.25, 0.25), (0.35, 0.25), (0.35, 0.35), (0.45, 0.35), (0.45, 0.45), (0.55, 0.45),
    (0.55, 0.55),

    # Seventh segment (23)
    (-0.25, -0.5), (-0.25, -0.45), (-0.15, -0.45), (-0.15, -0.35), (-0.05, -0.35), (-0.05, -0.25), (0.05, -0.25),
    (0.05, -0.15), (0.15, -0.15), (0.15, -0.05), (0.25, -0.05), (0.25, 0.05), (0.35, 0.05), (0.35, 0.15),
    (0.45, 0.15), (0.45, 0.25), (0.55, 0.25), (0.55, 0.35), (0.65, 0.35), (0.65, 0.45), (0.75, 0.45),
    (0.75, 0.55),

    # Eighth segment (18)
    (-0.05, -0.5), (-0.05, -0.45), (0.05, -0.45), (0.05, -0.35), (0.15, -0.35), (0.15, -0.25), (0.25, -0.25),
    (0.25, -0.15), (0.35, -0.15), (0.35, -0.05), (0.45, -0.05), (0.45, 0.05), (0.55, 0.05), (0.55, 0.15),
    (0.65, 0.15), (0.65, 0.25), (0.75, 0.25), (0.75, 0.35),

    # Ninth segment (22)
    (0.15, -0.5), (0.15, -0.45), (0.25, -0.45), (0.25, -0.35), (0.35, -0.35), (0.35, -0.25), (0.45, -0.25),
    (0.45, -0.15), (0.55, -0.15), (0.55, -0.05), (0.65, -0.05), (0.65, 0.05), (0.75, 0.05), (0.75, 0.15),
    (0.85, 0.15), (0.85, 0.25), (0.95, 0.25), (0.95, 0.35), (1.05, 0.35), (1.05, 0.45), (1.15, 0.45), (1.15, 0.55),
    (1.25, 0.55),

    # Tenth segment (22)
    (0.35, -0.5), (0.35, -0.45), (0.45, -0.45), (0.45, -0.35), (0.55, -0.35), (0.55, -0.25), (0.65, -0.25),
    (0.65, -0.15), (0.75, -0.15), (0.75, -0.05), (0.85, -0.05), (0.85, 0.05), (0.95, 0.05), (0.95, 0.15),
    (1.05, 0.15), (1.05, 0.25), (1.15, 0.25), (1.15, 0.35), (1.25, 0.35), (1.25, 0.45), (1.35, 0.45),
    (1.35, 0.55)
]

# Define the path codes
zigzag_codes = [
    # First segment (3)
    Path.MOVETO, Path.LINETO, Path.LINETO,  # First segment
    # Second segment (7)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Third segment (11)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO,  # Third segment
    # Fourth segment (15)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Fifth segment (21)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Sixth segment (22)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Seventh segment (23)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Eighth segment (18)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO,
    # Ninth segment (22)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    # Tenth segment (22)
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO,
    Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO
]

# Create the path
zigzag_path = Path(zigzag_vertices, zigzag_codes)

# Define the vertices for the hexagons
honeycomb_vertices = [
    (-0.5, 0.0), (-0.25, 0.5), (0.25, 0.5), (0.5, 0.0), (0.25, -0.5), (-0.25, -0.5), (-0.5, 0.0),  # First hexagon
    # (0, 0.5), (0.25, 0.75), (0.75, 0.75), (1, 0.5), (0.75, 0.25), (0.25, 0.25), (0, 0.5),  # Second hexagon
    # (0.5, 0), (0.75, 0.25), (1.25, 0.25), (1.5, 0), (1.25, -0.25), (0.75, -0.25), (0.5, 0),  # Third hexagon
    # (0, -0.5), (0.25, -0.25), (0.75, -0.25), (1, -0.5), (0.75, -0.75), (0.25, -0.75), (0, -0.5),  # Fourth hexagon
    # (-0.5, -0.25), (-0.25, 0), (0.25, 0), (0.5, -0.25), (0.25, -0.5), (-0.25, -0.5), (-0.5, -0.25)  # Fifth hexagon
]

# Define the path codes
honeycomb_codes = [
    Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # First hexagon
    # Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Second hexagon
    # Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Third hexagon
    # Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY,  # Fourth hexagon
    # Path.MOVETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.LINETO, Path.CLOSEPOLY   # Fifth hexagon
]

# def hexagon_vertices(center, size):
#     """Generate the vertices of a hexagon given a center and size."""
#     angle = np.linspace(0, 2 * np.pi, 7)
#     x_hexagon = center[0] + size * np.cos(angle)
#     y_hexagon = center[1] + size * np.sin(angle)
#     return x_hexagon, y_hexagon

# def rotate_vertices(vertices, angle):
#     """Rotate vertices by a given angle in radians."""
#     rotation_matrix = np.array([[np.cos(angle), -np.sin(angle)], [np.sin(angle), np.cos(angle)]])
#     rotated_vertices = np.dot(vertices, rotation_matrix)
#     return rotated_vertices

# def create_honeycomb_path():
#     """Create the vertices and path codes for a hexagon that fits within the specified bounds."""
#     size = 1 / np.sqrt(3)
#     center = (0, 0)
#     x_hexagon, y_hexagon = hexagon_vertices(center, size)

#     # Combine x and y coordinates into a single array for rotation
#     vertices = np.vstack((x_hexagon, y_hexagon)).T

#     # Rotate vertices by 90 degrees (π/2 radians)
#     rotated_vertices = rotate_vertices(vertices, 0.0)

#     # Define the vertices for the hexagon including half the line length for tiling
#     honeycomb_vertices = []
#     honeycomb_codes = []

#     for i in range(6):
#         x1, y1 = rotated_vertices[i]
#         x2, y2 = rotated_vertices[i + 1]
#         mid_x, mid_y = (x1 + x2) / 2, (y1 + y2) / 2

#         if i == 0:
#             honeycomb_vertices.append((x1, y1))
#             honeycomb_codes.append(Path.MOVETO)
#         else:
#             honeycomb_vertices.append((x1, y1))
#             honeycomb_codes.append(Path.LINETO)

#         honeycomb_vertices.append((mid_x, mid_y))
#         honeycomb_codes.append(Path.LINETO)

#     honeycomb_vertices.append((rotated_vertices[0][0], rotated_vertices[0][1]))
#     honeycomb_codes.append(Path.CLOSEPOLY)

#     return honeycomb_vertices, honeycomb_codes

# Create the path
honeycomb_path = Path(honeycomb_vertices, honeycomb_codes)
# honeycomb_path = Path(create_honeycomb_path())

wood_vertices = [
    (-0.5, 0.3),
    (-0.4, 0.5),

    (-0.5, 0.025),
    (-0.3, 0.4),
    (-0.2, 0.5),

    (-0.5, -0.15),
    (-0.4, -0.03),
    (-0.21, 0.35),
    (-0.035, 0.5),

    (-0.5, -0.235),
    (-0.31, -0.065),
    (-0.15, 0.28),
    (0.125, 0.5),

    (-0.31, -0.065),
    (0.05, 0.1),
    (0.17, 0.28),

    (-0.5, -0.33),
    (-0.3, -0.16),
    (0.13, 0.07),
    (0.335, 0.5),

    (-0.5, -0.44),
    (-0.3, -0.25),
    (0, -0.11),
    (0.22, 0),
    (0.475, 0.5),

    (-0.4, -0.5),
    (-0.25, -0.35),
    (0.25, -0.1),
    (0.5, 0.3),

    (-0.2, -0.5),
    (-0.16, -0.46),
    (0.24, -0.25),
    (0.5, 0.025),

    (-0.035, -0.5),
    (0.32, -0.32),
    (0.5, -0.15),

    (0.125, -0.5),
    (0.35, -0.38),
    (0.5, -0.235),

    (0.335, -0.5),
    (0.5, -0.33),

    (0.475, -0.5),
    (0.5, -0.44),

    (0.14, 0.32),
    (0, 0.15),
    (-0.1, 0.1),
    (-0.15, 0.12),
    (0, 0.28),
    (0.09, 0.38),
    (0.14, 0.4),
    (0.19, 0.43),
    (0.23, 0.48)
]

# Define the path codes
wood_codes = [Path.MOVETO] + [Path.LINETO] * (len(wood_vertices) - 1)
wood_codes[2] = Path.MOVETO
wood_codes[5] = Path.MOVETO
wood_codes[9] = Path.MOVETO
wood_codes[13] = Path.MOVETO
wood_codes[16] = Path.MOVETO
wood_codes[20] = Path.MOVETO
wood_codes[25] = Path.MOVETO
wood_codes[29] = Path.MOVETO
wood_codes[33] = Path.MOVETO
wood_codes[36] = Path.MOVETO
wood_codes[39] = Path.MOVETO
wood_codes[41] = Path.MOVETO
wood_codes[43] = Path.MOVETO

wood_path = Path(wood_vertices, wood_codes)


water_vertices = [
    (-0.5, 0.35),
    (-0.479166667, 0.324118095),
    (-0.458333333, 0.3),
    (-0.4375, 0.279289322),
    (-0.416666667, 0.26339746),
    (-0.395833333, 0.253407417),
    (-0.375, 0.25),
    (-0.354166667, 0.253407417),
    (-0.333333333, 0.26339746),
    (-0.3125, 0.279289322),
    (-0.291666667, 0.3),
    (-0.270833333, 0.324118095),
    (-0.25, 0.35),
    (-0.229166667, 0.324118095),
    (-0.208333333, 0.3),
    (-0.1875, 0.279289322),
    (-0.166666667, 0.26339746),
    (-0.145833333, 0.253407417),
    (-0.125, 0.25),
    (-0.104166667, 0.253407417),
    (-0.083333333, 0.26339746),
    (-0.0625, 0.279289322),
    (-0.041666667, 0.3),
    (-0.020833333, 0.324118095),
    (-1.17961E-16, 0.35),
    (0.020833333, 0.324118095),
    (0.041666667, 0.3),
    (0.0625, 0.279289322),
    (0.083333333, 0.26339746),
    (0.104166667, 0.253407417),
    (0.125, 0.25),
    (0.145833333, 0.253407417),
    (0.166666667, 0.26339746),
    (0.1875, 0.279289322),
    (0.208333333, 0.3),
    (0.229166667, 0.324118095),
    (0.25, 0.35),
    (0.270833333, 0.324118095),
    (0.291666667, 0.3),
    (0.3125, 0.279289322),
    (0.333333333, 0.26339746),
    (0.354166667, 0.253407417),
    (0.375, 0.25),
    (0.395833333, 0.253407417),
    (0.416666667, 0.26339746),
    (0.4375, 0.279289322),
    (0.458333333, 0.3),
    (0.479166667, 0.324118095),
    (0.5, 0.35),

    (-0.5, 0.05),
    (-0.479166667, 0.024118095),
    (-0.458333333, 0),
    (-0.4375, -0.020710678),
    (-0.416666667, -0.03660254),
    (-0.395833333, -0.046592583),
    (-0.375, -0.05),
    (-0.354166667, -0.046592583),
    (-0.333333333, -0.03660254),
    (-0.3125, -0.020710678),
    (-0.291666667, 0),
    (-0.270833333, 0.024118095),
    (-0.25, 0.05),
    (-0.229166667, 0.024118095),
    (-0.208333333, 0),
    (-0.1875, -0.020710678),
    (-0.166666667, -0.03660254),
    (-0.145833333, -0.046592583),
    (-0.125, -0.05),
    (-0.104166667, -0.046592583),
    (-0.083333333, -0.03660254),
    (-0.0625, -0.020710678),
    (-0.041666667, 0),
    (-0.020833333, 0.024118095),
    (-1.17961E-16, 0.05),
    (0.020833333, 0.024118095),
    (0.041666667, 0),
    (0.0625, -0.020710678),
    (0.083333333, -0.03660254),
    (0.104166667, -0.046592583),
    (0.125, -0.05),
    (0.145833333, -0.046592583),
    (0.166666667, -0.03660254),
    (0.1875, -0.020710678),
    (0.208333333, 0),
    (0.229166667, 0.024118095),
    (0.25, 0.05),
    (0.270833333, 0.024118095),
    (0.291666667, 0),
    (0.3125, -0.020710678),
    (0.333333333, -0.03660254),
    (0.354166667, -0.046592583),
    (0.375, -0.05),
    (0.395833333, -0.046592583),
    (0.416666667, -0.03660254),
    (0.4375, -0.020710678),
    (0.458333333, 0),
    (0.479166667, 0.024118095),
    (0.5, 0.05),

    (-0.5, -0.25),
    (-0.479166667, -0.275881905),
    (-0.458333333, -0.3),
    (-0.4375, -0.320710678),
    (-0.416666667, -0.33660254),
    (-0.395833333, -0.346592583),
    (-0.375, -0.35),
    (-0.354166667, -0.346592583),
    (-0.333333333, -0.33660254),
    (-0.3125, -0.320710678),
    (-0.291666667, -0.3),
    (-0.270833333, -0.275881905),
    (-0.25, -0.25),
    (-0.229166667, -0.275881905),
    (-0.208333333, -0.3),
    (-0.1875, -0.320710678),
    (-0.166666667, -0.33660254),
    (-0.145833333, -0.346592583),
    (-0.125, -0.35),
    (-0.104166667, -0.346592583),
    (-0.083333333, -0.33660254),
    (-0.0625, -0.320710678),
    (-0.041666667, -0.3),
    (-0.020833333, -0.275881905),
    (-1.17961E-16, -0.25),
    (0.020833333, -0.275881905),
    (0.041666667, -0.3),
    (0.0625, -0.320710678),
    (0.083333333, -0.33660254),
    (0.104166667, -0.346592583),
    (0.125, -0.35),
    (0.145833333, -0.346592583),
    (0.166666667, -0.33660254),
    (0.1875, -0.320710678),
    (0.208333333, -0.3),
    (0.229166667, -0.275881905),
    (0.25, -0.25),
    (0.270833333, -0.275881905),
    (0.291666667, -0.3),
    (0.3125, -0.320710678),
    (0.333333333, -0.33660254),
    (0.354166667, -0.346592583),
    (0.375, -0.35),
    (0.395833333, -0.346592583),
    (0.416666667, -0.33660254),
    (0.4375, -0.320710678),
    (0.458333333, -0.3),
    (0.479166667, -0.275881905),
    (0.5, -0.25)
]

# Define the path codes
water_codes = [Path.MOVETO] + [Path.LINETO] * (len(water_vertices) - 1)
water_codes[49] = Path.MOVETO
water_codes[98] = Path.MOVETO
water_codes[-1] = Path.MOVETO

water_path = Path(water_vertices, water_codes)


bituminous_vertices = [
    (-0.3, 0),
    (-0.275, 0.051763809),
    (-0.25, 0.1),
    (-0.225, 0.141421356),
    (-0.2, 0.173205081),
    (-0.175, 0.193185165),
    (-0.15, 0.2),
    (-0.125, 0.193185165),
    (-0.1, 0.173205081),
    (-0.075, 0.141421356),
    (-0.05, 0.1),
    (-0.025, 0.051763809),
    (0, 2.4503E-17),
    (0.025, -0.051763809),
    (0.05, -0.1),
    (0.075, -0.141421356),
    (0.1, -0.173205081),
    (0.125, -0.193185165),
    (0.15, -0.2),
    (0.175, -0.193185165),
    (0.2, -0.173205081),
    (0.225, -0.141421356),
    (0.25, -0.1),
    (0.275, -0.051763809),
    (0.3, -4.90059E-17)
]
bituminous_codes = [Path.MOVETO] + [Path.LINETO] * (len(bituminous_vertices) - 1)
bituminous_path = Path(bituminous_vertices, bituminous_codes)


# peat
peat_vertices = [
    (-0.5, 0.4),
    (-0.479166667, 0.374118095),
    (-0.458333333, 0.35),
    (-0.4375, 0.329289322),
    (-0.416666667, 0.31339746),
    (-0.395833333, 0.303407417),
    (-0.375, 0.3),
    (-0.354166667, 0.303407417),
    (-0.333333333, 0.31339746),
    (-0.3125, 0.329289322),
    (-0.291666667, 0.35),
    (-0.270833333, 0.374118095),
    (-0.25, 0.4),
    (-0.229166667, 0.374118095),
    (-0.208333333, 0.35),
    (-0.1875, 0.329289322),
    (-0.166666667, 0.31339746),
    (-0.145833333, 0.303407417),
    (-0.125, 0.3),
    (-0.104166667, 0.303407417),
    (-0.083333333, 0.31339746),
    (-0.0625, 0.329289322),
    (-0.041666667, 0.35),
    (-0.020833333, 0.374118095),
    (-1.17961E-16, 0.4),
    (0.020833333, 0.374118095),
    (0.041666667, 0.35),
    (0.0625, 0.329289322),
    (0.083333333, 0.31339746),
    (0.104166667, 0.303407417),
    (0.125, 0.3),
    (0.145833333, 0.303407417),
    (0.166666667, 0.31339746),
    (0.1875, 0.329289322),
    (0.208333333, 0.35),
    (0.229166667, 0.374118095),
    (0.25, 0.4),
    (0.270833333, 0.374118095),
    (0.291666667, 0.35),
    (0.3125, 0.329289322),
    (0.333333333, 0.31339746),
    (0.354166667, 0.303407417),
    (0.375, 0.3),
    (0.395833333, 0.303407417),
    (0.416666667, 0.31339746),
    (0.4375, 0.329289322),
    (0.458333333, 0.35),
    (0.479166667, 0.374118095),
    (0.5, 0.4),
    (0.5, 0.2),
    (0.479166667, 0.174118095),
    (0.458333333, 0.15),
    (0.4375, 0.129289322),
    (0.416666667, 0.11339746),
    (0.395833333, 0.103407417),
    (0.375, 0.1),
    (0.354166667, 0.103407417),
    (0.333333333, 0.11339746),
    (0.3125, 0.129289322),
    (0.291666667, 0.15),
    (0.270833333, 0.174118095),
    (0.25, 0.2),
    (0.229166667, 0.174118095),
    (0.208333333, 0.15),
    (0.1875, 0.129289322),
    (0.166666667, 0.11339746),
    (0.145833333, 0.103407417),
    (0.125, 0.1),
    (0.104166667, 0.103407417),
    (0.083333333, 0.11339746),
    (0.0625, 0.129289322),
    (0.041666667, 0.15),
    (0.020833333, 0.174118095),
    (-1.17961E-16, 0.2),
    (-0.020833333, 0.174118095),
    (-0.041666667, 0.15),
    (-0.0625, 0.129289322),
    (-0.083333333, 0.11339746),
    (-0.104166667, 0.103407417),
    (-0.125, 0.1),
    (-0.145833333, 0.103407417),
    (-0.166666667, 0.11339746),
    (-0.1875, 0.129289322),
    (-0.208333333, 0.15),
    (-0.229166667, 0.174118095),
    (-0.25, 0.2),
    (-0.270833333, 0.174118095),
    (-0.291666667, 0.15),
    (-0.3125, 0.129289322),
    (-0.333333333, 0.11339746),
    (-0.354166667, 0.103407417),
    (-0.375, 0.1),
    (-0.395833333, 0.103407417),
    (-0.416666667, 0.11339746),
    (-0.4375, 0.129289322),
    (-0.458333333, 0.15),
    (-0.479166667, 0.174118095),
    (-0.5, 0.2),
    (-0.5, 0.4),

    (-0.5, -0.1),
    (-0.479166667, -0.125881905),
    (-0.458333333, -0.15),
    (-0.4375, -0.170710678),
    (-0.416666667, -0.18660254),
    (-0.395833333, -0.196592583),
    (-0.375, -0.2),
    (-0.354166667, -0.196592583),
    (-0.333333333, -0.18660254),
    (-0.3125, -0.170710678),
    (-0.291666667, -0.15),
    (-0.270833333, -0.125881905),
    (-0.25, -0.1),
    (-0.229166667, -0.125881905),
    (-0.208333333, -0.15),
    (-0.1875, -0.170710678),
    (-0.166666667, -0.18660254),
    (-0.145833333, -0.196592583),
    (-0.125, -0.2),
    (-0.104166667, -0.196592583),
    (-0.083333333, -0.18660254),
    (-0.0625, -0.170710678),
    (-0.041666667, -0.15),
    (-0.020833333, -0.125881905),
    (-1.17961E-16, -0.1),
    (0.020833333, -0.125881905),
    (0.041666667, -0.15),
    (0.0625, -0.170710678),
    (0.083333333, -0.18660254),
    (0.104166667, -0.196592583),
    (0.125, -0.2),
    (0.145833333, -0.196592583),
    (0.166666667, -0.18660254),
    (0.1875, -0.170710678),
    (0.208333333, -0.15),
    (0.229166667, -0.125881905),
    (0.25, -0.1),
    (0.270833333, -0.125881905),
    (0.291666667, -0.15),
    (0.3125, -0.170710678),
    (0.333333333, -0.18660254),
    (0.354166667, -0.196592583),
    (0.375, -0.2),
    (0.395833333, -0.196592583),
    (0.416666667, -0.18660254),
    (0.4375, -0.170710678),
    (0.458333333, -0.15),
    (0.479166667, -0.125881905),
    (0.5, -0.1),
    (0.5, -0.3),
    (0.479166667, -0.325881905),
    (0.458333333, -0.35),
    (0.4375, -0.370710678),
    (0.416666667, -0.38660254),
    (0.395833333, -0.396592583),
    (0.375, -0.4),
    (0.354166667, -0.396592583),
    (0.333333333, -0.38660254),
    (0.3125, -0.370710678),
    (0.291666667, -0.35),
    (0.270833333, -0.325881905),
    (0.25, -0.3),
    (0.229166667, -0.325881905),
    (0.208333333, -0.35),
    (0.1875, -0.370710678),
    (0.166666667, -0.38660254),
    (0.145833333, -0.396592583),
    (0.125, -0.4),
    (0.104166667, -0.396592583),
    (0.083333333, -0.38660254),
    (0.0625, -0.370710678),
    (0.041666667, -0.35),
    (0.020833333, -0.325881905),
    (-1.17961E-16, -0.3),
    (-0.020833333, -0.325881905),
    (-0.041666667, -0.35),
    (-0.0625, -0.370710678),
    (-0.083333333, -0.38660254),
    (-0.104166667, -0.396592583),
    (-0.125, -0.4),
    (-0.145833333, -0.396592583),
    (-0.166666667, -0.38660254),
    (-0.1875, -0.370710678),
    (-0.208333333, -0.35),
    (-0.229166667, -0.325881905),
    (-0.25, -0.3),
    (-0.270833333, -0.325881905),
    (-0.291666667, -0.35),
    (-0.3125, -0.370710678),
    (-0.333333333, -0.38660254),
    (-0.354166667, -0.396592583),
    (-0.375, -0.4),
    (-0.395833333, -0.396592583),
    (-0.416666667, -0.38660254),
    (-0.4375, -0.370710678),
    (-0.458333333, -0.35),
    (-0.479166667, -0.325881905),
    (-0.5, -0.3)
]

# Define the path codes
peat_codes = [Path.MOVETO] + [Path.LINETO] * (len(peat_vertices) - 1)
peat_codes[98] = Path.CLOSEPOLY
peat_codes[99] = Path.MOVETO
peat_codes[-1] = Path.CLOSEPOLY

peat_path = Path(peat_vertices, peat_codes)

# metamorphic
metamorphic_vertices = [
    (-0.46, 0.35),
    (-0.415, 0.43),

    (-0.4, 0.025),
    (-0.48, 0.11),
    (-0.43, 0.25),

    (-0.46, -0.08),
    (-0.4, -0.03),

    (-0.19, 0.36),
    (-0.1, 0.46),

    (-0.36, 0.35),
    (-0.28, 0.44),

    (-0.18, 0.31),
    (-0.1, 0.26),
    (-0.08, 0.16),
    (-0.02, 0.13),

    (-0.27, -0.085),
    (-0.17, 0.02),

    (-0.03, 0.05),
    (0.1, -0.05),

    (-0.3, 0.13),
    (-0.23, 0.08),

    (-0.45, -0.47),
    (-0.43, -0.38),

    (-0.34, -0.21),
    (-0.2, -0.34),

    (-0.15, -0.22),
    (-0.09, -0.1),

    (-0.05, -0.24),
    (0.02, -0.28),
    (0.13, -0.14),

    (-0.2, -0.43),
    (-0.1, -0.45),

    (-0.035, -0.38),
    (0.05, -0.45),

    (0.1, -0.31),
    (0.25, -0.38),

    (0.27, -0.2),
    (0.38, -0.03),

    (0.335, -0.45),
    (0.42, -0.33),

    (0.445, 0.1),
    (0.47, 0.2),

    (0.19, 0.12),
    (0.05, 0.15),

    (0, 0.28),
    (0.09, 0.38),

    (0.27, 0.42),
    (0.36, 0.38),
    (0.36, 0.44),

    (0.24, 0.23),
    (0.3, 0.21),
    (0.36, 0.3),

    (0.28, 0.1),
    (0.25, 0.02),
    (0.34, -0.03)
]

# Define the path codes
metamorphic_codes = [Path.MOVETO] + [Path.LINETO] * (len(metamorphic_vertices) - 1)
metamorphic_codes[2] = Path.MOVETO
metamorphic_codes[5] = Path.MOVETO
metamorphic_codes[7] = Path.MOVETO
metamorphic_codes[9] = Path.MOVETO
metamorphic_codes[11] = Path.MOVETO
metamorphic_codes[15] = Path.MOVETO
metamorphic_codes[17] = Path.MOVETO
metamorphic_codes[19] = Path.MOVETO
metamorphic_codes[21] = Path.MOVETO
metamorphic_codes[23] = Path.MOVETO
metamorphic_codes[25] = Path.MOVETO
metamorphic_codes[27] = Path.MOVETO
metamorphic_codes[30] = Path.MOVETO
metamorphic_codes[32] = Path.MOVETO
metamorphic_codes[34] = Path.MOVETO
metamorphic_codes[36] = Path.MOVETO
metamorphic_codes[38] = Path.MOVETO
metamorphic_codes[40] = Path.MOVETO
metamorphic_codes[42] = Path.MOVETO
metamorphic_codes[44] = Path.MOVETO
metamorphic_codes[46] = Path.MOVETO
metamorphic_codes[49] = Path.MOVETO
metamorphic_codes[52] = Path.MOVETO

metamorphic_path = Path(metamorphic_vertices, metamorphic_codes)

# brick
brick_vertices = [
    (-0.5, 0.5), (0.5, 0.5), (0.5, -0.5), (-0.5, -0.5), (-0.5, 0.5)
]

# Define the path codes
brick_codes = [Path.MOVETO] + [Path.LINETO] * (len(brick_vertices) - 1)

brick_path = Path(brick_vertices, brick_codes)


class SquareTilingHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'e'.
    """
    filled = False
    size = 1.0
    path = earth_path

    def __init__(self, hatch, density=1):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('e')) * density)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))

    def set_vertices_and_codes(self, vertices, codes):
        """Set the vertices and codes for the custom hatch.

        Args:
            vertices (ndarray): The array to store the vertices.
            codes (ndarray): The array to store the codes.
        """
        offset = 0.0
        if self.num_rows > 0:
            offset = 1.0 / self.num_rows
        shape_vertices = self.shape_vertices * offset * self.size
        shape_codes = self.shape_codes
        if not self.filled:
            shape_vertices = np.concatenate(  # Forward, then backward.
                [shape_vertices, shape_vertices[::-1] * 0.9])
            shape_codes = np.concatenate([shape_codes, shape_codes])
        vertices_parts = []
        codes_parts = []
        for row in range(self.num_rows + 1):

            cols = np.linspace(0, 1, self.num_rows + 1)

            row_pos = row * offset
            for col_pos in cols:
                vertices_parts.append(shape_vertices + [col_pos, row_pos])
                codes_parts.append(shape_codes)
        np.concatenate(vertices_parts, out=vertices)
        np.concatenate(codes_parts, out=codes)


class SandHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 's'.
    """
    filled = True
    size = 1.0
    path = sand_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('s')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes
        matplotlib.hatch.Shapes.__init__(self, hatch, density)


class ConcreteHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 't'.
    """
    filled = True
    size = 1.0
    path = concrete_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('t')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes
        matplotlib.hatch.Shapes.__init__(self, hatch, density)


class ClayHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'c'.
    """
    filled = False
    size = 1.0
    path = clay_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('c')) * density)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes
        matplotlib.hatch.Shapes.__init__(self, hatch, density)

# class RockHatch(matplotlib.hatch.Shapes):
#     """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

#     Identifier 'r'.
#     """
#     filled = False
#     size = 1.0
#     path = rock_path

#     def __init__(self, hatch, density):
#         """Initialize the custom hatch."""
#         self.num_rows = int((hatch.count('r')) * density / 3)
#         self.shape_vertices = self.path.vertices
#         self.shape_codes = self.path.codes
#         matplotlib.hatch.Shapes.__init__(self, hatch, density)


class RockHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'r'.
    """
    filled = False
    size = 1.0
    path = rock_path

    def __init__(self, hatch, density=1):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('r')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes
        matplotlib.hatch.Shapes.__init__(self, hatch, density)


class EarthHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'e'.
    """
    filled = False
    size = 1.0
    path = earth_path

    def __init__(self, hatch, density=1):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('e')) * density / 2)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class GrassHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'g'.
    """
    filled = False
    size = 1.0
    path = grass_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('g')) * (density))
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes
        matplotlib.hatch.Shapes.__init__(self, hatch, density)


class MetalHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'm'.
    """
    filled = False
    size = 1.0
    path = metal_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('m')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class ZigzagHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'z'.
    """
    filled = False
    size = 1.0
    path = zigzag_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('z')) * density / 6)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class HoneycombHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'h'.
    """
    filled = False
    size = 1.0
    path = honeycomb_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('h')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class WoodHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'd'.
    """
    filled = False
    size = 1.0
    path = wood_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('d')) * density / 6)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class WaterHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'w'.
    """
    filled = False
    size = 1.0
    path = water_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('w')) * density / 6)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class BituminousHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'u'.
    """
    filled = False
    size = 1.0
    path = bituminous_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('u')) * density / 2)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class PeatHatch(SquareTilingHatch):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier '~'.
    """
    filled = True
    size = 1.0
    path = peat_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('~')) * density / 6)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        self.num_shapes = (self.num_rows + 1) ** 2
        self.num_vertices = (self.num_shapes * len(self.shape_vertices) * (1 if self.filled else 2))


class MetamorphicHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'a'.
    """
    filled = False
    size = 1.0
    path = metamorphic_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('a')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        matplotlib.hatch.Shapes.__init__(self, hatch, density)


class BrickHatch(matplotlib.hatch.Shapes):
    """Custom hatches defined by a path drawn inside [-0.5, 0.5] square.

    Identifier 'b'.
    """
    filled = False
    size = 1.0
    path = brick_path

    def __init__(self, hatch, density):
        """Initialize the custom hatch."""
        self.num_rows = int((hatch.count('b')) * density / 3)
        self.shape_vertices = self.path.vertices
        self.shape_codes = self.path.codes

        matplotlib.hatch.Shapes.__init__(self, hatch, density)


# Used identifiers: *+-./OX\ox| ~, a, b, c, d, e, g, h, m, p, r, s, t, u, w, z
