"""Weir Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import abstractmethod
import copy
import sys
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.variables.variable import Variable
from xms.FhwaVariable.interface_adapters.view_model.main.tree_data import TreeData


class VariableGroup:
    """A base class that defines the basics of a defined group of variables."""

    @staticmethod
    def get_image_filters() -> list:
        """Returns a string formated for specifying the filters in an open dialog with supported image types."""
        image_formats = [
            "All files (*)",
            "All image files (*.bmp *.dib *.jpeg *.jpg *.jpe *.jp2 *.png *.webp *.avif *.pbm *.pgm *.ppm "
            "*.pxm *.pnm *.pfm *.sr *.ras *.tiff *.tif *.exr *.hdr *.pic *.gif *.asc)",
            "Windows bitmaps (*.bmp *.dib)",
            "JPEG files (*.jpeg *.jpg *.jpe)",
            "JPEG 2000 files (*.jp2)",
            "Portable Network Graphics (*.png)",
            "WebP (*.webp)",
            "AVIF (*.avif)",
            "Portable image format (*.pbm *.pgm *.ppm *.pxm *.pnm)",
            "PFM files (*.pfm)",
            "Sun rasters (*.sr *.ras)",
            "TIFF files (*.tiff *.tif)",
            "OpenEXR Image files (*.exr)",
            "Radiance HDR (*.hdr *.pic)",
            # "Raster and Vector geospatial data supported by GDAL"
            "GIF (*.gif)",
            "Arc/Info ASCII Grid (*.asc)",
        ]

        # filter_string = ";;".join(image_formats)

        return image_formats

    image_filters = get_image_filters()

    def __init__(self, input: list = None, app_data=None, model_name: str = None, project_uuid: str = None
                 ) -> None:
        """Initializes the CalcData.

        Args:
            input (list of variable): the input variables
            app_data (AppData): the application data
            model_name (string): the model name
            project_uuid (string): the project uuid
        """
        self.max_value = sys.float_info.max

        self.name = 'variable group'  # changed by user
        self.type = 'var_group'  # Used only in code

        self.uuid = uuid.uuid4()
        self.icon = None

        self.can_compute = False  # True if the calculator can compute
        self.calculator = None  # Plug-in for the calculator
        # If the calc is standalone, it will clear its own results
        # If the calc is a child, the parent will likely clear the results
        self.clear_my_own_results = True

        self.update_inpute_after_compute = False

        self.app_data = app_data
        self.model_name = model_name
        self.project_uuid = project_uuid

        # Input
        # Make Input a dict
        self.input: dict[str, Variable] = {}
        if input is not None:
            self.input = input

        # TODO: Implement that changing this setting affects all children
        # In particular, the profiles settings should be able to be set to read only and change all input variables
        self.read_only = False
        self.help_url = None  # URL to help the user understand the variable

        self.tree_data = TreeData()

        self.complexity: int = 0

        # Intermediate
        self.compute_prep_functions = []  # Functions to call before compute_data
        self.intermediate_to_copy = []  # Define variables to copy to the calculator
        self.compute_finalize_functions = []  # Functions to call after compute_data

        self.warnings: list[str] = []

        # Results
        self.results: dict[str, Variable] = {}

        self.plot_dict = {}

        # Shorthand for units:
        self.us_short_length = [['ft', 'in', 'mm']]
        self.si_short_length = [['m', 'mm']]

        self.us_mid_length = [['yd', 'ft', 'in']]
        self.si_mid_length = [['m', 'mm']]

        self.us_long_length = [['mi', 'yd', 'ft']]
        self.si_long_length = [['km', 'm']]

        self.us_slope = [['ft/ft', '%']]
        self.si_slope = [['m/m', '%']]

        self.us_velocity = [['ft/s', 'mi/hr']]
        self.si_velocity = [['m/s', 'km/hr']]

        self.us_flow = [['cfs', 'gpm']]
        self.si_flow = [['cms', 'lpm']]

        self.us_temperature = [['°F', 'K']]
        self.si_temperature = [['°C', 'K']]

        # # Get the supported image formats from Pillow
        # supported_formats = [f".{format.lower()}" for format in Image.registered_extensions()]
        # supported_formats_string = " ".join(supported_formats)
        # self.image_filters = 'All Files (*);;Images (' + supported_formats_string + ")"

        self.uuid = uuid.uuid4()

    def initialize(self) -> None:
        """Initializes the CalcData and variables."""
        if hasattr(self, 'check_list_length'):
            self.check_list_length()
        self._initialize_dict(self.input)
        self._initialize_dict(self.results)
        if hasattr(self, 'item_list'):
            self._initialize_list(self.item_list)

    def _initialize_dict(self, list: dict) -> None:
        """Initializes the CalcData and variables."""
        for key in list:
            if hasattr(list[key], 'check_list_length'):
                list[key].check_list_length()
            if isinstance(list[key], dict):
                self._initialize_dict(list[key])
            elif hasattr(list[key], 'value'):
                if hasattr(list[key].value, 'initialize'):
                    list[key].value.initialize()

    def _initialize_list(self, list: list) -> None:
        """Initializes the CalcData and variables."""
        for item in list:
            if hasattr(item, 'check_list_length'):
                item.check_list_length()
            if isinstance(item, dict):
                self._initialize_dict(item)
            elif hasattr(item, 'value'):
                if hasattr(item.value, 'initialize'):
                    item.value.initialize()

    def get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Returns:
            bool: True if can compute
            dict: A dictionary of warnings if not
        """
        self.warnings = []
        if self.calculator is None:
            return False, {'calculator': 'Calculator is not defined'}
        self.calculator.warnings = {}
        # Run prepare functions and copy intermediate data to the calculator
        self.prepare_for_compute()

        # Create an input dictionary for the calculator
        self.calculator.input_dict, self.calculator.plot_dict = self.prepare_input_dict()

        self.can_compute = self.calculator._get_can_compute()
        self.convert_warning_dict_to_warning_list()
        return self.can_compute, self.warnings

    def set_app_data_model_name_and_project_uuid(self, app_data=None, model_name: str = None,
                                                 project_uuid: str = None) -> None:
        """Initializes the CalcData.

        Args:
            app_data (AppData): the application data
            model_name (string): the model name
            project_uuid (string): the project uuid
        """
        self.app_data = app_data
        self.model_name = model_name
        self.project_uuid = project_uuid

        self._set_app_data_model_name_and_project_uuid_recursive(self.input, app_data, model_name, project_uuid)
        self._set_app_data_model_name_and_project_uuid_recursive(self.results, app_data, model_name, project_uuid)

    def _set_app_data_model_name_and_project_uuid_recursive(self, items, app_data=None,
                                                            model_name: str = None, project_uuid: str = None) -> None:
        """Initializes the CalcData.

        Args:
            items (list of variables): the input variables
            app_data (AppData): the application data
            model_name (string): the model name
            project_uuid (string): the project uuid
        """
        for item in items:
            if isinstance(item, dict):
                self._set_app_data_model_name_and_project_uuid_recursive(item, app_data, model_name, project_uuid)
            elif hasattr(item, 'value'):
                if hasattr(item.value, 'set_app_data_model_name_and_project_uuid'):
                    item.value.set_app_data_model_name_and_project_uuid(app_data, model_name, project_uuid)

    def check_float_vars_to_greater_zero(self, var_list, result=True) -> bool:
        """Check if the variables are greater than 0.

        Args:
            var_list
            result (bool): initial result

        Returns:
            True if the variable is greater than 0, False otherwise
        """
        _, zero_tol = self.get_setting('Zero tolerance')
        for var_name in var_list:
            if var_name in self.input:
                var = self.input[var_name]
                if var.get_val() <= zero_tol:
                    self.warnings.append(f'Please enter {var.name}')
                    result = False

        return result

    def get_input_group(self, unknown: str = None) -> dict:
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        input_vars = copy.deepcopy(self.input)

        return input_vars

    def _get_item_by_name(self, is_input: bool, name: str):
        """Returns the input variable by name.

        Args:
            is_input (bool): True if input, False if results
            name (string): variable name

        Returns:
            variable: input variable
        """
        if is_input:
            dictionary = self.input
        else:
            dictionary = self.results

        result, item = self._get_item_by_name_and_dict(dictionary, is_input, name)

        return result, item

    def _get_item_by_name_and_dict(self, dictionary: dict, is_input: bool, name: str):
        """Returns the input variable by name.

        Args:
            dictionary (dictionary): dictionary of variables
            is_input (bool): True if input, False if results
            name (string): variable name

        Returns:
            variable: input variable
        """
        if name in dictionary:
            return True, dictionary[name]

        for var in dictionary:
            if isinstance(var, dict):
                result, item = self._get_item_by_name_and_dict(dictionary[var], is_input, name)
                if result:
                    return result, item
            if dictionary[var].type in ['class', 'table', 'calc_list']:
                result, item = dictionary[var].get_val()._get_item_by_name(is_input, name)
                if result:
                    return result, item
        return False, None

    def get_results_group(self, unknown: str = None) -> dict:
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if not self.can_compute:
            return result_vars

        return result_vars

    def clear_results(self) -> None:
        """Clears the results and those of subclasses to prepare for computation."""
        if self.clear_my_own_results:
            self.warnings = []

    def prepare_for_compute(self) -> bool:
        """Prepares the data for computation.

        Returns:
            bool: True if successful
        """
        # If there are functions that are not necessary to determine if we can compute, but are necessary to prepare
        # for compute, they should be called here.
        for func in self.compute_prep_functions:
            func()

        # Copy intermediate data to calculator if needed (this can be useful for transferring data between
        # calculators, that we don't want/need to expose to the user).
        if self.calculator:
            self.intermediate_to_copy.append('uuid')
            for attr in self.intermediate_to_copy:
                if hasattr(self, attr):
                    setattr(self.calculator, attr, getattr(self, attr))

        return True

    def finalize_compute(self) -> None:
        """Finalizes the computation."""
        for attr in self.intermediate_to_copy:
            if hasattr(self, attr):
                setattr(self, attr, getattr(self.calculator, attr))

        for func in self.compute_finalize_functions:
            func()

        self.can_compute = self.calculator.can_compute

    def prepare_input_dict(self, include_settings: bool = True, size_limit: int = None,
                           source_input_dict: dict = None) -> tuple[dict, dict]:
        """Prepares the input dictionary for the calculator.

        Args:
            include_settings (bool): True if settings should be included
            size_limit (int): the maximum length of items
            source_input_dict (dict): Used to set app_settings, profiles, and project_settings

        Returns:
            input_dict (dict): the input dictionary
        """
        # Add settings, profile, project settings, and input to the input dictionary
        # That way the calculator will have access to all of them, and the calc can override
        # the settings if needed. More local settings will override more global settings.
        input_dict = {}
        settings_plot_dict = {}
        plot_dict = {}

        if source_input_dict is not None and 'app_settings' in source_input_dict and 'profile' in source_input_dict \
                and 'project_settings' in source_input_dict:
            input_dict = copy.copy(source_input_dict)
        elif include_settings:
            # Add App settings to the input dictionary
            input_dict['app_settings'], settings_plot_dict = self._fill_input_dict_recursive(
                self.app_data.app_settings.input, settings_plot_dict, size_limit)
            # Change complexity to an integer to simplify complexity decisions in the calculator
            sys_complexity = self.get_system_complexity_index()
            if 'Preferences' in input_dict['app_settings'] and 'Complexity' in input_dict['app_settings'][
                    'Preferences']:
                input_dict['app_settings']['Preferences']['Complexity'] = sys_complexity

            # Add profile to the input dictionary
            profile = self.app_data.get_profile()
            input_dict['profile'], settings_plot_dict = self._fill_input_dict_recursive(
                profile.input, settings_plot_dict, size_limit)

            # Add project settings to the input dictionary
            if self.project_uuid is not None:
                result, project_settings = self.app_data.get_project_settings(self.model_name, self.project_uuid)
                if result and project_settings is not None:
                    input_dict['project_settings'], settings_plot_dict = self._fill_input_dict_recursive(
                        project_settings.input, settings_plot_dict, size_limit)

        # Add calcdata input to the input dictionary
        input_dict['calc_data'], plot_dict = self._fill_input_dict_recursive(self.input, plot_dict, size_limit,
                                                                             source_input_dict=input_dict)

        return input_dict, plot_dict

    def compute_data(self, source_input_dict: dict = None) -> tuple[bool, list]:
        """Computes the data.

        Args:
            source_input_dict (dict): Used to set app_settings, profiles, and project_settings

        Returns:
            bool: True if successful
            list: A list of warnings if not
        """
        if self.calculator is None:
            return False, ['No calculator is defined']

        self.clear_results()

        # Run prepare functions and copy intermediate data to the calculator
        self.prepare_for_compute()

        # Create an input dictionary for the calculator
        input_dict, plot_dict = self.prepare_input_dict(source_input_dict=source_input_dict)

        # Perform the calculation, and get the results
        self.can_compute, compute_success, results_dict, warnings, plot_dict = self.calculator.compute_data(
            input_dict, plot_dict)

        # if result:
        self._fill_results_dict_recursive(self.results, results_dict, input_dict)
        if 'Results' in self.results and 'Results' not in results_dict and isinstance(self.results['Results'], dict):
            self._fill_results_dict_recursive(self.results['Results'], results_dict)
        self.handle_results(self.can_compute, compute_success, results_dict, warnings, plot_dict, self.calculator)

        self.plot_dict = plot_dict

        # Correct the warnings to have the name that users will recognize (that was not available in the calculator)
        self.convert_warning_dict_to_warning_list(warnings)

        # Move intermediate and can_compute variables from the calculator
        self.finalize_compute()

        return compute_success, warnings

    def convert_warning_dict_to_warning_list(self, warning_dict: dict = None) -> list:
        """Converts a warning dictionary to a warning list.

        Args:
            warning_dict (dict): the warning dictionary

        Returns:
            list: the warning list
        """
        if warning_dict is None:
            warning_dict = self.calculator.warnings

        for warning_key in warning_dict:
            result, name = self._get_name_from_key_recursive(self.input, warning_key)
            if result:
                warning = warning_dict[warning_key].replace(f"[{warning_key}]", name)
                self.warnings.append(warning)
            else:
                self.warnings.append(warning_dict[warning_key])

        return self.warnings

    @abstractmethod
    def handle_results(self, can_compute: bool, compute_success: bool, results_dict: dict, warnings: list,
                       plot_dict: dict, calculator):
        """Handles the results of the computation.

        Args:
            can_compute (bool): whether the computation can be performed
            compute_success (bool): whether the computation was successful
            results_dict (dict): the results dictionary
            warnings (list): the list of warnings
            plot_dict (dict): the plot dictionary
            calculator (Calculator): the calculator instance
        """
        # This method is provided as an option to override in subclasses, if wanted, to handle results differently.♦
        pass

    def _get_name_from_key_recursive(self, source_dict: dict, key: str) -> tuple[bool, str]:
        """Get the name from the key.

        Args:
            source_dict (dict): the source dictionary
            key (str): the key

        Returns:
            str: the name
        """
        if key in source_dict:
            return True, source_dict[key].name
        for var in source_dict:
            if isinstance(source_dict[var], dict):
                result, name = self._get_name_from_key_recursive(source_dict[var], key)
                if result:
                    return True, name
        return False, None

    def _fill_results_dict_recursive(self, source_dict: dict, results_dict: dict, input_dict: dict = None) -> dict:
        """Fills the results dictionary with the results.

        Args:
            results_dict (dict): the results dictionary

        Returns:
            results_dict (dict): the results dictionary
        """
        for key in results_dict:
            if key in source_dict:
                if isinstance(results_dict[key], dict):
                    changed_dict = self._fill_results_dict_recursive(copy.deepcopy(source_dict[key]),
                                                                     copy.deepcopy(results_dict[key]))
                    source_dict[key] = changed_dict
                elif results_dict[key] is not None:
                    # app_data is intentionally not passed to set_val (no unit conversions)
                    if hasattr(source_dict[key], 'set_val'):
                        source_dict[key].set_val(results_dict[key])
                    else:
                        source_dict[key] = results_dict[key]
            else:
                # key in results_dict but not in source_dict; check if it should be copied from the source_dict
                if isinstance(results_dict[key], dict):
                    for sub_key in results_dict[key]:
                        if sub_key in source_dict:
                            source_dict[sub_key].set_val(results_dict[key][sub_key])
                            if key not in source_dict:
                                source_dict[key] = results_dict[key]
                            source_dict[key][sub_key] = source_dict[sub_key]
                # self._fill_results_dict_recursive(source_dict, results_dict[key])

        if input_dict is not None:
            if self.update_inpute_after_compute and 'calc_data' in input_dict:
                for key in input_dict['calc_data']:
                    if key in self.input and not isinstance(input_dict['calc_data'][key], dict):
                        if (type(self.input[key].value) is type(input_dict['calc_data'][key])) or \
                                self.type in ['list', 'float_list', 'int_list', 'string_list']:
                            self.input[key].set_val(input_dict['calc_data'][key])
            # If the input_dict is provided, copy the input values to the source_dict
            self._check_and_fill_calculators_in_input_recursive(self.input, input_dict['calc_data'], [])

        return source_dict

    def _check_and_fill_calculators_in_input_recursive(self, source_dict: dict, input_dict: dict, key_list: list
                                                       ) -> dict:
        """Checks and fills the calculators in the input dictionary.

        Args:
            source_dict (dict): the source dictionary

        Returns:
            source_dict (dict): the source dictionary with calculators filled
        """
        for key in input_dict:
            if isinstance(input_dict[key], dict):
                new_key_list = copy.deepcopy(key_list)
                new_key_list.append(key)
                self._check_and_fill_calculators_in_input_recursive(source_dict, input_dict[key], new_key_list)
            elif key == 'calculator':
                self.navigate_and_modify(source_dict, key_list, input_dict[key])
                # keyed_dict = source_dict
                # for sub_key in key_list:
                #     if isinstance(keyed_dict, dict) and sub_key in keyed_dict:
                #         keyed_dict = keyed_dict[sub_key]
                #     elif hasattr(keyed_dict, 'value') and isinstance(keyed_dict.value, dict) and \
                #             sub_key in keyed_dict.value:
                #         keyed_dict = keyed_dict.value[sub_key]
                #     elif hasattr(keyed_dict, 'value') and hasattr(keyed_dict.value, 'input') and \
                #             isinstance(keyed_dict.value.input, dict) and sub_key in keyed_dict.value.input:
                #         keyed_dict = keyed_dict.value.input[sub_key]
                #     elif hasattr(keyed_dict, 'input') and isinstance(keyed_dict.input, dict) and \
                #             sub_key in keyed_dict.input:
                #         keyed_dict = keyed_dict.input[sub_key]
                #     elif hasattr(keyed_dict, 'value') and hasattr(keyed_dict.value, 'item_list') and \
                #             isinstance(keyed_dict.value.item_list, list) and sub_key < len(
                #                 keyed_dict.value.item_list):
                #         keyed_dict = keyed_dict.value.item_list[sub_key]
                #     else:
                #         pass
                # if hasattr(keyed_dict, 'calculator'):
                #     keyed_dict.calculator = input_dict[key]
                # elif hasattr(keyed_dict, 'value') and hasattr(keyed_dict.value, 'calculator'):
                #     keyed_dict.value.calculator = input_dict[key]
                #     keyed_dict.value._fill_results_dict_recursive(keyed_dict.value.results, input_dict[key].results)
                # else:
                #     pass

        return source_dict

    def navigate_and_modify(self, source_dict: dict, key_list: list, modified_calc: dict) -> None:
        """Navigate to a location in nested dict/object structure and modify it.

        Args:
            source_dict (dict): the source dictionary or object
            key_list (list): list of keys to navigate through the structure
            modified_calc: the modified calculator to set at the target location
        """
        if not key_list:
            return

        # Navigate to parent
        self._navigate_dict_recursive(source_dict, key_list, modified_calc)

    def _navigate_dict_recursive(self, source_dict: dict, key_list: list, modified_calc: dict) -> dict:
        """Recursively navigate through a nested dict/object structure to set a modified calculator.

        Args:
            source_dict (dict): the source dictionary or object
            key_list (list): list of keys to navigate through the structure
            modified_calc: the modified calculator to set at the target location
        Returns:
            source_dict (dict): the modified source dictionary or object
        """
        if len(key_list) > 0:
            sub_key = key_list[0]
            if isinstance(source_dict, dict) and sub_key in source_dict:
                self._navigate_dict_recursive(source_dict[sub_key], key_list[1:], modified_calc)
                return source_dict
            elif hasattr(source_dict, 'value') and isinstance(source_dict.value, dict) and sub_key in source_dict.value:
                self._navigate_dict_recursive(source_dict.value[sub_key], key_list[1:], modified_calc)
                return source_dict
            elif hasattr(source_dict, 'value') and hasattr(source_dict.value, 'input') and \
                    isinstance(source_dict.value.input, dict) and sub_key in source_dict.value.input:
                self._navigate_dict_recursive(source_dict.value.input[sub_key], key_list[1:], modified_calc)
                return source_dict
            elif hasattr(source_dict, 'input') and isinstance(source_dict.input, dict) and sub_key in source_dict.input:
                self._navigate_dict_recursive(source_dict.input[sub_key], key_list[1:], modified_calc)
                return source_dict
            elif hasattr(source_dict, 'value') and hasattr(source_dict.value, 'item_list') and \
                    isinstance(source_dict.value.item_list, list) and sub_key < len(source_dict.value.item_list):
                self._navigate_dict_recursive(source_dict.value.item_list[sub_key], key_list[1:], modified_calc)
                return source_dict
            else:
                return None  # Can't navigate further

        if hasattr(source_dict, 'calculator'):
            source_dict.calculator = modified_calc
            source_dict.finalize_compute()
            return source_dict
        elif hasattr(source_dict, 'value') and hasattr(source_dict.value, 'calculator'):
            source_dict.value.calculator = modified_calc
            source_dict.value._fill_results_dict_recursive(source_dict.value.results, modified_calc.results)
            modified_input_calc = modified_calc.input_dict
            if 'calc_data' in modified_input_calc:
                modified_input_calc = modified_calc.input_dict['calc_data']
            for input_key in modified_input_calc:
                if input_key in source_dict.value.input and hasattr(source_dict.value.input[input_key], 'type') and \
                        source_dict.value.input[input_key].type in [
                            'int', 'float', 'str', 'bool', 'list']:
                    source_dict.value.input[input_key].set_val(modified_input_calc[input_key])
            source_dict.value.finalize_compute()
            return source_dict

        return None

    def _fill_input_dict_recursive(self, source_dict: dict, plot_dict: dict = None, size_limit: int = None,
                                   source_input_dict: dict = None) -> tuple[dict, dict]:
        """Fills the input dictionary with the input variables.

        Args:
            input_dict (dict): the input dictionary
            plot_dict (dict): the plot dictionary
            size_limit (int): the maximum length of items
            source_input_dict (dict): Used to set app_settings, profiles, and project_settings

        Returns:
            input_dict (dict): the input dictionary
        """
        input_dict = {}
        cur_num_items = None
        if 'Table options' in source_dict and 'Data input' in source_dict:
            cur_num_items = source_dict['Table options'].value.input['Number of items'].value

        for key in source_dict:
            if isinstance(source_dict[key], dict):
                if key in ['Plot options']:  # and hasattr(source_dict[key], 'get_plot_dict'):
                    if plot_dict is None:
                        plot_dict = {}
                    for plot_name in source_dict[key]:
                        plot_dict[plot_name] = source_dict[key][plot_name].value.get_plot_options_dict()
                else:
                    input_dict[key], plot_dict = self._fill_input_dict_recursive(
                        source_dict[key], plot_dict, size_limit=size_limit, source_input_dict=source_input_dict)
            elif source_dict[key] is not None:
                if source_dict[key].type in ['calc', 'class', 'PersonalDetails',]:
                    size_limit = None
                    if key == 'Data input' and cur_num_items is not None:
                        size_limit = cur_num_items
                    new_input, new_plot = source_dict[key].value.prepare_input_dict(
                        False, size_limit=size_limit, source_input_dict=source_input_dict)
                    if new_input is not None and 'calc_data' in new_input:
                        input_dict[key] = new_input['calc_data']
                    if new_plot is not None and len(new_plot) > 0:
                        if plot_dict is None:
                            plot_dict = {}
                        # plot_dict.update(new_plot)
                        plot_dict[key] = new_plot
                    if hasattr(source_dict[key].value, 'calculator') and source_dict[key].value.calculator:
                        source_dict[key].value.prepare_for_compute()
                        input_dict[key]['calculator'] = source_dict[key].value.calculator
                        input_dict[key]['calculator'].input_dict = new_input
                        input_dict[key]['calculator'].plot_dict = new_plot
                elif source_dict[key].type in ['table']:
                    if self.app_data is not None and self.app_data.app_settings is not None:
                        source_dict[key].value.compute_data(source_input_dict=source_input_dict)
                    num_items = source_dict[key].value.input['Table options'].value.input['Number of items'].value
                    input_dict[key], plot_dict = self._fill_input_dict_recursive(
                        source_dict[key].value.input['Data input'].value.input, plot_dict, size_limit=num_items)
                    if 'Plot options' in source_dict[key].value.input:
                        if plot_dict is None:
                            plot_dict = {}
                        for plot_name in source_dict[key].value.input['Plot options']:
                            plot_dict[plot_name] = source_dict[key].value.input['Plot options'][
                                plot_name].value.get_plot_options_dict()
                    if hasattr(source_dict[key].value, 'plot_dict') and key in source_dict[key].value.plot_dict:
                        plot_dict[key] = source_dict[key].value.plot_dict[key]
                    if hasattr(source_dict[key].value, 'calculator'):
                        # The calculator's input_dict should already be set in the compute_data call above
                        input_dict[key]['calculator'] = source_dict[key].value.calculator
                elif source_dict[key].type in ['calc_list']:
                    input_dict[key] = {}
                    if source_dict[key].value.select_one is True:
                        index = source_dict[key].value.input['Selected item'].value
                        # item = source_dict[key].value.item_list[index]
                        # new_input, plot_dict = self._fill_input_dict_recursive(item.input, plot_dict,
                        #                                                        size_limit=size_limit)
                        # if new_input is not None:
                        input_dict[key]['Selected item'] = index
                        # if new_plot is not None and len(new_plot) > 0:
                        #     if plot_dict is None:
                        #         plot_dict = {}
                        #     plot_dict.update(new_plot)
                        # if hasattr(source_dict[key].value.item_list[index], 'calculator') and \
                        #         source_dict[key].value.item_list[index].calculator:
                        #     source_dict[key].value.prepare_for_compute()
                        #     input_dict[key]['calculator'] = source_dict[key].value.item_list[index].calculator

                    # Add all items in the calc list
                    index = 0

                    num_items = source_dict[key].value.input['Number of items'].value
                    # input_dict[key]['Number of items'] = num_items  # Putting this in, breaks existing for loops
                    for item in source_dict[key].value.item_list[:num_items]:
                        new_input, new_plot = item.prepare_input_dict(False, source_input_dict=source_input_dict)
                        if new_input:
                            input_dict[key][index] = new_input['calc_data']
                        if 'Name' not in input_dict[key][index] and hasattr(item, 'name'):
                            input_dict[key][index]['Name'] = item.name
                        if new_plot is not None and len(new_plot) > 0:
                            if plot_dict is None:
                                plot_dict = {}
                        # plot_dict.update(new_plot)
                        if key not in plot_dict:
                            plot_dict[key] = {}
                        if index not in plot_dict[key]:
                            plot_dict[key][index] = {}
                        plot_dict[key][index] = new_plot
                        if hasattr(source_dict[key].value.item_list[index], 'calculator') and source_dict[
                                key].value.item_list[index].calculator:
                            item.prepare_for_compute()
                            input_dict[key][index]['calculator'] = \
                                item.calculator
                        index += 1
                        if index >= num_items:
                            break
                elif source_dict[key].type in ['UserArray']:
                    input_dict[key] = source_dict[key].get_val().get_result()
                else:
                    if size_limit is None:
                        input_dict[key] = source_dict[key].get_val()
                    else:
                        val = source_dict[key].get_val()
                        input_dict[key] = val[:size_limit] if isinstance(val, list) else val

        return input_dict, plot_dict

    def find_var_by_uuid(self, uuid: str) -> tuple[bool, any]:
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            variable: the variable
        """
        for var in self.input:
            result, item = self.input[var].find_var_by_uuid(uuid)
            if result:
                return result, item

        for var in self.results:
            result, item = self.input[var].find_var_by_uuid(uuid)
            if result:
                return result, item

        return False, None

    def find_item_by_uuid(self, uuid: str, uuid_index: int = None) -> tuple[bool, any]:
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        return self._find_item_by_uuid_recursive(uuid, uuid_index=uuid_index)

    def _find_item_by_uuid_recursive(self, var_uuid: str, uuid_index: int = None) -> tuple[bool, any]:
        """Finds a variable by its uuid.

        Args:
            var_uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        if self.uuid == var_uuid:
            if uuid_index is not None and hasattr(self, 'item_list') and len(self.item_list) > uuid_index:
                return True, self.item_list[uuid_index]
            return True, self
        if hasattr(self, 'input'):
            for _, var in self.input.items():
                result, new_item = self._check_item_for_uuid_recursive(var, var_uuid, uuid_index)
                if result:
                    return result, new_item
        if hasattr(self, 'item_list'):
            for var in self.item_list:
                result, new_item = self._check_item_for_uuid_recursive(var, var_uuid, uuid_index)
                if result:
                    return result, new_item
        if hasattr(self, 'results'):
            for _, var in self.results.items():
                result, new_item = self._check_item_for_uuid_recursive(var, var_uuid, uuid_index)
                if result:
                    return result, new_item
        return False, None

    def _check_item_for_uuid_recursive(self, item, var_uuid: str, uuid_index: int = None) -> tuple[bool, any]:
        """Check if the item has the uuid.

        Args:
            item: the item to check
            var_uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        if item is None:
            return False, None
        if isinstance(item, dict):  # Check and handle dictionaries
            for _, sub_item in item.items():
                result, new_item = self._check_item_for_uuid_recursive(sub_item, var_uuid, uuid_index)
                if result:
                    return result, new_item
            return False, None
        if hasattr(item, 'uuid'):  # Check if there is a value and if it can check children/groups
            if item.uuid == var_uuid:  # Check if is the item
                return True, item
        if hasattr(item, 'value'):  # Check if there is a value and if it can check children/groups
            if hasattr(item.value, '_find_item_by_uuid_recursive'):
                result, new_item = item.value._find_item_by_uuid_recursive(var_uuid, uuid_index)
                if result:
                    return result, new_item
        if hasattr(item, '_find_item_by_uuid_recursive'):  # Check if it can check children/groups
            result, new_item = item._find_item_by_uuid_recursive(var_uuid, uuid_index)
            if result:
                return result, new_item
        return False, None

    def get_setting_var(self, name: str, model_name: str = None, project_uuid: str = None, skip_locations: list = None
                        ) -> tuple[bool, any, str]:
        """Returns the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            model (string): model name

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        if model_name is None and self.model_name is not None:
            model_name = self.model_name
        if project_uuid is None and self.project_uuid is not None:
            project_uuid = self.project_uuid
        if self.app_data:
            return self.app_data.get_setting_var(name=name, model_name=model_name, project_uuid=project_uuid,
                                                 skip_locations=skip_locations)
        return False, None, None

    def get_setting(self, name: str, default: any = None, model_name: str = None, project_uuid: str = None) -> tuple:
        """Returns a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            default (varies): if the setting isn't found, this is returned
            model_name (string): model name
            project_uuid (string): project name

        Returns:
            if the setting was successfully found
            value (varies): Value of the setting or return_in_case_of_failure
        """
        result, var, _ = self.get_setting_var(name, model_name, project_uuid)
        if result:
            return result, var.get_val()
        return result, default

    def set_setting(self, name: str, new_value: any, model_name: str = None, project_uuid: str = None) -> bool:
        """Changes a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            new_value (varies): new value to set
            model (string): model name

        Returns:
            if the setting was successfully found
        """
        if model_name is None and self.model_name is not None:
            model_name = self.model_name
        if project_uuid is None and self.project_uuid is not None:
            project_uuid = self.project_uuid
        set_results = False
        not_complete = True
        skip_locations = []
        while not_complete:
            result, var, location = self.get_setting_var(name, model_name, project_uuid, skip_locations)
            if result:
                set_results = True
                var.set_val(new_value)
                if location == 'project_settings':
                    not_complete = False
                else:
                    skip_locations.append(location)
            else:
                not_complete = False
        return set_results

    def get_theme(self):
        """Get the theme for the application.

        Returns:
            string: the theme
        """
        if self.app_data:
            return self.app_data.get_theme()
        return None

    def get_system_complexity_index(self) -> int:
        """Get the complexity level index.

        Returns:
            int: the complexity level index
        """
        result = False
        if self.app_data:
            # Don't use self.get_setting_var because it maybe overridden in SettingGroup items
            result, sys_complexity_var, _ = self.app_data.get_setting_var(
                name='Complexity', model_name=self.model_name, project_uuid=self.project_uuid)
        sys_complexity = 0
        if result:
            sys_complexity = sys_complexity_var.value
        return sys_complexity
