"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.variables.variable import Variable
# from xms.FhwaVariable.core_data.calculator.tree_data import TreeData
# from xms.FhwaVariable.core_data.calculator.metadata import Metadata
from xms.FhwaVariable.interface_adapters.view_model.main.tree_model import FolderItem, Ugrid


class Project(FolderItem):
    """Class to hold project data."""

    def __init__(self, model_name=None, project_name=None, icon=None, tool_tip=None):
        """Initializes the Project class.

        Args:
            model_name (string): The name of the model
            project_name (string): The name of the project
            parent (FolderItem): The parent folder
            icon (string): The icon for the project
            tool_tip (string): The tool tip for the project
        """
        self.project_settings = AppData().get_default_project_settings_by_name(model_name)
        super().__init__(name=project_name, class_name='Project', item_class='Project', icon=icon, tool_tip=tool_tip)
        self.name = project_name
        self.is_drag_enabled = False
        self.is_drop_enabled = True
        self.current_file = None

        # Create geometries, coverages, and GIS data lists
        if AppData.geometry_enabled:
            self.children.append(FolderItem(
                name='Geometries', class_name='Geometry Folder', tool_tip='Project folder for geometric objects',
                parent_uuid=self.uuid, project_uuid=self.uuid))
        if AppData.coverages_enabled:
            self.children.append(FolderItem(
                name='Coverages', class_name='Coverage Folder', tool_tip='Project folder for coverage layers',
                parent_uuid=self.uuid, project_uuid=self.uuid))

        images_name = AppData.image_folder_name_plural
        image_name = AppData.image_folder_name_singular
        self.children.append(FolderItem(
            name=images_name, class_name='GIS Folder', tool_tip=f'Project folder for {image_name} data',
            parent_uuid=self.uuid, project_uuid=self.uuid))

        self.children.append(FolderItem(
            name='Models', class_name='Model Folder', tool_tip='Project folder for Model data',
            parent_uuid=self.uuid, project_uuid=self.uuid))

    def get_geometry_folder(self):
        """Returns the geometry folder."""
        for child in self.children:
            if child.class_name == 'Geometry Folder':
                return child
        return None

    def get_coverage_folder(self):
        """Returns the coverage folder."""
        for child in self.children:
            if child.class_name == 'Coverage Folder':
                return child
        return None

    def get_gis_folder(self):
        """Returns the GIS folder."""
        for child in self.children:
            if child.class_name == 'GIS Folder':
                return child
        return None

    def get_model_folder(self):
        """Returns the model folder."""
        for child in self.children:
            if child.class_name == 'Model Folder':
                return child
        return None


class ProjectsManager():
    """Class to manage the CalcData data."""

    def __init__(self, model_name, model_icon, project_icon):
        """Initialize the ProjectManager class.

        Args:
            model_name (string): The name of the model
            model_icon (string): The icon for the model
            project_icon (string): The icon for the project
        """
        self.model_name = model_name
        self.model_icon = model_icon
        self.project_icon = project_icon

        # Data
        # This does not work with project data in each model!
        # geometries = []  # UGrids
        # coverages = []  # Coverages
        # gis_data = []  # GIS, images, rasters, etc.
        self.projects_list = []  # Simulations/Calculators

        project = self.add_project()

        # self.uuid_of_selected_item = None
        self.delete_dict = {}

        self.first_project_uuid = project.uuid

    def add_project(self, project_name=None):
        """Adds a project to the project list.

        Args:
            project_name (string): The name of the project
        """
        if project_name is None:
            if len(self.projects_list) > 0:
                project_name = f'New project {len(self.projects_list)}'
            else:
                project_name = 'New project'

        self.set_all_items_selected(False)
        project = Project(self.model_name, project_name, icon=self.project_icon, tool_tip='Project')
        project.is_selected = True
        self.projects_list.append(project)
        return project

    def add_or_update_project(self, project=None, data=None):
        """Adds a project to the project list.

        Args:
            project (Project): The project
            data (dict): The data to add to the project
        """
        self.set_all_items_selected(False)
        if project is None:
            if data is not None and self.projects_list and len(self.projects_list) == 1:
                project = self.projects_list[0]
            else:
                project = self.add_project()

        project.is_selected = True
        is_added = False
        for proj in self.projects_list:
            if proj.uuid == project.uuid:
                proj = project
                is_added = True
                break
        if not is_added:
            self.projects_list.append(project)

        if data is not None:
            if not isinstance(data, dict):
                raise TypeError("Data must be a dictionary")
            if 'ugrid' in data:
                ugrid_data = data.get('ugrid')
                filename = data.get('ugrid filename')
                geom_folder = project.get_geometry_folder()
                ugrid = Ugrid(name=filename.stem, item_class='Ugrid',
                              parent_uuid=geom_folder.uuid, project_uuid=geom_folder.uuid)
                ugrid.ugrid_data = ugrid_data
                ugrid.filename = filename
                geom_folder.children.append(ugrid)
        return project

    # def add_project_folder(self, new_project_folder):
    #     """Adds a project folder to the project.

    #     Args:
    #         new_project_folder (FolderItem): The new project folder to add to the project
    #     """
    #     self.project_folder.name = new_project_folder.name
    #     self.project_folder.uuid = new_project_folder.uuid
    #     self.project_folder.is_expanded = new_project_folder.is_expanded

    #     for child in new_project_folder.children:
    #         self.project_folder.children.append(child)
    #         child.parent = self.project_folder

    def close_project(self, project_uuid=None):
        """Adds a project to the project list."""
        if project_uuid is None:
            folder_items = self.get_selected_folder_items()
            if folder_items and len(folder_items) > 0:
                for folder_item in folder_items:
                    if folder_item.class_name == 'Project':
                        project_uuid = folder_item.uuid
                        break

        if project_uuid is not None:
            project_uuid = self.projects_list[0].uuid

        if project_uuid is not None:
            projects_list, _ = self.get_items_with_uuids([project_uuid])
            if projects_list and len(projects_list) > 0:
                self.projects_list.remove(projects_list[0])

                if len(self.projects_list) == 0:
                    self.add_project()
                return True

        return False

    def reset_project_settings_to_defaults(self):
        """Reset the project settings to the defaults."""
        for project in self.projects_list:
            project.project_settings = AppData().get_default_project_settings_by_name(self.model_name)

    def set_current_file_for_project(self, project_uuid, current_file):
        """Sets the current project."""
        for project in self.projects_list:
            if project.uuid == project_uuid:
                project.current_file = current_file
                return True
        return False

    def add_folder_by_uuid(self, location_uuid=None, class_name='Folder'):
        """Adds a folder to the project.

        Args:
            location_uuid (UUID): The unique identifier of the tree item which will have a folder
            class_name (string): The class that will reference to the calcdata_dict
        """
        location_uuid = self.handle_none_location_uuid(location_uuid)
        items, _ = self.get_items_with_uuids([location_uuid])
        parent = items[0]
        if class_name == 'Project':
            return False  # Only allow the main folders to be added to the project
        return self.add_folder(parent=parent, class_name=class_name)

    def add_calcdata(self, calcdata, location_calc):
        """Adds a CalcData to the project.

        Args:
            calcdata (CalcData): The CalcData to add to the project
            location_calc (CalcData): The location to add the CalcData to
            icon (string): The icon for the CalcData
        """
        # CalcData.app_data = AppData()
        # CalcData.model_name = self.model_name
        # location_uuid = self.handle_none_location_uuid(location_uuid)
        # location_calc_list, _ = self.get_items_with_uuids([location_uuid])
        # project_uuid = self.get_project_uuid_of_location_uuid(location_calculator.uuid)
        # CalcData.project_uuid = project_uuid

        # self.set_all_items_selected(False)
        # CalcData.tree_data.is_selected = True
        location = location_calc

        if hasattr(location_calc, 'class_name') and location_calc.class_name == 'Project':
            location = location_calc.get_model_folder()
        elif hasattr(location_calc, 'class_name') and location_calc.class_name in [
                'Geometry Folder', 'Coverage Folder', 'GIS Folder']:
            project = self.get_items_with_uuids([location_calc.project_uuid])[0]
            location = project.get_model_folder()

        location.children.append(calcdata)
        if hasattr(calcdata, 'tree_data'):
            calcdata.tree_data.parent_uuid = location.uuid
            calcdata.tree_data.project_uuid = location.project_uuid
        else:
            calcdata.parent_uuid = location.uuid
            calcdata.project_uuid = location.project_uuid

    def get_project_settings_by_uuid(self, project_uuid):
        """Returns the project settings by the project uuid.

        Args:
            project_uuid (UUID): The unique identifier of the project
        """
        projects_list, _ = self.get_items_with_uuids([project_uuid])
        if projects_list and len(projects_list) > 0:
            return True, projects_list[0].project_settings
        return False, None

    def get_selected_items_uuid_list(self):
        """Returns the is_selected item."""
        selected_items = self.get_selected_items()

        uuid_list = []
        for item in selected_items:
            uuid_list.append(item.uuid)
        return uuid_list

    def handle_none_location_uuid(self, location_uuid):
        """Handle the case where the location_uuid is None.

        Args:
            location_uuid (UUID): The unique identifier of the CalcData which will have a folder
        """
        if location_uuid is None:
            folder_items = self.get_selected_folder_items()
            if folder_items and len(folder_items) > 0:
                location_uuid = folder_items[0].uuid
            if location_uuid is None:
                location_uuid = self.projects_list[0].uuid
        return location_uuid

    def get_selected_folder_items(self):
        """Returns the is_selected item as long as they are a project or folder."""
        selected_items = self.get_selected_items()
        folder_items = []
        for item in selected_items:
            if hasattr(item, 'class_name') and item.class_name == 'Folder' or item.class_name == 'Folder':
                folder_items.append(item)
            if hasattr(item, 'class_name') and item.class_name == 'Project' or item.class_name == 'Project':
                folder_items.append(item)
        return folder_items

    def get_selected_items(self):
        """Returns the is_selected item."""
        selected_items = []
        for project in self.projects_list:
            if hasattr(project, 'tree_data') and project.tree_data.is_selected or \
                    hasattr(project, 'is_selected') and project.is_selected:
                selected_items.append(project)
            if hasattr(project, 'children'):
                for child in project.children:
                    selected_items = self._get_selected_items_recursive(child, selected_items)
        return selected_items

    def _get_selected_items_recursive(self, item, selected_items):
        """Returns the is_selected item."""
        if hasattr(item, 'tree_data') and item.tree_data.is_selected or \
                hasattr(item, 'is_selected') and item.is_selected:
            selected_items.append(item)
        if hasattr(item, 'children'):
            for child in item.children:
                selected_items = self._get_selected_items_recursive(child, selected_items)
        return selected_items

    def set_selected_items(self, selected_items_uuid_list):
        """Sets the is_selected item.

        Args:
            selected_items_uuid_list (list): List of uuids of selected items
        """
        found_uuids = []
        for project in self.projects_list:
            if hasattr(project, 'tree_data'):
                if project.uuid in selected_items_uuid_list:
                    project.tree_data.is_selected = True
                    found_uuids.append(project.uuid)
                else:
                    project.tree_data.is_selected = False
            else:
                if project.uuid in selected_items_uuid_list:
                    project.is_selected = True
                    found_uuids.append(project.uuid)
                else:
                    project.is_selected = False
            if project.project_settings and project.project_settings.uuid in selected_items_uuid_list:
                project.project_settings.tree_data.is_selected = True
                found_uuids.append(project.project_settings.uuid)
            elif project.project_settings:
                project.project_settings.tree_data.is_selected = False
            if hasattr(project, 'children'):
                for child in project.children:
                    result, found_2 = self._set_selected_items_recursive(child, selected_items_uuid_list)
                    if result:
                        found_uuids.extend(found_2)
        return len(found_uuids) > 0, found_uuids

    def _set_selected_items_recursive(self, item, selected_items_uuid_list):
        """Sets the is_selected item.

        Args:
            item (TreeItem): The item to set as is_selected
            selected_items_uuid_list (list): List of uuids of selected items
        """
        found_uuids = []
        if hasattr(item, 'tree_data'):
            if item.uuid in selected_items_uuid_list:
                found_uuids.append(item.uuid)
                item.tree_data.is_selected = True
            else:
                item.tree_data.is_selected = False
        else:
            if item.uuid in selected_items_uuid_list:
                found_uuids.append(item.uuid)
                item.is_selected = True
            else:
                item.is_selected = False
        if hasattr(item, 'children'):
            for child in item.children:
                result, found_2 = self._set_selected_items_recursive(child, selected_items_uuid_list)
                if result:
                    found_uuids.extend(found_2)

        return len(found_uuids) > 0, found_uuids

    def set_all_items_selected(self, is_selected=True):
        """Sets all items as selected.

        Args:
            is_selected (bool): Whether the item is is_selected
        """
        for project in self.projects_list:
            if hasattr(project, 'tree_data'):
                project.tree_data.is_selected = is_selected
            else:
                project.is_selected = is_selected
            if hasattr(project, 'children'):
                for child in project.children:
                    self._set_all_items_selected_recursive(child, is_selected)

    def _set_all_items_selected_recursive(self, item, is_selected=True):
        """Sets the is_selected item.

        Args:
            item (TreeItem): The item to set as is_selected
            is_selected (bool): Whether the item is is_selected
        """
        if hasattr(item, 'tree_data'):
            item.tree_data.is_selected = is_selected
        else:
            item.is_selected = is_selected
        if hasattr(item, 'children'):
            for child in item.children:
                self._set_all_items_selected_recursive(child, is_selected)

    def set_all_items_not_editing(self):
        """Sets all items as not editing."""
        for project in self.projects_list:
            if hasattr(project, 'tree_data'):
                project.tree_data.is_editing = False
            else:
                project.is_editing = False
            if hasattr(project, 'children'):
                for child in project.children:
                    self._set_all_items_not_editing_recursive(child)

    def _set_all_items_not_editing_recursive(self, item):
        """Sets the is_editing item.

        Args:
            item (TreeItem): The item to set as is_editing
        """
        if hasattr(item, 'tree_data'):
            item.tree_data.is_editing = False
        else:
            item.is_editing = False
        if hasattr(item, 'children'):
            for child in item.children:
                self._set_all_items_selected_recursive(child)

    def invert_selected_items(self):
        """Inverts the is_selected item."""
        for project in self.projects_list:
            if hasattr(project, 'tree_data'):
                project.tree_data.is_selected = not project.tree_data.is_selected
            else:
                project.is_selected = not project.is_selected
            if hasattr(project, 'children'):
                for child in project.children:
                    self._invert_selected_items_recursive(child)

    def _invert_selected_items_recursive(self, item):
        """Inverts the is_selected item."""
        if hasattr(item, 'tree_data'):
            item.tree_data.is_selected = not item.tree_data.is_selected
        else:
            item.is_selected = not item.is_selected
        if hasattr(item, 'children'):
            for child in item.children:
                self._invert_selected_items_recursive(child)

    def get_items_with_uuids(self, uuid_list):
        """Returns the is_selected item."""
        item_list = []
        project_uuid_list = []
        for project in self.projects_list:
            if project.uuid in uuid_list:
                item_list.append(project)
                project_uuid_list.append(project.uuid)
            if project.project_settings and project.project_settings.uuid in uuid_list:
                item_list.append(project.project_settings)
                project_uuid_list.append(project.uuid)
            if hasattr(project, 'children'):
                for child in project.children:
                    item_list, project_uuid_list = self._get_items_with_uuids_recursive(
                        uuid_list, child, item_list, project.uuid, project_uuid_list)
        return item_list, project_uuid_list

    def _get_items_with_uuids_recursive(self, uuid_list, item, item_list, project_uuid, project_uuid_list):
        """Returns the is_selected item.

        Args:
            uuid_list (list): List of uuids of selected items
            item (TreeItem): The item to set as is_selected
            item_list (list): List of items
            project_uuid (UUID): The unique identifier of the project
            project_uuid_list (list): List of project uuids
        """
        if item.uuid in uuid_list:
            item_list.append(item)
            project_uuid_list.append(project_uuid)
        if hasattr(item, 'children'):
            for child in item.children:
                item_list, project_uuid_list = self._get_items_with_uuids_recursive(uuid_list, child, item_list,
                                                                                    project_uuid, project_uuid_list)
        return item_list, project_uuid_list

    # def set_item_as_selected(self, uuid):
    #     """Sets the is_selected item."""
    #     self.uuid_of_selected_item = uuid

    def get_projects_list(self):
        """Returns the projects list."""
        return self.projects_list

    def add_folder(self, parent, class_name='Folder'):
        """Adds a CalcData to the project.

        Args:
            parent (FolderItem): The parent folder to add the new folder to
            class_name (string): The name of the Folder Item
        """
        self.set_all_items_selected(False)
        name = 'New folder'
        if class_name == 'Proposed':
            name = 'New proposed folder'
        elif class_name == 'Existing':
            name = 'New existing folder'
        folder = FolderItem(name, class_name=class_name, is_selected=True,
                            parent_uuid=parent.uuid)
        parent.children.append(folder)
        folder.parent = parent
        return folder

    def extend_calcdatas(self, calcdatas, location_uuid=None):
        """Adds multiple calcdatas to the project.

        Args:
            calcdatas (list): List of calcdatas to add to the project
            location_uuid (UUID): The unique identifier of the CalcData to add the new calcdatas to
        """
        location_uuid = self.handle_none_location_uuid(location_uuid)
        self.set_all_items_selected(False)
        self.project_folder.children.extend(calcdatas)

    def delete_item(self, item_uuid):
        """Deletes an item from the project.

        Args:
            item_uuid (UUID): The unique identifier of the item to delete
        """
        item_list, _ = self.get_items_with_uuids([item_uuid])
        if item_list and len(item_list) > 0:
            delete_item = item_list[0]
            if hasattr(delete_item, 'tree_data'):
                parent_items, _ = self.get_items_with_uuids([delete_item.tree_data.parent_uuid])
            else:
                parent_items, _ = self.get_items_with_uuids([delete_item.parent_uuid])
            if parent_items and len(parent_items) > 0:
                parent_item = parent_items[0]
                child_index = parent_item.children.index(delete_item)
                self.delete_dict[item_uuid] = (delete_item, child_index)
                parent_item.children.pop(child_index)
                return True
        return False

    def move_calc(self, source_uuid, target_uuid):
        """Moves a CalcData from one location to another.

        Args:
            source_uuid (string): The unique identifier of the CalcData to move
            target_uuid (string): The unique identifier of the CalcData to move the CalcData to
        """
        # print(f"Item '{source_uuid}' was dropped onto '{target_uuid}'")
        item_list, _ = self.get_items_with_uuids([source_uuid])
        source_calc = item_list[0]
        item_list, _ = self.get_items_with_uuids([target_uuid])
        target_calc = item_list[0]
        if source_calc and target_calc:
            if hasattr(source_calc, 'tree_data'):
                source_parent_uuid = source_calc.tree_data.parent_uuid
            else:
                source_parent_uuid = source_calc.parent_uuid
            if source_parent_uuid:
                item_list, _ = self.get_items_with_uuids([source_parent_uuid])
                source_parent = item_list[0]
                if source_parent:
                    source_parent.children.remove(source_calc)
                    target_calc.children.append(source_calc)

    def set_val(self, item_uuid, val):
        """Sets the value of a CalcData.

        Args:
            item_uuid (UUID): The unique identifier of the CalcData
            val (various): The value to set

        Returns:
            bool: True if the value was set, False otherwise
            CalcData: The CalcData with the new value
            is_setting: True if the value was a setting, False otherwise
        """
        if isinstance(item_uuid, str):
            item_uuid = uuid.uuid4(item_uuid)
        for project in self.projects_list:
            result, new_item = self._set_val_recursive(project, item_uuid, val)
            if result:
                project = new_item
                return True, new_item, False

            result, new_item = self._set_val_recursive(AppData.profile_var, item_uuid, val)
            if result:
                AppData.profile_var = new_item
                return True, new_item, True

            result, new_item = self._set_val_recursive(AppData.app_settings, item_uuid, val)
            if result:
                AppData.app_settings = new_item
                return True, new_item, True
        return False, None, False

    def _set_val_recursive(self, item, item_uuid, val):
        """Sets the value of a CalcData.

        Args:
            item (CalcData): The CalcData to set the value of
            item_uuid (UUID): The unique identifier of the CalcData
            val (various): The value to set
        """
        if isinstance(item, dict):
            for key, value in item.items():
                result, new_item = self._set_val_recursive(value, item_uuid, val)
                if result:
                    item[key] = new_item
                    return True, item
        elif item.uuid == item_uuid:
            if isinstance(item, CalcData):
                item = val
            elif isinstance(item, Variable):
                item.set_val(val)
            return True, item
        if hasattr(item, 'children'):
            for index, child in enumerate(item.children):
                result, new_item = self._set_val_recursive(child, item_uuid, val)
                if result:
                    item.children[index] = new_item
                    return True, item
        if hasattr(item, 'input'):
            for key, input in item.input.items():
                result, new_item = self._set_val_recursive(input, item_uuid, val)
                if result:
                    item.input[key] = new_item
                    return True, item
        if hasattr(item, 'results'):
            for key, result in item.results.items():
                result, new_item = self._set_val_recursive(result, item_uuid, val)
                if result:
                    item.results[key] = new_item
                    return True, item

        return False, None

    def new(self):
        """Starts a new project."""
        self.__init__(self.model_name, self.model_icon, self.project_icon)
