"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionAreaCalc:
    """Perform area unit conversion computations."""
    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Area
    # ----------------------------------------------------------------------
    # US Units
    square_mile = ['mi^2', 'sq mi', 'square miles', 'Square Miles', 'SQUARE MILES']
    square_yard = ['yd^2', 'sq yd', 'square yards', 'Square Yards', 'SQUARE YARDS']
    square_foot = ['ft^2', 'sq ft', 'square feet', 'Square Feet', 'SQUARE FEET']
    square_inch = ['in^2', 'sq in', 'square inches', 'Square Inches', 'SQUARE INCHES']
    acre = ['acre', 'ac', 'AC', 'ACRE']
    # SI Units
    square_km = ['km^2', 'sq km', 'square kilometers', 'Square Kilometers', 'SQUARE KILOMETERS']
    square_m = ['m^2', 'sq m', 'square meters', 'Square Meters', 'SQUARE METERS']
    square_cm = ['cm^2', 'sq cm', 'square centimeters', 'Square Centimeters', 'SQUARE CENTIMETERS']
    square_mm = ['mm^2', 'sq mm', 'square millimeters', 'Square Millimeters', 'SQUARE MILLIMETERS']
    hectare = ['hectare', 'ha', 'Hectare', 'HECTARE', 'HA']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.square_mile:
            return True, 'km^2'
        elif from_unit in self.square_yard:
            return True, 'm^2'
        elif from_unit in self.square_foot:
            return True, 'm^2'
        elif from_unit in self.square_inch:
            return True, 'mm^2'
        elif from_unit in self.acre:
            return True, 'hectare'
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Area
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert the area unit.

        Args:
            from_unit (string): unit that value is currently in
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.square_mile:
            return self.convert_from_square_mile(to_unit, value)
        elif from_unit in self.square_yard:
            return self.convert_from_square_yard(to_unit, value)
        elif from_unit in self.square_foot:
            return self.convert_from_square_foot(to_unit, value)
        elif from_unit in self.square_inch:
            return self.convert_from_square_inch(to_unit, value)
        elif from_unit in self.acre:
            return self.convert_from_acre(to_unit, value)
        # SI Units
        elif from_unit in self.square_km:
            return self.convert_from_square_km(to_unit, value)
        elif from_unit in self.square_m:
            return self.convert_from_square_m(to_unit, value)
        elif from_unit in self.square_cm:
            return self.convert_from_square_cm(to_unit, value)
        elif from_unit in self.square_mm:
            return self.convert_from_square_mm(to_unit, value)
        elif from_unit in self.hectare:
            return self.convert_from_hectare(to_unit, value)

        return False, value

    def convert_from_square_mile(self, to_unit, value):
        """Convert from the square mile unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value
        elif to_unit in self.square_yard:
            return True, value * 1760.0 ** 2.0
        elif to_unit in self.square_foot:
            return True, value * 5280.0 ** 2.0
        elif to_unit in self.square_inch:
            return True, value * 63360.0 ** 2.0
        elif to_unit in self.acre:
            return True, value * 640.0
        # SI
        elif to_unit in self.square_km:
            return True, value * 2.58999
        elif to_unit in self.square_m:
            return True, value * 2589990.0
        elif to_unit in self.square_cm:
            return True, value * 25899900000.0
        elif to_unit in self.square_mm:
            return True, value * 2589990000000.0
        elif to_unit in self.hectare:
            return True, value * 258.999

        else:
            return False, value

    def convert_from_square_yard(self, to_unit, value):
        """Convert from the square yard unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value / 1760.0 ** 2
        elif to_unit in self.square_yard:
            return True, value
        elif to_unit in self.square_foot:
            return True, value * 3.0 ** 2
        elif to_unit in self.square_inch:
            return True, value * 36.0 ** 2
        elif to_unit in self.acre:
            return True, value / 4840
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.0009144 ** 2
        elif to_unit in self.square_m:
            return True, value * 0.9144 ** 2
        elif to_unit in self.square_cm:
            return True, value * 91.44 ** 2
        elif to_unit in self.square_mm:
            return True, value * 914.4 ** 2
        elif to_unit in self.hectare:
            return True, value / 11959.9

        else:
            return False, value

    def convert_from_square_foot(self, to_unit, value):
        """Convert from the square foot unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.000189394 ** 2
        elif to_unit in self.square_yard:
            return True, value * 0.33333333 ** 2
        elif to_unit in self.square_foot:
            return True, value
        elif to_unit in self.square_inch:
            return True, value * 12.0 ** 2
        elif to_unit in self.acre:
            return True, value / 43560.0
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.0003048 ** 2
        elif to_unit in self.square_m:
            return True, value * 0.3048 ** 2
        elif to_unit in self.square_cm:
            return True, value * 30.48 ** 2
        elif to_unit in self.square_mm:
            return True, value * 304.8 ** 2
        elif to_unit in self.hectare:
            return True, value / 107639.0

        else:
            return False, value

    def convert_from_square_inch(self, to_unit, value):
        """Convert from the square inch unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value / 4014489600
        elif to_unit in self.square_yard:
            return True, value / 1296.0
        elif to_unit in self.square_foot:
            return True, value / 144.0
        elif to_unit in self.square_inch:
            return True, value
        elif to_unit in self.acre:
            return True, value / 6272640.0
        # SI
        elif to_unit in self.square_km:
            return True, value * 2.54e-5 ** 2
        elif to_unit in self.square_m:
            return True, value * 0.0254 ** 2
        elif to_unit in self.square_cm:
            return True, value * 2.54 ** 2
        elif to_unit in self.square_mm:
            return True, value * 25.4 ** 2
        elif to_unit in self.hectare:
            return True, value / 15500016.0

        else:
            return False, value

    def convert_from_acre(self, to_unit, value):
        """Convert from the acre unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.0015625
        elif to_unit in self.square_yard:
            return True, value * 4840
        elif to_unit in self.square_foot:
            return True, value * 43560
        elif to_unit in self.square_inch:
            return True, value * 6.273e+6
        elif to_unit in self.acre:
            return True, value
        # SI
        elif to_unit in self.square_km:
            return True, value / 247.105
        elif to_unit in self.square_m:
            return True, value * 4046.86
        elif to_unit in self.square_cm:
            return True, value * 40468600.0
        elif to_unit in self.square_mm:
            return True, value * 4046860000.0
        elif to_unit in self.hectare:
            return True, value / 2.47105

        else:
            return False, value

    def convert_from_square_km(self, to_unit, value):
        """Convert from the square km unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.621371 ** 2
        elif to_unit in self.square_yard:
            return True, value * 1195989.1737
        elif to_unit in self.square_foot:
            return True, value * 3280.84 ** 2
        elif to_unit in self.square_inch:
            return True, value * 1550001969.12
        elif to_unit in self.acre:
            return True, value * 247.105
        # SI
        elif to_unit in self.square_km:
            return True, value
        elif to_unit in self.square_m:
            return True, value * 1000.0 ** 2
        elif to_unit in self.square_cm:
            return True, value * 100000.0 ** 2
        elif to_unit in self.square_mm:
            return True, value * 1.0e+6 ** 2
        elif to_unit in self.hectare:
            return True, value * 100

        else:
            return False, value

    def convert_from_square_m(self, to_unit, value):
        """Convert from the square m unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.000621371 ** 2
        elif to_unit in self.square_yard:
            return True, value * 1.1959891737
        elif to_unit in self.square_foot:
            return True, value * 3.28084 ** 2
        elif to_unit in self.square_inch:
            return True, value * 1550.00196912
        elif to_unit in self.acre:
            return True, value * 0.000247105
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.001 ** 2
        elif to_unit in self.square_m:
            return True, value
        elif to_unit in self.square_cm:
            return True, value * 100.0 ** 2
        elif to_unit in self.square_mm:
            return True, value * 1000.0 ** 2
        elif to_unit in self.hectare:
            return True, value / 10000

        else:
            return False, value

    def convert_from_square_cm(self, to_unit, value):
        """Convert from the square cm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.00000621371 ** 2
        elif to_unit in self.square_yard:
            return True, value / 8361.27970041
        elif to_unit in self.square_foot:
            return True, value * 0.0328084 ** 2
        elif to_unit in self.square_inch:
            return True, value / 6.45160470711
        elif to_unit in self.acre:
            return True, value * 0.0000000247105
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.00001 ** 2
        elif to_unit in self.square_m:
            return True, value * 0.01 ** 2
        elif to_unit in self.square_cm:
            return True, value
        elif to_unit in self.square_mm:
            return True, value * 10.0 ** 2
        elif to_unit in self.hectare:
            return True, value / 100000000.0

        else:
            return False, value

    def convert_from_square_mm(self, to_unit, value):
        """Convert from the square mm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.000000621371 ** 2
        elif to_unit in self.square_yard:
            return True, value / 836127.970041
        elif to_unit in self.square_foot:
            return True, value * 0.00328084 ** 2
        elif to_unit in self.square_inch:
            return True, value / 645.160470711
        elif to_unit in self.acre:
            return True, value * 0.000000000247105
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.000001 ** 2
        elif to_unit in self.square_m:
            return True, value * 0.001 ** 2
        elif to_unit in self.square_cm:
            return True, value * 0.1 ** 2
        elif to_unit in self.square_mm:
            return True, value
        elif to_unit in self.hectare:
            return True, value / 10000000000.0

        else:
            return False, value

    def convert_from_hectare(self, to_unit, value):
        """Convert from the hectare unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.square_mile:
            return True, value * 0.00386102
        elif to_unit in self.square_yard:
            return True, value * 11959.895552000595
        elif to_unit in self.square_foot:
            return True, value * 107639.05996800534194
        elif to_unit in self.square_inch:
            return True, value * 15500024.635392770171
        elif to_unit in self.acre:
            return True, value * 2.47105
        # SI
        elif to_unit in self.square_km:
            return True, value * 0.01
        elif to_unit in self.square_m:
            return True, value * 10000
        elif to_unit in self.square_cm:
            return True, value * 100000000
        elif to_unit in self.square_mm:
            return True, value * 10000000000

        else:
            return False, value
