"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.units.length import ConversionLengthCalc
from xms.FhwaVariable.core_data.units.time import ConversionTimeCalc


class ConversionVelocityCalc:
    """Test the velocity unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Length
    # ----------------------------------------------------------------------
    # US Units
    mile_per_hour = ['mph', 'mi per hr', 'mi / hr', 'miles / hour',
                     'miles per hour', 'Miles per hour', 'MILES PER HOUR', 'MPH']
    foot_per_second = ["fps", 'ft/s', 'ft per s', 'feet per second', 'Feet per second', 'FEET PER SECOND', 'FPS']
    # SI Units
    km_per_hour = ['kmph', 'kilometer per hour', 'km per hr', 'kilometer / hour',
                   'km / hr', 'Kilometer per hour', 'KILOMETER PER HOUR', 'KMPH']
    m_per_second = ['mps', 'm/s', 'm per s', 'meters per second', 'Meters per second', 'METERS PER SECOND', 'MPS']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.mile_per_hour:
            return True, 'kmph'
        elif from_unit in self.foot_per_second:
            return True, 'm/s'
        return False, ''

    # ----------------------------------------------------------------------
    # Length
    # ----------------------------------------------------------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert velocity conversions.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        result1, from_unit_length, from_unit_time = self.determine_length_and_per_time(from_unit)
        result2, to_unit_length, to_unit_time = self.determine_length_and_per_time(to_unit)

        if result1 and result2:
            length_convert = ConversionLengthCalc()
            time_convert = ConversionTimeCalc()
            result1, value = length_convert.convert_units(from_unit_length, to_unit_length, value)
            result2, value = time_convert.convert_units(from_unit_time, to_unit_time, value, True)
            if result1 and result2:
                return True, value

        return False, value

    def determine_length_and_per_time(self, original_unit):
        """Determine the two component units for the given compound unit.

        Args:
            original_unit (string): the original unit to be broken down into two units
        """
        length_from_unit = ''
        per_time_from_unit = ''
        # US Units
        if original_unit in self.mile_per_hour:
            length_from_unit = 'mile'
            per_time_from_unit = 'hour'
        elif original_unit in self.foot_per_second:
            length_from_unit = 'ft'
            per_time_from_unit = 'second'
        # SI Units
        elif original_unit in self.km_per_hour:
            length_from_unit = 'km'
            per_time_from_unit = 'hour'
        elif original_unit in self.m_per_second:
            length_from_unit = 'm'
            per_time_from_unit = 'second'

        return length_from_unit != '' and per_time_from_unit != '', length_from_unit, per_time_from_unit
