"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import statistics

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation
from xms.HydraulicToolboxCalc.util.intersection import find_intersection_and_closest_index
from xms.HydraulicToolboxCalc.util.list_utils import remove_nans


class ScenarioCalc(Calculator):
    """A class that defines a pier scour at a bridge contraction."""
    scenario_index = None

    @staticmethod
    def is_left_channel_bank_specified(bridge_geometry):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        channel_left_bank_station = bridge_geometry['Channel left bank station']

        return ScenarioCalc._is_left_channel_bank_specified(channel_left_bank_station)

    @staticmethod
    def _is_left_channel_bank_specified(channel_left_bank_station):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        if channel_left_bank_station == 0.0:
            return False
        return True

    @staticmethod
    def is_right_channel_bank_specified(bridge_geometry):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        channel_right_bank_station = bridge_geometry['Channel right bank station']
        return ScenarioCalc._is_right_channel_bank_specified(channel_right_bank_station)

    @staticmethod
    def _is_right_channel_bank_specified(channel_right_bank_station):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        if channel_right_bank_station == 0.0:
            return False
        return True

    @staticmethod
    def determine_if_banks_are_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        channel_left_bank_station = bridge_geometry['Channel left bank station']
        channel_right_bank_station = bridge_geometry['Channel right bank station']

        return ScenarioCalc._determine_if_banks_are_specified(channel_left_bank_station,
                                                              channel_right_bank_station)

    @staticmethod
    def _determine_if_banks_are_specified(channel_left_bank_station, channel_right_bank_station):
        """Determines if the channel banks are specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        if not ScenarioCalc._is_left_channel_bank_specified(channel_left_bank_station) and \
                not ScenarioCalc._is_right_channel_bank_specified(channel_right_bank_station):
            return False
        if channel_left_bank_station > channel_right_bank_station:
            return False
        return True

    @staticmethod
    def is_left_abutments_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        left_abutment_station = bridge_geometry['Left abutment toe station']
        return ScenarioCalc._is_left_abutments_specified(left_abutment_station)

    @staticmethod
    def _is_left_abutments_specified(left_abutment_station):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        if left_abutment_station == 0.0:
            return False
        return True

    @staticmethod
    def is_right_abutments_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        left_abutment_station = bridge_geometry['Left abutment toe station']
        right_abutment_station = bridge_geometry['Right abutment toe station']
        return ScenarioCalc._is_right_abutments_specified(left_abutment_station, right_abutment_station)

    @staticmethod
    def _is_right_abutments_specified(left_abutment_station, right_abutment_station):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        if right_abutment_station == 0:
            return False
        if left_abutment_station > right_abutment_station:
            return False
        return True

    def compute_cross_section(self, plot_dict, bridge_geometry, x_var, y_var, _x_data, _y_data,
                              thalweg_elevation, null_data, zero_tol=None):
        """Computes the cross section.

        Args:
            plot_dict (dict): plot options
            bridge_geometry (BridgeGeometry): the bridge geometry
            x_var (FhwaVariable): the variable for the X values
            y_var (FhwaVariable): the variable for the Y values
            _x_data (list): list of floats for the x-values; Original data, do not modify
            _y_data (list): list of floats for the y-values; Original data, do not modify
            thalweg_elevation (float): the thalweg elevation
            null_data (float): the null data value
            zero_tol (float): the zero tolerance value
        """
        self.plot_dict = plot_dict[self.name]
        applied_ltd = False
        applied_cs = False
        applied_labs = False
        applied_rabs = False
        applied_ps = False
        wse_x = None
        wse_y = None

        pressure_scour_used = False

        if zero_tol is None:
            _, zero_tol = self.get_data('Zero tolerance')
        x_data, y_data, _ = remove_nans(_x_data, _y_data, null_data=null_data)

        # Plot the WSE
        wse_plot_dict = self.get_the_wse(plot_dict)
        # plot_options.set_item_by_name(wse_plot_options)
        if wse_plot_dict:
            _wse_x = wse_plot_dict['x var'].get_val()
            _wse_y = wse_plot_dict['y var'].get_val()
            wse_x, wse_y, _ = remove_nans(_wse_x, _wse_y, null_data=null_data)
            if wse_plot_dict['y var'] and len(wse_y) > 0 and wse_y[0] != 0.0:
                avg_wse = statistics.mean(wse_y)
            self.results['Average WSE'] = avg_wse

        min_scour_elevation = thalweg_elevation

        # apply long-term degradation
        prior_x = copy.deepcopy(x_data)
        prior_y = copy.deepcopy(y_data)
        applied_ltd, x_data, y_data, ltd = self.apply_longterm_degradation(x_data, y_data, bridge_geometry,
                                                                           null_data=null_data, zero_tol=zero_tol)
        if applied_ltd:
            self.add_scour_to_plot_data(self.plot_dict, 'Long-term degradation', x_var, y_var,
                                        x_data, y_data, prior_x=prior_x, prior_y=prior_y, null_data=null_data)
            self.results['Long-term degradation'] = {}
            self.results['Long-term degradation']['Long-term degradation'] = ltd
            self.results['Long-term degradation']['Streambed thalweg elevation'] = thalweg_elevation
            self.results['Long-term degradation']['Minimum channel elevation with LTD'] = min(y_data)

        # apply contraction scour
        prior_x = copy.deepcopy(x_data)
        prior_y = copy.deepcopy(y_data)
        applied_cs, x_data, y_data, left_cs, main_cs, right_cs = self.apply_contraction_scour(
            x_data, y_data, bridge_geometry, ltd)
        if applied_cs:
            self.add_scour_to_plot_data(self.plot_dict, 'Contraction scour', x_var, y_var, x_data,
                                        y_data, prior_x=prior_x, prior_y=prior_y, null_data=null_data)
        cs_types_dict = {'Left overbank contraction scour': left_cs,
                         'Main channel contraction scour': main_cs,
                         'Right overbank contraction scour': right_cs}
        for cs_type in cs_types_dict:
            cs_calc = self.input_dict['calc_data']['Contraction scour'][cs_type]['calculator']
            if cs_calc.can_compute:
                main_cs_results = cs_calc.results
                main_cs_results['Applied contraction scour depth'] = cs_types_dict[cs_type]

                main_cs_results['Clear-water contraction scour depth'] = cs_calc.results[
                    'Clear-water scour depth (ys)']

                main_cs_results['Live-bed contraction scour depth'] = cs_calc.results['Live-bed scour depth (ys)']

                pressure_scour_enabled = self.input_dict['calc_data']['Contraction scour'][cs_type][
                    'Compute pressure flow']
                if pressure_scour_enabled:
                    main_cs_results['Pressure scour depth'] = cs_calc.results['Pressure scour'][
                        'Pressure scour depth (y2)']

                if cs_type == 'Main channel contraction scour':
                    main_cs_results['Contraction scour depth and long-term degradation'] = cs_types_dict[cs_type] + ltd

                    main_cs_results['Streambed thalweg elevation'] = thalweg_elevation
                    main_cs_results['Applied contraction scour elevation with LTD'] = \
                        thalweg_elevation - cs_types_dict[cs_type] - ltd

                if cs_calc.results['Governing scour method'] == 'Pressure flow':
                    pressure_scour_used = True
                # Set results
                if 'Contraction scour' not in self.results:
                    self.results['Contraction scour'] = {}
                if cs_type not in self.results:
                    self.results['Contraction scour'][cs_type] = {}
                self.results['Contraction scour'][cs_type]['Contraction scour method'] = \
                    cs_calc.results['Governing scour method']
                self.results['Contraction scour'][cs_type]['Applied contraction scour depth'] = \
                    main_cs_results['Applied contraction scour depth']
                self.results['Contraction scour'][cs_type]['Clear-water scour depth'] = \
                    main_cs_results['Clear-water contraction scour depth']
                self.results['Contraction scour'][cs_type]['Live-bed scour depth'] = \
                    main_cs_results['Live-bed contraction scour depth']
                if pressure_scour_enabled:
                    self.results['Contraction scour'][cs_type]['Pressure scour depth'] = \
                        main_cs_results['Pressure scour depth']
                if cs_calc.results['Governing scour method'] == 'Cohesive':
                    self.results['Contraction scour'][cs_type]['Cohesive scour depth'] = \
                        main_cs_results['Applied contraction scour depth']

        if 'Contraction scour' in self.results and 'Main channel contraction scour' in self.results[
                'Contraction scour']:
            self.results['Contraction scour']['Main channel contraction scour']['Streambed thalweg elevation'] = \
                thalweg_elevation
            self.results['Contraction scour']['Main channel contraction scour'][
                'Minimum channel elevation with contraction scour and LTD'] = min(y_data)

        # apply pier scour
        prior_x = copy.deepcopy(x_data)
        prior_y = copy.deepcopy(y_data)
        is_applied, x_data, y_data, _, _ = self.apply_pier_scour(x_data, y_data, bridge_geometry,
                                                                 _x_data, _y_data, wse_x, wse_y)
        if is_applied:
            applied_ps = True
            self.add_scour_to_plot_data(plot_dict, 'Pier scour', x_var, y_var, x_data,
                                        y_data, prior_x=prior_x, prior_y=prior_y, null_data=null_data)
            for pier_index in self.input_dict['calc_data']['Pier scour']:
                pier = self.input_dict['calc_data']['Pier scour'][pier_index]['calculator']
                dict_name = self.input_dict['calc_data']['Pier scour'][pier_index]['Name']
                if not pier.can_compute:
                    continue

                if 'Pier scour' not in self.results:
                    self.results['Pier scour'] = {}
                if dict_name not in self.results['Pier scour']:
                    self.results['Pier scour'][dict_name] = {}
                self.results['Pier scour'][dict_name]['Pier computation method'] = pier.input_dict['calc_data'][
                    'Computation method']
                self.results['Pier scour'][dict_name]['Pier scour depth'] = pier.scour_depth
                if pier.max_flow_depth is not None:
                    self.results['Pier scour'][dict_name]['Max flow depth including pier scour'] = pier.max_flow_depth
                self.results['Pier scour'][dict_name]['Total scour at pier'] = pier.total_scour_depth
                self.results['Pier scour'][dict_name]['Scour reference point for pier'] = pier.scour_reference_point
                if pier.scour_reference_point == 'Thalweg':
                    self.results['Pier scour'][dict_name]['Streambed thalweg elevation'] = thalweg_elevation
                self.results['Pier scour'][dict_name]['Streambed elevation at pier'] = \
                    pier.streambed_elev_prior_to_scour
                self.results['Pier scour'][dict_name]['Total scour elevation at pier'] = pier.total_scour_elevation

        # apply abutment scour
        if not pressure_scour_used:
            left_or_right = 'Left'
            self.apply_data_to_abutment(left_or_right, x_data, y_data,
                                        bridge_geometry['Left abutment toe station'], _x_data, _y_data,
                                        wse_x=wse_x, wse_y=wse_y, channel_cs=main_cs, overbank_cs=left_cs,
                                        ltd=ltd, null_data=null_data, zero_tol=zero_tol)

            left_or_right = 'Right'
            self.apply_data_to_abutment(left_or_right, x_data, y_data,
                                        bridge_geometry['Right abutment toe station'], _x_data, _y_data,
                                        wse_x=wse_x, wse_y=wse_y, channel_cs=main_cs, overbank_cs=right_cs, ltd=ltd,
                                        null_data=null_data, zero_tol=zero_tol)

            if ScenarioCalc.is_left_abutments_specified(bridge_geometry):
                self.apply_abutment_shear_decay_data(
                    bridge_geometry['Left abutment toe station'],
                    self.input_dict['calc_data']['Abutment scour']['Left abutment scour']['calculator'],
                    bridge_geometry)
                can_compute, warnings = \
                    self.input_dict['calc_data']['Abutment scour']['Left abutment scour']['calculator'].\
                    get_can_compute_with_subdict(
                        self.input_dict, self.input_dict['calc_data']['Abutment scour']['Left abutment scour'])
                if can_compute:
                    applied_labs = True
                    prior_x = copy.deepcopy(x_data)
                    prior_y = copy.deepcopy(y_data)
                    abutment = self.input_dict['calc_data']['Abutment scour']['Left abutment scour']['calculator']
                    is_applied, x_data, y_data, _, _, streambed_elev, left_scour_elev = self.apply_abutment_scour(
                        x_data, y_data, bridge_geometry['Left abutment toe station'], abutment, 'Left abutment scour',
                        null_data=null_data, zero_tol=zero_tol)

                    # Get Scour summary report
                    if 'Abutment scour' not in self.results:
                        self.results['Abutment scour'] = {}
                    if 'Left abutment' not in self.results['Abutment scour']:
                        self.results['Abutment scour']['Left abutment'] = {}

                    self.results['Abutment scour']['Left abutment']['Abutment scour depth'] = \
                        self.input_dict['calc_data']['Abutment scour']['Left abutment scour']['calculator'].scour_depth

                    self.results['Abutment scour']['Left abutment']['Max flow depth including abutment scour'] = \
                        self.input_dict['calc_data']['Abutment scour']['Left abutment scour']['calculator'].results[
                            'Max flow depth including scour (ymax)']

                    self.results['Abutment scour']['Left abutment']['Streambed elevation at abutment'] = \
                        streambed_elev

                    self.results['Abutment scour']['Left abutment']['Streambed thalweg elevation'] = thalweg_elevation

                    total_scour = streambed_elev - left_scour_elev
                    self.results['Abutment scour']['Left abutment']['Total scour at abutment'] = total_scour

                    self.results['Abutment scour']['Left abutment']['Total scour elevation at abutment'] = \
                        left_scour_elev

            if ScenarioCalc.is_right_abutments_specified(bridge_geometry):
                self.apply_abutment_shear_decay_data(
                    bridge_geometry['Right abutment toe station'],
                    self.input_dict['calc_data']['Abutment scour']['Right abutment scour']['calculator'],
                    bridge_geometry)

                can_compute, warnings = \
                    self.input_dict['calc_data']['Abutment scour']['Right abutment scour']['calculator']. \
                    get_can_compute_with_subdict(
                        self.input_dict, self.input_dict['calc_data']['Abutment scour']['Right abutment scour'])

                if can_compute:
                    applied_rabs = True
                    # Do not reset the prior; we want to include both abutments in the same dataset
                    if not applied_labs:
                        prior_x = copy.deepcopy(x_data)
                        prior_y = copy.deepcopy(y_data)
                    abutment = self.input_dict['calc_data']['Abutment scour']['Right abutment scour']['calculator']
                    is_applied, x_data, y_data, _, _, streambed_elev, right_scour_elev = self.apply_abutment_scour(
                        x_data, y_data, bridge_geometry['Right abutment toe station'], abutment, 'Right abutment scour',
                        null_data=null_data, zero_tol=zero_tol)

                    # Get Scour summary report
                    if 'Abutment scour' not in self.results:
                        self.results['Abutment scour'] = {}
                    if 'Right abutment' not in self.results['Abutment scour']:
                        self.results['Abutment scour']['Right abutment'] = {}

                    self.results['Abutment scour']['Right abutment']['Abutment scour depth'] = \
                        self.input_dict['calc_data']['Abutment scour']['Right abutment scour']['calculator'].scour_depth

                    self.results['Abutment scour']['Right abutment']['Max flow depth including abutment scour'] = \
                        self.input_dict['calc_data']['Abutment scour']['Right abutment scour']['calculator'].results[
                            'Max flow depth including scour (ymax)']

                    self.results['Abutment scour']['Right abutment']['Streambed elevation at abutment'] = \
                        streambed_elev

                    self.results['Abutment scour']['Right abutment']['Streambed thalweg elevation'] = thalweg_elevation

                    total_scour = streambed_elev - right_scour_elev
                    self.results['Abutment scour']['Right abutment']['Total scour at abutment'] = total_scour

                    self.results['Abutment scour']['Right abutment']['Total scour elevation at abutment'] = \
                        right_scour_elev

            if ScenarioCalc.is_left_abutments_specified(bridge_geometry) and applied_labs or \
                    ScenarioCalc.is_right_abutments_specified(bridge_geometry) and applied_rabs:
                self.add_scour_to_plot_data(plot_dict, 'Abutment scour', x_var, y_var,
                                            x_data, y_data, prior_x=prior_x, prior_y=prior_y, null_data=null_data)
        else:  # No abutment scour because of pressure flow
            self.warnings['Abutment and pressure scour'] = \
                'Abutment scour is not computed, because pressure scour governs. Use an abutment scour ' \
                'countermeasure with pressure flow.'

        if applied_ltd or applied_cs or applied_labs or applied_rabs or applied_ps:
            self.add_scour_to_plot_data(plot_dict, 'Total scour', x_var, y_var, x_data, y_data, null_data=null_data)
            min_scour_elevation = min(y_data)
            self.results['Lowest scour elevation'] = min_scour_elevation

        return applied_ltd, applied_cs, left_cs, main_cs, right_cs, applied_labs, applied_rabs, applied_ps

    def apply_data_to_abutment(self, left_or_right, x_data, y_data, abutment_toe_station, _x_data, _y_data, wse_x,
                               wse_y, channel_cs, overbank_cs, ltd, null_data, zero_tol):
        """Applies the data to the abutment.

        Args:
            left_or_right (string): The abutment that we are working with.
            x_data (list): The x data.
            y_data (list): The y data.
            abutment_toe_station (float): The abutment toe station.
            _x_data (list): list of floats for the x-values; Original data, do not modify
            _y_data (list): list of floats for the y-values; Original data, do not modify
            wse_x (list): The water surface elevation x data.
            wse_y (list): The water surface elevation y data.
            channel_cs (float): The channel contraction scour.
            overbank_cs (float): The overbank contraction scour.
            ltd (float): The long-term degradation.
            null_data (float): The null data value.
            zero_tol (float): The zero tolerance value.
        """
        abutment = self.input_dict['calc_data']['Abutment scour'][f'{left_or_right} abutment scour']['calculator']
        abutment.input_dict = copy.deepcopy(self.input_dict)
        abutment.input_dict['calc_data'] = self.input_dict['calc_data']['Abutment scour'][
            f'{left_or_right} abutment scour']

        orig_streambed_interp = Interpolation(_x_data, _y_data, null_data=null_data, zero_tol=zero_tol)
        streambed_elev_prior_to_scour, _ = orig_streambed_interp.interpolate_y(abutment_toe_station)
        abutment.streambed_elev_prior_to_scour = streambed_elev_prior_to_scour

        abutment.cross_section_x = x_data
        abutment.cross_section_y = y_data
        abutment.abutment_toe_station = abutment_toe_station
        abutment.wse_x = wse_x
        abutment.wse_y = wse_y
        if self.input_dict['calc_data']['Abutment scour'][f'{left_or_right} abutment scour']['Scour location'] == \
                'Type a (main channel)':
            abutment.contraction_scour_depth = channel_cs
        else:
            abutment.contraction_scour_depth = overbank_cs
        abutment.ltd = ltd

        post_scour_streambed_interp = Interpolation(x_data, y_data, null_data=null_data, zero_tol=zero_tol)
        streambed_elev_post_scour, _ = post_scour_streambed_interp.interpolate_y(abutment_toe_station)
        abutment.streambed_elev_post_scour = streambed_elev_post_scour

    def add_scour_to_plot_data(self, plot_dict, scour_name, x_var, y_var, x_data, y_data, null_data, prior_x=None,
                               prior_y=None):
        """Adds the scour to the plot data, and determines a closed shape of the scour, if prior x, y passed.

        Args:
            plot_dict (dict): plot options
            scour_name (string): named used as dictionary index
            x_var (FhwaVariable): the variable for the X values
            y_var (FhwaVariable): the variable for the Y values
            x_data (list): list of floats for the x-values
            y_data (list): list of floats for the y-values
            prior_x (list): list of floats for the previous x-values (scour polygon will be between these sets)
            prior_y (list): list of floats for the previous y-values
            null_data (float): the null data value; if None, will get from settings
        """
        # scour_plot_options = plot_options.get_item_by_name(scour_name)
        scour_plot_dict, plot_index = self.get_plot_subdict_and_key_by_name(scour_name, 'series',
                                                                            plot_dict=plot_dict)
        if scour_plot_dict is None:
            scour_plot_dict = {}
            plot_index = 0
        scour_plot_dict['x var'] = copy.deepcopy(x_var)
        scour_plot_dict['y var'] = copy.deepcopy(y_var)

        first_x = null_data
        first_y = null_data

        if prior_x is not None and prior_y is not None:
            cur_size = min(len(x_data), len(y_data))
            prior_size = min(len(prior_x), len(prior_y))
            if cur_size == 0 or prior_size == 0:
                return
            scour_hole_x = []
            scour_hole_y = []
            open_shape = False
            prior_index = 0

            # We need to crawl through the data, following the current data (x_data, y_data)
            # and when they diverge, we follow the current data until we find that it meets back
            # together. When we meet back together, we then add all of the top line back to
            # where they diverged. Put in a null data pair between scour holes.
            for cur_index in range(cur_size):
                if not open_shape:
                    if x_data[cur_index] == prior_x[prior_index] and y_data[cur_index] == prior_y[prior_index]:
                        prior_index += 1
                        continue  # No change on this point
                    open_shape = True
                    if len(scour_hole_x) > 0:
                        scour_hole_x.append(null_data)
                        scour_hole_y.append(null_data)

                    first_x = x_data[cur_index]
                    first_y = y_data[cur_index]
                    if cur_index > 0:
                        first_x = x_data[cur_index - 1]
                        first_y = y_data[cur_index - 1]
                        scour_hole_x.append(x_data[cur_index - 1])
                        scour_hole_y.append(y_data[cur_index - 1])
                    scour_hole_x.append(x_data[cur_index])
                    scour_hole_y.append(y_data[cur_index])
                else:
                    intersection_found = False
                    for p_index_2 in range(prior_index, prior_size):
                        if x_data[cur_index] == prior_x[p_index_2] and y_data[cur_index] == prior_y[p_index_2]:
                            intersection_found = True
                            break
                    if intersection_found:
                        for rev_index in range(p_index_2, prior_index - 2, -1):
                            if prior_size > rev_index >= 0:
                                scour_hole_x.append(prior_x[rev_index])
                                scour_hole_y.append(prior_y[rev_index])
                        prior_index = p_index_2 + 1
                        open_shape = False
                        continue
                    else:
                        scour_hole_x.append(x_data[cur_index])
                        scour_hole_y.append(y_data[cur_index])

            if open_shape:
                for rev_index in range(prior_size - 1, prior_index - 1, -1):
                    if prior_size > rev_index >= 0:
                        scour_hole_x.append(prior_x[rev_index])
                        scour_hole_y.append(prior_y[rev_index])

            if len(scour_hole_x) > 0 and len(scour_hole_y) > 0:
                if first_x != scour_hole_x[-1] or first_y != scour_hole_y[-1]:
                    scour_hole_x.append(first_x)
                    scour_hole_y.append(first_y)
            scour_plot_dict['x var'].set_val(scour_hole_x)
            scour_plot_dict['y var'].set_val(scour_hole_y)

        else:
            scour_plot_dict['x var'].set_val(x_data)
            scour_plot_dict['y var'].set_val(y_data)

        if scour_name in self.plot_dict:
            self.plot_dict[scour_name]['series'][plot_index] = scour_plot_dict
        else:
            self.plot_dict['series'][plot_index] = scour_plot_dict

    def get_the_wse(self, plot_dict):
        """Get the water surface elevation.

        Args:

        Returns:
            dict: The plot data.
        """
        name = 'WSE'
        if name not in plot_dict['Scenarios'][self.scenario_index]:
            return None
        wse_subdict, _ = self.get_plot_subdict_and_key_by_name(name, 'series', plot_dict=plot_dict['Scenarios'][
            self.scenario_index])
        _, orig_key = self.get_plot_subdict_and_key_by_name(name, 'series', plot_dict=plot_dict)
        self.plot_dict['series'][orig_key] = wse_subdict

        if all(value == 0 for value in wse_subdict['x var'].get_val()) and \
                all(value == 0 for value in wse_subdict['y var'].get_val()):
            return None

        return wse_subdict

    def apply_scour_hole(self, x_data, y_data, x_station, top_elevation, scour_hole):
        """Apply the scour hole.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            scour_hole (ScourHoleCalc): The scour hole

        Returns:
            is_applied (bool): whether scour was applied or not
            final_x (list): The x data
            final_y (list): The y data
            scour_hole_x (list): The x data for only the scour hole
            scour_hole_y (list): The y data for only the scour hole
        """
        _, zero_tol = self.get_data('Zero tolerance')
        _, null_data = self.get_data('Null data')
        final_x = []
        final_y = []
        scour_hole_x = []
        scour_hole_y = []

        if 'Top width' not in scour_hole.results or 'Bottom width' not in scour_hole.results:
            return False, x_data, y_data, scour_hole_x, scour_hole_y
        t_width = scour_hole.results['Top width']
        b_width = scour_hole.results['Bottom width']
        depth = scour_hole.input_dict['calc_data']['Scour depth']
        if isinstance(t_width, list):
            if len(t_width) == 0:
                t_width = null_data
            else:
                t_width = t_width[-1]
        if isinstance(b_width, list):
            if len(b_width) == 0:
                b_width = null_data
            else:
                b_width = b_width[-1]

        if depth < zero_tol or t_width < zero_tol or depth == null_data or t_width == null_data:
            return False, x_data, y_data, scour_hole_x, scour_hole_y

        int_x = null_data
        int_y = null_data

        # We will make the sides much larger than necessary (100*), to ensure that we get an intersection

        # 1 is the left intersection of the scour hole and streambed
        # 2 is the left bottom of the scour hole
        # 3 is the right bottom of the scour hole
        # 4 is the right intersection of the scour hole and streambed

        scour_hole_x_1 = x_station - (t_width / 2) - 50.0 * (t_width - b_width)
        scour_hole_y_1 = top_elevation + depth * 100.0

        scour_hole_x_2 = x_station - b_width / 2.0
        scour_hole_y_2 = top_elevation - depth

        scour_hole_x_3 = x_station + b_width / 2.0
        scour_hole_y_3 = top_elevation - depth

        scour_hole_x_4 = x_station + (t_width / 2.0) + 50.0 * (t_width - b_width)
        scour_hole_y_4 = top_elevation + depth * 100.0

        intersection_found = True
        cs_below_sh = True  # Start with cross-section below the scour and alternate with each intersection
        first_int = True

        scour_hole_x = [scour_hole_x_1, scour_hole_x_2, scour_hole_x_3, scour_hole_x_4]
        scour_hole_y = [scour_hole_y_1, scour_hole_y_2, scour_hole_y_3, scour_hole_y_4]
        sh_interp = Interpolation(scour_hole_x, scour_hole_y, null_data=null_data, zero_tol=zero_tol)
        if scour_hole_x_1 < x_data[0]:
            scour_hole_x_1 = x_data[0]
            scour_hole_y_1, _ = sh_interp.interpolate_y(scour_hole_x_1)
            if scour_hole_y_1 < y_data[0]:
                cs_below_sh = False

        if scour_hole_x_4 > x_data[-1]:
            scour_hole_x_4 = x_data[-1]
            scour_hole_y_4, _ = sh_interp.interpolate_y(scour_hole_x_4)

        cur_xy_index = 0
        cur_sh_index = 0
        size_xy = len(x_data)
        size_sh = 4

        scour_hole_x = [scour_hole_x_1, scour_hole_x_2, scour_hole_x_3, scour_hole_x_4]
        scour_hole_y = [scour_hole_y_1, scour_hole_y_2, scour_hole_y_3, scour_hole_y_4]

        while intersection_found:
            intersection_found, int_x, int_y, int_xy_index, int_sh_index = find_intersection_and_closest_index(
                x_data, y_data, scour_hole_x, scour_hole_y, cur_xy_index, cur_sh_index, [[int_x, int_y]], tol=zero_tol)

            if intersection_found:
                if first_int and cs_below_sh:
                    first_int = False
                    if cs_below_sh:
                        scour_hole_x_1 = int_x
                        scour_hole_y_1 = int_y

                if cs_below_sh:
                    cur_sh_index = int_sh_index
                    for cur_xy_index_2 in range(cur_xy_index, size_xy):
                        if x_data[cur_xy_index_2] < int_x:
                            if len(final_x) == 0 or final_x[-1] < x_data[cur_xy_index_2]:
                                final_x.append(x_data[cur_xy_index_2])
                                final_y.append(y_data[cur_xy_index_2])
                        else:
                            cur_xy_index_2 -= 1
                            final_x.append(int_x)
                            final_y.append(int_y)
                            scour_hole_x_4 = int_x
                            scour_hole_y_4 = int_y
                            break
                    cur_xy_index = cur_xy_index_2
                else:
                    cur_xy_index = int_xy_index
                    for cur_sh_index_2 in range(cur_sh_index, size_sh):
                        if scour_hole_x[cur_sh_index_2] < int_x:
                            if len(final_x) == 0 or final_x[-1] < scour_hole_x[cur_sh_index_2]:
                                final_x.append(scour_hole_x[cur_sh_index_2])
                                final_y.append(scour_hole_y[cur_sh_index_2])
                        else:
                            cur_sh_index_2 -= 1
                            final_x.append(int_x)
                            final_y.append(int_y)
                            scour_hole_x_4 = int_x
                            scour_hole_y_4 = int_y
                            break
                    cur_sh_index = cur_sh_index_2

                cs_below_sh = not cs_below_sh

            else:
                if cs_below_sh:
                    for cur_xy_index_2 in range(cur_xy_index, size_xy):
                        if len(final_x) == 0 or final_x[-1] < x_data[cur_xy_index_2]:
                            final_x.append(x_data[cur_xy_index_2])
                            final_y.append(y_data[cur_xy_index_2])
                    cur_xy_index = cur_xy_index_2
                else:
                    for cur_sh_index_2 in range(cur_sh_index, size_sh):
                        if len(final_x) == 0 or final_x[-1] < scour_hole_x[cur_sh_index_2]:
                            final_x.append(scour_hole_x[cur_sh_index_2])
                            final_y.append(scour_hole_y[cur_sh_index_2])
                    cur_sh_index = cur_sh_index_2

        # Reset scour hole to be original size/shape (indicate the slopes below or above grade)
        scour_hole_x_1 = x_station - (t_width / 2.0)
        scour_hole_y_1 = top_elevation

        scour_hole_x_2 = x_station - b_width / 2.0
        scour_hole_y_2 = top_elevation - depth

        scour_hole_x_3 = x_station + b_width / 2.0
        scour_hole_y_3 = top_elevation - depth

        scour_hole_x_4 = x_station + (t_width / 2.0)
        scour_hole_y_4 = top_elevation

        scour_hole_x = [scour_hole_x_1, scour_hole_x_2, scour_hole_x_3, scour_hole_x_4]
        scour_hole_y = [scour_hole_y_1, scour_hole_y_2, scour_hole_y_3, scour_hole_y_4]

        return True, final_x, final_y, scour_hole_x, scour_hole_y

    def apply_longterm_degradation(self, x_data, y_data, bridge_geometry, null_data, zero_tol):
        """Apply the long-term degradation.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            bridge_geometry (BridgeGeometry): The bridge geometry

        Returns:
            Applied (bool): Whether long-term degradation is computed
            x_data (list): The x data
            y_data (list): The y data
            longterm_degradation (float): The long-term degradation
        """
        longterm_var = self.input_dict['calc_data']['Long-term degradation']['calculator']
        can_compute, warnings = longterm_var.get_can_compute_with_subdict(
            self.input_dict, self.input_dict['calc_data']['Long-term degradation'])
        if not can_compute:
            return False, x_data, y_data, 0.0

        is_banks_specified = ScenarioCalc.determine_if_banks_are_specified(bridge_geometry)
        if is_banks_specified:
            left_bank_station = bridge_geometry['Channel left bank station']
            right_bank_station = bridge_geometry['Channel right bank station']
        else:
            self.warnings['define banks'] = "Please enter the channel banks to plot the long-term degradation"
            self.warnings.update(longterm_var.warnings)
            return False, x_data, y_data, 0.0

        longterm_degradation = longterm_var.scour_depth
        can_compute, _ = longterm_var.get_can_compute()
        # if longterm_degradation is None or not can_compute:
        #     return False, x_data, y_data, 0.0

        width = abs(right_bank_station - left_bank_station)
        longterm_var.input_dict['calc_data']['Scour hole geometry']['calculator'].use_specified_b_width = True
        longterm_var.input_dict['calc_data']['Scour hole geometry']['calculator'].b_width = width

        # longterm_degradation = longterm_var.get_scour_depth()  # Get updated scour hole calculation
        longterm_var.compute_data(longterm_var.input_dict, longterm_var.plot_dict)
        longterm_degradation = longterm_var.scour_depth
        if longterm_degradation < 0:
            longterm_degradation = 0.0

        # Determine Scour hole geometry
        top_width = 0.0
        if 'Top width' in longterm_var.input_dict['calc_data']['Scour hole geometry']['calculator'].results:
            top_width = longterm_var.input_dict['calc_data']['Scour hole geometry']['calculator'].results['Top width']

        left_s, right_s = self._determine_left_and_right(left_bank_station, right_bank_station, top_width, width)

        streambed_interp = Interpolation(x_data, y_data, null_data=null_data, zero_tol=zero_tol)
        left_t_elevation, _ = streambed_interp.interpolate_y(left_s)
        left_b_elevation = streambed_interp.interpolate_y(left_bank_station)[0] - longterm_degradation
        right_t_elevation, _ = streambed_interp.interpolate_y(right_s)
        right_b_elevation = streambed_interp.interpolate_y(right_bank_station)[0] - longterm_degradation

        final_x, final_y = self._apply_ltd_or_cs_to_cross_section(x_data, y_data, left_s, left_t_elevation,
                                                                  left_bank_station, left_b_elevation,
                                                                  right_bank_station, right_b_elevation,
                                                                  right_s, right_t_elevation, longterm_degradation)

        self.warnings.update(longterm_var.warnings)
        return True, final_x, final_y, longterm_degradation

    def apply_contraction_scour(self, x_data, y_data, bridge_geometry, ltd):
        """Apply the contraction scour.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            bridge_geometry (BridgeGeometry): The bridge geometry

        Returns:
            Applied (bool): Whether contraction scour is computed
            list: The x data
            list: The y data
        """
        main_cs = 0.0
        l_cs = 0.0
        r_cs = 0.0
        if len(x_data) == 0 or len(y_data) == 0:
            return False, x_data, y_data, l_cs, main_cs, r_cs
        is_banks_specified = ScenarioCalc.determine_if_banks_are_specified(bridge_geometry)
        main_var = self.input_dict['calc_data']['Contraction scour']['Main channel contraction scour']['calculator']
        main_var.longterm_scour_depth = ltd
        if len(main_var.input_dict) == 0:
            main_var.input_dict = copy.deepcopy(self.input_dict)
            main_var.input_dict['calc_data'] = self.input_dict['calc_data']['Contraction scour'][
                'Main channel contraction scour']
        can_compute, _ = main_var.get_can_compute_with_subdict(
            self.input_dict, self.input_dict['calc_data']['Contraction scour']['Main channel contraction scour'])
        if is_banks_specified:
            left_var = \
                self.input_dict['calc_data']['Contraction scour']['Left overbank contraction scour']['calculator']
            right_var = \
                self.input_dict['calc_data']['Contraction scour']['Right overbank contraction scour']['calculator']
            if len(left_var.input_dict) == 0:
                left_var.input_dict = copy.deepcopy(self.input_dict)
                left_var.input_dict['calc_data'] = self.input_dict['calc_data']['Contraction scour'][
                    'Left overbank contraction scour']
            if len(right_var.input_dict) == 0:
                right_var.input_dict = copy.deepcopy(self.input_dict)
                right_var.input_dict['calc_data'] = self.input_dict['calc_data']['Contraction scour'][
                    'Right overbank contraction scour']
            left_can_compute, _ = left_var.get_can_compute()
            right_can_compute, _ = right_var.get_can_compute()
            if not left_can_compute and not can_compute and not right_can_compute:
                return False, x_data, y_data, l_cs, main_cs, r_cs
            left_var.compute_data(left_var.input_dict, self.plot_dict)
            l_cs = left_var.scour_depth
            right_var.compute_data(right_var.input_dict, self.plot_dict)
            r_cs = right_var.scour_depth
        else:
            if not can_compute:
                return False, x_data, y_data, l_cs, main_cs, r_cs
        main_var.compute_data(main_var.input_dict, self.plot_dict)
        main_cs = main_var.scour_depth

        # constrain all contraction scour to be between abutments
        left_abut_x = x_data[0]
        right_abut_x = x_data[-1]
        if ScenarioCalc.is_left_abutments_specified(bridge_geometry):
            left_abut_x = bridge_geometry['Left abutment toe station']
        if ScenarioCalc.is_right_abutments_specified(bridge_geometry):
            right_abut_x = bridge_geometry['Right abutment toe station']

        left_station = left_abut_x
        right_station = right_abut_x

        new_x = copy.deepcopy(x_data)
        new_y = copy.deepcopy(y_data)

        if is_banks_specified:
            # Adjust the banks
            left_station = bridge_geometry['Channel left bank station']
            right_station = bridge_geometry['Channel right bank station']

        m_left_t_s, m_left_t_e, m_left_b_s, m_left_b_e, m_right_b_s, m_right_b_e, m_right_t_s, m_right_t_e, m_cs = \
            self._determine_scour_hole_geometry(x_data, y_data, left_station, right_station, main_var)

        applied_left_cs = False
        applied_right_cs = False

        if is_banks_specified:
            # Apply the contraction scour to each component of the channel

            # Determine the scour hole geometry
            # Left CS
            l_left_t_s, l_left_t_e, l_left_b_s, l_left_b_e, l_right_b_s, l_right_b_e, l_right_t_s, l_right_t_e, l_cs = \
                self._determine_scour_hole_geometry(x_data, y_data, left_abut_x, left_station, left_var)

            # Right CS
            r_left_t_s, r_left_t_e, r_left_b_s, r_left_b_e, r_right_b_s, r_right_b_e, r_right_t_s, r_right_t_e, r_cs = \
                self._determine_scour_hole_geometry(x_data, y_data, right_station, right_abut_x, right_var)

            # Apply the contraction scour
            # Left CS
            if l_cs and l_cs > 0:
                new_x, new_y = self._apply_ltd_or_cs_to_cross_section(new_x, new_y, l_left_t_s, l_left_t_e,
                                                                      l_left_b_s, l_left_b_e, l_right_b_s, l_right_b_e,
                                                                      l_right_t_s, l_right_t_e, l_cs)
                applied_left_cs = True

            # Right CS
            if r_cs and r_cs > 0:
                new_x, new_y = self._apply_ltd_or_cs_to_cross_section(new_x, new_y, r_left_t_s, r_left_t_e,
                                                                      r_left_b_s, r_left_b_e, r_right_b_s,
                                                                      r_right_b_e, r_right_t_s, r_right_t_e, r_cs,
                                                                      applied_left_cs)
                applied_right_cs = True

        # Main CS
        final_x, final_y = self._apply_ltd_or_cs_to_cross_section(new_x, new_y, m_left_t_s, m_left_t_e, m_left_b_s,
                                                                  m_left_b_e, m_right_b_s, m_right_b_e, m_right_t_s,
                                                                  m_right_t_e, m_cs, applied_left_cs, applied_right_cs)

        return True, final_x, final_y, l_cs, m_cs, r_cs

    def _determine_left_and_right(self, left_station, right_station, top_width, bottom_width):
        """Determine the overhang.

        Args:
            left_station (float): The left station
            right_station (float): The right station
            top_width (float): The top width
            bottom_width (float): The bottom width

        Returns:
            left_s (float): The left station
            right_s (float): The right station
        """
        if isinstance(top_width, list):
            top_width = top_width[-1]
        overhang = (top_width - bottom_width) / 2.0
        left_s = left_station - overhang
        right_s = right_station + overhang

        return left_s, right_s

    def _determine_scour_hole_geometry(self, x_data, y_data, left_station, right_station, cs_calc):
        """Apply the single contraction scour.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            left_station (float): The left station
            right_station (float): The right station
            cs_calc (ContractionScourCalc): The contraction scour calculation

        Returns:
            left_t (float): The left top station of the scour hole
            left_t_elevation (float): The left top elevation of the scour hole
            left_b (float): The left bottom station of the scour hole
            left_b_elevation (float): The left bottom elevation of the scour hole
            right_b (float): The right bottom station of the scour hole
            right_b_elevation (float): The right bottom elevation of the scour hole
            right_t (float): The right top station of the scour hole
            right_t_elevation (float): The right top elevation of the scour hole
            cs (float): The scour depth
        """
        _, zero_tol = self.get_data('Zero tolerance')
        width = abs(right_station - left_station)
        # cs_calc.input['Scour hole geometry'].get_val().use_specified_b_width = True
        # cs_calc.input['Scour hole geometry'].get_val().b_width = width
        cs_calc.input_dict['calc_data']['Scour hole geometry']['calculator'].use_specified_b_width = True
        cs_calc.input_dict['calc_data']['Scour hole geometry']['calculator'].b_width = width

        left_t = 0.0
        left_t_elevation = 0.0
        left_b_elevation = 0.0
        right_b_elevation = 0.0
        right_t = 0.0
        right_t_elevation = 0.0

        cs = cs_calc.scour_depth
        if cs is None or cs <= zero_tol:
            return left_t, left_t_elevation, left_station, left_b_elevation, right_station, right_b_elevation, \
                right_t, right_t_elevation, cs

        # cs_calc.input['Scour hole geometry'].get_val().compute_data()
        cs_calc.input_dict['calc_data']['Scour hole geometry']['calculator'].compute_data_with_subdict(
            cs_calc.input_dict, cs_calc.input_dict['calc_data']['Scour hole geometry'], self.plot_dict)
        top_width = cs_calc.input_dict['calc_data']['Scour hole geometry']['calculator'].results['Top width']
        left_t, right_t = self._determine_left_and_right(left_station, right_station, top_width, width)

        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')
        streambed_interp = Interpolation(x_data, y_data, null_data=null_data, zero_tol=zero_tol)
        tol = 0.01  # Just a little extra to ensure the lines intersect
        left_t_elevation = streambed_interp.interpolate_y(left_t)[0] + tol
        left_t -= tol
        left_b_elevation = streambed_interp.interpolate_y(left_station)[0] - cs
        right_t_elevation = streambed_interp.interpolate_y(right_t)[0] + tol
        right_t += tol
        right_b_elevation = streambed_interp.interpolate_y(right_station)[0] - cs

        # final_x, final_y = self._apply_ltd_or_cs_to_cross_section(x_data, y_data, left_station, right_station, left_t,
        #                                                           right_t, cs)

        return left_t, left_t_elevation, left_station, left_b_elevation, right_station, right_b_elevation, right_t, \
            right_t_elevation, cs

    def _apply_ltd_or_cs_to_cross_section(self, x_data, y_data, left_t_station, left_t_elevation, left_b_station,
                                          left_b_elevation, right_b_station, right_b_elevation, right_t_station,
                                          right_t_elevation, scour_depth, applied_left_cs=False,
                                          applied_right_cs=False):
        """Apply the long-term degradation or contraction scour to the cross section.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            left_t_station (float): The left top station of the scour hole
            left_t_elevation (float): The left top elevation of the scour hole
            left_b_station (float): The left bottom station of the scour hole
            left_b_elevation (float): The left bottom elevation of the scour hole
            right_b_station (float): The right bottom station of the scour hole
            right_b_elevation (float): The right bottom elevation of the scour hole
            right_t_station (float): The right top station of the scour hole
            right_t_elevation (float): The right top elevation of the scour hole
            scour_depth (float): The scour depth

        Returns:
            list: The x data
            list: The y data
        """
        final_x = []
        final_y = []
        scour_x = []
        scour_y = []

        pts_added = False

        if scour_depth <= 0.0:
            return x_data, y_data

        # Generate scour hole geometry
        if left_b_station > x_data[0]:
            scour_x.append(left_t_station)
            scour_y.append(left_t_elevation)
        scour_x.append(left_b_station)
        scour_y.append(left_b_elevation)
        for index in range(len(x_data)):
            if left_b_station < x_data[index] < right_b_station:
                scour_x.append(x_data[index])
                scour_y.append(y_data[index] - scour_depth)
        scour_x.append(right_b_station)
        scour_y.append(right_b_elevation)
        if right_b_station < x_data[-1]:
            scour_x.append(right_t_station)
            scour_y.append(right_t_elevation)

        _, zero_tol = self.get_data('Zero tolerance')

        # Adjust the left side to tie into the left scour hole
        if applied_left_cs:
            int_found, int_x, int_y, _, _ = find_intersection_and_closest_index(x_data, y_data,
                                                                                scour_x, scour_y, tol=zero_tol)
            if int_found:
                while (len(scour_x) > 0 and scour_x[0] < int_x):
                    scour_x.pop(0)
                    scour_y.pop(0)
                scour_x.insert(0, int_x)
                scour_y.insert(0, int_y)
                left_t_station = int_x  # Update so we tie into the cross-section
        # Adjust the right side to tie into the left scour hole
        if applied_right_cs:
            # determine intersection between y_data_scour and y_data
            rev_x_data = x_data[::-1]
            rev_y_data = y_data[::-1]
            rev_scour_x = scour_x[::-1]
            rev_scour_y = scour_y[::-1]
            int_found, int_x, int_y, _, _ = find_intersection_and_closest_index(rev_x_data, rev_y_data, rev_scour_x,
                                                                                rev_scour_y, tol=zero_tol)

            if int_found:
                while (scour_x[-1] > int_x):
                    scour_x.pop()
                    scour_y.pop()
                scour_x.append(int_x)
                scour_y.append(int_y)
                right_t_station = int_x  # Update so we tie into the cross-section

        # Add the x, y data outside the scour hole
        for index in range(len(x_data)):
            if left_t_station < x_data[index] < right_t_station:
                if not pts_added:
                    final_x.extend(scour_x)
                    final_y.extend(scour_y)
                    pts_added = True
            else:
                final_x.append(x_data[index])
                final_y.append(y_data[index])

        return final_x, final_y

    def apply_pier_scour(self, x_data, y_data, bridge_geometry, orig_x, orig_y, wse_x, wse_y, null_data=None,
                         zero_tol=None):
        """Apply the contraction scour.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            bridge_geometry (BridgeGeometry): The bridge geometry
            orig_x (list): The original x data
            orig_y (list): The original y data
            wse_x (list): The water surface elevation x data
            wse_y (list): The water surface elevation y data
            null_data (any, optional): The null data value
            zero_tol (float, optional): The zero tolerance value

        Returns:
            any_applied (bool): whether any pier scour holes were applied
            final_x (list): The x data
            final_y (list): The y data
            scour_holes_x (list): x values for the scour hole
            scour_holes_y (list): y values for the scour hole

        """
        final_x = copy.deepcopy(x_data)
        final_y = copy.deepcopy(y_data)
        scour_holes_x = []
        scour_holes_y = []

        if null_data is None:
            _, null_data = self.get_data('Null data')
        if zero_tol is None:
            _, zero_tol = self.get_data('Zero tolerance')
        cs_streambed_interp = Interpolation(final_x, final_y, null_data=null_data, zero_tol=zero_tol)
        orig_streambed_interp = Interpolation(orig_x, orig_y, null_data=null_data, zero_tol=zero_tol)
        wse_interp = None
        if wse_x is not None:
            wse_interp = Interpolation(wse_x, wse_y, null_data=null_data, zero_tol=zero_tol)

        thalweg_elevation = min(y_data)

        any_applied = False

        left_bank_station = bridge_geometry['Channel left bank station']
        right_bank_station = bridge_geometry['Channel right bank station']

        for pier_index in bridge_geometry['Piers']:
            srp = bridge_geometry['Piers'][pier_index]['Scour reference point']
            pier_widths = bridge_geometry['Piers'][pier_index]['Pier geometry']['Width']
            pier_elevations = bridge_geometry['Piers'][pier_index]['Pier geometry']['Elevation']
            pier_cl = bridge_geometry['Piers'][pier_index]['Centerline']
            streambed_elev, _ = cs_streambed_interp.interpolate_y(pier_cl)
            scour_top_elevation = streambed_elev
            if srp == 'Thalweg':
                scour_top_elevation = thalweg_elevation
            streambed_elev_prior_to_scour, _ = orig_streambed_interp.interpolate_y(pier_cl)

            pier = self.input_dict['calc_data']['Pier scour'][pier_index]['calculator']
            pier.scour_reference_point = srp
            self.input_dict['calc_data']['Pier scour'][pier_index]['Scour reference point'] = srp
            self.input_dict['calc_data']['Pier scour'][pier_index]['Scour hole geometry'][
                'calculator'].pier_widths = pier_widths
            self.input_dict['calc_data']['Pier scour'][pier_index]['Scour hole geometry'][
                'calculator'].pier_elevations = pier_elevations
            self.input_dict['calc_data']['Pier scour'][pier_index]['Scour hole geometry'][
                'calculator'].streambed_elevation = streambed_elev
            pier.streambed_elev_prior_to_scour = streambed_elev_prior_to_scour
            pier.local_streambed_after_cs_ltd = streambed_elev
            pier.wse_x = wse_x
            pier.wse_y = wse_y
            compute_shear_decay = self.input_dict['calc_data']['Compute shear decay']
            if compute_shear_decay:
                con = self.input_dict['calc_data']['Contraction scour']['Main channel contraction scour'][
                    'calculator']
                if pier_cl < left_bank_station and self.is_left_channel_bank_specified(bridge_geometry):
                    con = self.input_dict['calc_data']['Contraction scour']['Left overbank contraction scour'][
                        'calculator']
                elif pier_cl > right_bank_station and self.is_right_channel_bank_specified(bridge_geometry):
                    con = self.input_dict['calc_data']['Contraction scour']['Right overbank contraction scour'][
                        'calculator']
                if 'Shear decay results' in con.results:
                    pier.con_decay_x = con.results['Shear decay results']['Shear decay curve shear']
                    pier.con_decay_y = con.results['Shear decay results']['Shear decay curve elevation']
                    pier.con_decay_x_markers = con.results['Shear decay results']['Shear decay curve shear markers']
                    pier.con_decay_y_markers = con.results['Shear decay results'][
                        'Shear decay curve elevation markers']
            can_compute, _, _, _, _ = pier.compute_data_with_subdict(
                self.input_dict, self.input_dict['calc_data']['Pier scour'][pier_index],)
            if can_compute:
                sh = self.input_dict['calc_data']['Pier scour'][pier_index]['Scour hole geometry']['calculator']
                is_applied, final_x, final_y, sh_x, sh_y = self.apply_scour_hole(
                    final_x, final_y, pier_cl, scour_top_elevation, sh)
                if is_applied:
                    any_applied = True
                if pier_index > 0:
                    scour_holes_x.append(None)
                    scour_holes_y.append(None)

                scour_holes_x.extend(sh_x)
                scour_holes_y.extend(sh_y)

            scour_interp = Interpolation(final_x, final_y, null_data=null_data, zero_tol=zero_tol)
            scour_elev, _ = scour_interp.interpolate_y(pier_cl)
            if wse_interp is not None:
                wse_elev, _ = wse_interp.interpolate_y(pier_cl)
                pier.max_flow_depth = wse_elev - scour_elev
            else:
                pier.max_flow_depth = None
            pier.total_scour_depth = streambed_elev_prior_to_scour - scour_elev
            pier.total_scour_elevation = scour_elev

        return any_applied, final_x, final_y, scour_holes_x, scour_holes_y

    def apply_abutment_shear_decay_data(self, abutment_toe_station, abutment, bridge_geometry):
        """Set data for shear decay to the abutment.

        Args:
            abutment_toe_station (float): The abutment toe station
            abutment (AbutmentScourData): The abutment
            bridge_geometry (BridgeGeometry): The bridge geometry
        """
        compute_shear_decay = self.input_dict['calc_data']['Compute shear decay']
        if compute_shear_decay:
            left_bank_station = bridge_geometry['Channel left bank station']
            right_bank_station = bridge_geometry['Channel right bank station']

            con = self.input_dict['calc_data']['Contraction scour']['Main channel contraction scour']['calculator']
            if abutment_toe_station < left_bank_station and self.is_left_channel_bank_specified(bridge_geometry):
                con = self.input_dict['calc_data']['Contraction scour']['Left overbank contraction scour']['calculator']
            elif abutment_toe_station > right_bank_station and self.is_right_channel_bank_specified(bridge_geometry):
                con = self.input_dict['calc_data']['Contraction scour']['Right overbank contraction scour'][
                    'calculator']
            if 'Shear decay results' in con.results:
                abutment.con_decay_x = con.results['Shear decay results']['Shear decay curve shear']
                abutment.con_decay_y = con.results['Shear decay results']['Shear decay curve elevation']
                abutment.con_decay_x_markers = con.results['Shear decay results'][
                    'Shear decay curve shear markers']
                abutment.con_decay_y_markers = con.results['Shear decay results'][
                    'Shear decay curve elevation markers']

    def apply_abutment_scour(self, x_data, y_data, abutment_toe_station, abutment, abutment_text, null_data, zero_tol):
        """Apply the contraction scour.

        Args:
            x_data (list): The x data
            y_data (list): The y data
            abutment (AbutmentScourData): The abutment with scour to apply
            longterm_degradation (float): The long-term degradation
            abutment_toe_station (float): The abutment toe station
            abutment_text (str): The abutment text (e.g., 'Left' or 'Right')
            null_data (any, optional): The null data value
            zero_tol (float, optional): The zero tolerance value

        Returns:
            is_applied (bool) whether the scour was applied or not
            list: The x data
            list: The y data
        """
        streambed_interp = Interpolation(x_data, y_data, null_data=null_data, zero_tol=zero_tol)
        streambed_elev, _ = streambed_interp.interpolate_y(abutment_toe_station)

        abutment.cross_section_x = x_data
        abutment.cross_section_y = y_data
        abutment.abutment_toe_station = abutment_toe_station

        abutment.compute_data_with_subdict(
            self.input_dict, self.input_dict['calc_data']['Abutment scour'][abutment_text], self.plot_dict)
        is_applied, final_x, final_y, sh_x, sh_y = self.apply_scour_hole(
            x_data, y_data, abutment_toe_station, streambed_elev,
            abutment.input_dict['calc_data']['Scour hole geometry']['calculator'])

        scour_interp = Interpolation(final_x, final_y, null_data=null_data, zero_tol=zero_tol)
        scour_elev, _ = scour_interp.interpolate_y(abutment_toe_station)

        return is_applied, final_x, final_y, sh_x, sh_y, abutment.streambed_elev_prior_to_scour, scour_elev
