"""Bridge Deck Table Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.table_data import TableData

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class CrossSectionCalc(TableCalc):
    """A class that defines the bridge deck table calculator."""
    def __init__(self):
        """Initializes the calculator."""
        super().__init__()
        self.name = 'Cross-section'
        self.plot_names = ['Cross-section']
        self.define_cross_section = True
        self.mannings_n_long = []

    def _compute_data(self):
        """Compute the data.

        Returns:
            True, if successful; otherwise, False
        """
        super()._compute_data()

        _, null_data = self.get_data('Null data', -9999.0)
        _, zero_tol = self.get_data('Zero tolerance', 0.0001)

        if self.define_cross_section:
            self.stations = copy.copy(self.input_dict['calc_data']['Data input']['Station'])
            self.elevations = copy.copy(self.input_dict['calc_data']['Data input']['Elevation'])
            self.mannings_n_long = self.input_dict['calc_data']['Data input']["Manning's n"]

            self.cs_stations = copy.copy(self.stations)
            self.cs_elevations = copy.copy(self.elevations)
            self.cs_mannings_n = copy.copy(self.mannings_n_long)

        else:
            self.stations = copy.copy(self.cs_stations)
            self.elevations = copy.copy(self.cs_elevations)

            if self.stations is None or self.elevations is None or len(self.stations) == 0 or len(self.elevations) == 0:
                return

            self.mannings_n_long = []
            n_stations = self.input_dict['calc_data']['Data input']["Manning's n stations"]
            n_coefs = self.input_dict['calc_data']['Data input']["Manning's n coefficients"]

            xy_interp = Interpolation([], [], null_data=null_data, zero_tol=zero_tol)

            # Make sure we have every station from the Manning's n stations in the cross-section
            min_station = min(s for s in self.stations if s != null_data)
            max_station = max(s for s in self.stations if s != null_data)
            for station in n_stations:
                if station not in self.stations and station != null_data and min_station < station < max_station:
                    xy_interp.set_xy(self.stations, self.elevations)  # Update the geometry (we may have inserted)
                    elevation, _ = xy_interp.interpolate_y(station)
                    if elevation is not None and elevation != null_data and not math.isnan(elevation):
                        insert_index = 0
                        while not (self.stations[insert_index] < station < self.stations[insert_index + 1]
                                   or self.stations[insert_index] > station > self.stations[insert_index + 1]):
                            insert_index += 1
                        self.stations.insert(insert_index + 1, station)
                        self.elevations.insert(insert_index + 1, elevation)

            if n_stations[0] > self.stations[0]:
                self.warnings['First Manning\'s n station'] = \
                    'The first Manning\'s n station is greater than the first cross-section station.'
            if n_stations[-1] < self.stations[-1]:
                self.warnings['Last Manning\'s n station'] = \
                    'The last Manning\'s n station is less than the last cross-section station.'

            n_idx = 0
            for index in range(1, len(self.stations)):
                # Advance n_idx if the next n_station is <= current station
                while n_idx + 1 < len(n_stations) and self.stations[index] >= n_stations[n_idx + 1]:
                    n_idx += 1
                if n_idx < len(n_coefs):
                    self.mannings_n_long.append(n_coefs[n_idx])
                else:
                    self.mannings_n_long.append(n_coefs[-1])

        self.mannings_n = []
        self.mannings_stations = []

        prev_n = None
        size = min(len(self.stations), len(self.elevations), len(self.mannings_n_long))
        for i in range(size):
            n = self.mannings_n_long[i]
            if prev_n is None and n is not None and n is not math.isnan(n) and n != null_data and \
                    self.stations[i] != null_data and self.elevations[i] != null_data:
                self.mannings_n.append(n)
                self.mannings_stations.append(self.stations[i])
                prev_n = n
            if n != prev_n or self.stations[i] == null_data or self.elevations[i] == null_data and prev_n is not None:
                self.mannings_n.append(prev_n)
                if self.stations[i] == null_data or self.elevations[i] == null_data:
                    self.mannings_stations.append(self.stations[i - 1])
                else:
                    self.mannings_stations.append(self.stations[i])
                self.mannings_n.append(math.nan)
                self.mannings_stations.append(math.nan)
                prev_n = n
                if n is not None and n is not math.isnan(n) and n != null_data and self.stations[i] != null_data and \
                        self.elevations[i] != null_data:
                    self.mannings_n.append(n)
                    self.mannings_stations.append(self.stations[i])
                else:
                    prev_n = None
        if self.stations[-1] not in self.mannings_stations and prev_n is not None:
            self.mannings_n.append(prev_n)
            self.mannings_stations.append(self.stations[-1])

        self.plot_dict['Cross-section']['series'][0]['x var'].set_val(self.stations)
        self.plot_dict['Cross-section']['series'][0]['y var'].set_val(self.elevations)

        self.plot_dict['Cross-section']['series'][0]['xx var'].set_val(self.mannings_stations)
        self.plot_dict['Cross-section']['series'][0]['yy var'].set_val(self.mannings_n)

        return True


class CrossSectionTable(TableData):
    """Override the table class, so we can draw plots based on widths."""
    def __init__(self, theme, name, plot_names, name_of_items='points', stand_alone_calc=True, input=None, min_items=1,
                 max_items=sys.maxsize, show_increment=False, app_data=None, model_name=None, project_uuid=None,
                 define_cross_section=True):
        """Initializes the calculator.

        Args:
            theme (dict): the theme
            name (str): the name of the calculator
            plot_names (list of str): list of plot names
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the calculator is a stand-alone calculator
            input (dict): dictionary of input variables
            min_items (int): minimum number of items
            max_items (int): maximum number of items
            show_increment (bool): whether to show the increment
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
            define_cross_section (bool): True to define the cross-section by station and elevation; False to define
                the Manning's n values by station
        """
        super().__init__(theme=theme, name=name, plot_names=plot_names, name_of_items=name_of_items,
                         stand_alone_calc=stand_alone_calc, input=input, min_items=min_items, max_items=max_items,
                         show_increment=show_increment, app_data=app_data, model_name=model_name,
                         project_uuid=project_uuid)

        # Set to False, if this is to display the cross-section only, but define the Manning's n values by station
        self.calculator = CrossSectionCalc()
        self.define_cross_section = define_cross_section
        # self.calculator.define_cross_section = define_cross_section

        # Manning's variables
        self.mannings_n_disp_var = copy.copy(input["Manning's n"])
        self.mannings_stations_disp_var = copy.copy(input['Station'])
        self.mannings_n_var = copy.copy(input["Manning's n"])
        self.mannings_stations_var = copy.copy(input['Station'])
        self.mannings_n = []
        self.mannings_stations = []
        self.cs_stations = []
        self.cs_elevations = []
        self.cs_mannings_n = []
        self.stations = []
        self.elevations = []
        self.mannings_n_long = []

        self.intermediate_to_copy.extend(
            ['define_cross_section', 'mannings_n', 'mannings_stations', 'stations', 'elevations', 'mannings_n_long',
             'cs_stations', 'cs_elevations', 'cs_mannings_n'])

        self.input['Data input'].value.input["Manning's n stations"] = copy.copy(self.mannings_stations_var)
        self.input['Data input'].value.input["Manning's n"] = copy.copy(self.mannings_n_var)

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input_list[
            "Manning's n stations display"] = copy.copy(self.mannings_stations_var)
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input_list[
            "Manning's n coefficients display"] = copy.copy(self.mannings_n_var)

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input_list[
            "Manning's n stations"] = copy.copy(self.mannings_stations_var)
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input_list[
            "Manning's n coefficients"] = copy.copy(self.mannings_n_var)

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['X axis'].set_val('Station')
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Y axis'].set_val('Elevation')

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Plot yy'].set_val(True)

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['XX axis'].set_val("Manning's n stations display")
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['YY axis'].set_val("Manning's n coefficients display")

        # Cross-section data
        structure_color = (77, 77, 77)
        structure_fill_color = (191, 191, 191)
        roughness_color = (255, 215, 0)
        if self.theme is not None:
            structure_color = self.theme['Plot structure color']
            structure_fill_color = self.theme['Plot structure fill color']
            roughness_color = self.theme['Plot roughness color']

        # Set cross-section plot options
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line color'].set_val(structure_color)
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill color'].set_val(structure_fill_color)
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill pattern'].set_val('sand')
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line type'].set_val('solid')
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line width'].set_val(2.0)

        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Name yy'].set_val("Manning's n")
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line color yy'].set_val(roughness_color)
        # self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
        #     'Plot series'].value.item_list[0].input['Fill color yy'].set_val(soil_fill_color)
        # self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
        #     'Plot series'].value.item_list[0].input['Fill pattern yy'].set_val('sand')
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line type yy'].set_val('dash-dot')
        self.input['Plot options']['Cross-section'].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line width yy'].set_val(1.5)

    def get_input_dual_tab_group(self, unknown=None):
        """Get the input dual tab group.

        Args:
            unknown (str): the unknown variable

        Returns:
            dict: the input tab group
        """
        input_dual = super().get_input_dual_tab_group(unknown=unknown)

        # Manning's n value needs to be reduced by 1, as the value applies to a segment (two rows of the table)
        if 'Data table' in input_dual and "Manning's n" in input_dual['Data table']:
            input_dual['Data table']["Manning's n"].value -= 1
            input_dual['Data table']["Manning's n"].value_options = input_dual['Data table'][
                "Manning's n"].value_options[:-1]

        if 'Data table' in input_dual and "Manning's n coefficients" in input_dual['Data table']:
            input_dual['Data table']["Manning's n coefficients"].value -= 1
            input_dual['Data table']["Manning's n coefficients"].value_options = input_dual['Data table'][
                "Manning's n coefficients"].value_options[:-1]

        if "Manning's n stations display" in input_dual['Data table']:
            input_dual['Data table'].pop("Manning's n stations display")
        if "Manning's n coefficients display" in input_dual['Data table']:
            input_dual['Data table'].pop("Manning's n coefficients display")

        if self.define_cross_section:
            if "Manning's n stations" in input_dual['Data table']:
                input_dual['Data table'].pop("Manning's n stations")
            if "Manning's n coefficients" in input_dual['Data table']:
                input_dual['Data table'].pop("Manning's n coefficients")
        else:
            if "Manning's n" in input_dual['Data table']:
                input_dual['Data table'].pop("Manning's n")
            if "Station" in input_dual['Data table']:
                input_dual['Data table'].pop("Station")
            if "Elevation" in input_dual['Data table']:
                input_dual['Data table'].pop("Elevation")

        return input_dual
