"""CalcData for performing Culvert crossing operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules
from sortedcontainers import SortedDict

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.culvert.culvert_barrel_data import CulvertBarrelData
from xms.HydraulicToolboxCalc.hydraulics.culvert.culvert_crossing_calc import CulvertCrossingCalc
from xms.HydraulicToolboxCalc.hydraulics.culvert.roadway_data import RoadwayData
from xms.HydraulicToolboxCalc.hydraulics.culvert.tailwater_data import TailwaterData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class CulvertCrossingData(CalcData):
    """A class that defines a culvert crossing and performs Culvert crossing computations."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the culvert crossing calculator.

        Args:
            clear_my_own_results (bool): Whether this calculator should clear its own results or be told when to.
            app_data (AppData): The application data (settings).
            model_name (str): The name of the model.
            project_uuid (str): The project uuid.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Culvert Crossing Calculator"
        self.type = "CulvertCrossingCalc"
        self.class_name = 'Culvert Crossing Calculator'

        self.calculator = CulvertCrossingCalc()

        self.calc_support_dual_input = True
        self.stack_input_tables = False  # Stack the input tables, False will put them side by side

        # Input
        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow,
                                              select_name='specify flow(s) as:', name_append='flow'))
        self.input['Tailwater data'] = Variable(
            'Tailwater data', 'class', TailwaterData(app_data=app_data, model_name=model_name,
                                                     project_uuid=project_uuid))
        self.input['Roadway data'] = Variable(
            'Roadway data', 'class', RoadwayData(app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        # self.input['Number of barrels'] = Variable('Number of unique culvert barrels', 'int', 1, limits=(1, 100))
        # self.input['Selected culvert'] = Variable('Selected culvert', 'list', 0)
        # self.input['Culvert data'] = Variable('Culvert data', 'class', None)

        self.input['Culvert data'] = Variable(
            'Culvert data', 'calc_list',
            CalcOrVarlist(CulvertBarrelData, show_define_btn=False, default_name='culvert barrel',
                          default_plural_name='culvert barrels', min_number_of_items=1, app_data=app_data,
                          model_name=model_name, project_uuid=project_uuid, clear_my_own_results=False,
                          in_crossing=True))

        # Maybe?
        self.input['Display channel slope as flat'] = Variable('Display channel slope as flat', 'bool', False)

        self.culvert_barrels = []
        self.unknown = None

        # Intermediate
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'culvert_barrels', 'unknown', 'manning_n_calc', 'tailwater_channel', 'cross_flow_unk', 'culv_hw_flow_list',
            'cross_invert_elevation', 'max_rise', 'single_barrel_flow', 'flowing_full', 'max_open_channel_flow',
            'single_barrel_flow', 'flowing_full', 'max_open_channel_flow', 'inlet_control_depth',
            'outlet_control_depth', 'headwaterDepth', 'barrel_segments', 'critical_depths', 'critical_velocities',
            'normal_depths', 'normal_velocities', 'wse_stations', 'wse_elevations', 'tw_depth', 'tw_velocity',
            'outlet_depth', 'outlet_velocity', 'flow', 'hyd_jump_exists', 'hyd_jump_swept_out',
            'hyd_jump_station_absolute', 'hyd_jump_final_station_absolute', 'hyd_jump_elevation',
            'hyd_jump_sequent_elevation', 'hyd_jump_depth', 'hyd_jump_sequent_depth', 'tailwater_channel',
            'tw_crossing_flows', 'tailwater_depths', 'tw_elevations', 'tailwater_velocities', 'tw_constant',
            'intermediate_results', 'depth', 'flow_area', 'wetted_perimeter', 'top_width', 'hydraulic_radius',
            'manning_n', 'velocity', 'energy', 'energy_loss', 'energy_slope', 'friction_slope_ave', 'delta_x', 'x',
            'y', 'sequent_depth', 'froude', 'boundary_shear_stress', 'max_shear_stress', 'results_upstream',
            'results_downstream', 'plot_x', 'plot_y', 'compute_vena_contracta', 'ignore_energy_in_full_flow_calcs',
            'hyd_jump_type_b_over_break', 'hyd_jump_exists', 'hyd_jump_swept_out', 'froude_jump', 'join_curves',
            'selected_culvert'
        ])

        manning_n_data = ManningNData(standalone_calc=False, app_data=app_data, model_name=model_name,
                                      project_uuid=project_uuid)
        manning_n_data.prepare_for_compute()
        self.manning_n_calc = manning_n_data.calculator
        self.manning_n_calc.input_dict, self.manning_n_calc.plot_dict = manning_n_data.prepare_input_dict()

        # self.tailwater_channel = TailwaterData(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        tailwater_data = TailwaterData(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        tailwater_data.prepare_for_compute()
        self.tailwater_channel = tailwater_data.calculator
        self.tailwater_channel.input_dict, self.tailwater_channel.plot_dict = tailwater_data.prepare_input_dict()

        self.cross_hw_flow = {}
        self.culv_hw_flow_list = []

        self.cross_invert_elevation = 0.0
        self.max_rise = 0.0

        self.single_barrel_flow = 0.0
        self.flowing_full = False
        self.max_open_channel_flow = 0.0

        self.inlet_control_depth = 0.0
        self.outlet_control_depth = 0.0
        self.headwaterDepth = 0.0

        self.barrel_segments = []
        self.critical_depths = []
        self.critical_velocities = []
        self.normal_depths = []
        self.normal_velocities = []
        self.wse_stations = []
        self.wse_elevations = []
        self.tw_depth = 0.0
        self.tw_velocity = 0.0
        self.outlet_depth = 0.0
        self.outlet_velocity = 0.0

        self.flow = 0.0

        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.hyd_jump_station_absolute = []
        self.hyd_jump_final_station_absolute = []
        self.hyd_jump_elevation = []
        self.hyd_jump_sequent_elevation = []
        self.hyd_jump_depth = []
        self.hyd_jump_sequent_depth = []

        # tailwater (tw)
        self.tailwater_channel = None
        self.tw_crossing_flows = []
        self.tailwater_depths = []
        self.tw_elevations = []
        self.tailwater_velocities = []
        self.tw_constant = False

        # lists for direct step point data
        self.intermediate_results = {
            'Depth': [],
            'flow_area': [],
            'wetted_perimeter': [],
            'top_width': [],
            'hydraulic_radius': [],
            'manning_n': [],
            'Velocity': [],
            'Energy': [],
            'energy_loss': [],
            'energy_slope': [],
            'friction_slope_ave': [],
            'delta_x': [],
            'x': [],
            'y': [],
            'sequent_depth': [],
            'froude': [],
            'boundary_shear_stress': [],
            'max_shear_stress': []
        }

        self.depth = []
        self.flow_area = []
        self.wetted_perimeter = []
        self.top_width = []
        self.hydraulic_radius = []
        self.manning_n = []
        self.velocity = []
        self.energy = []
        self.energy_loss = []
        self.energy_slope = []
        self.friction_slope_ave = []
        self.delta_x = []
        self.x = []
        self.y = []
        self.sequent_depth = []
        self.froude = []
        self.boundary_shear_stress = []
        self.max_shear_stress = []

        self.results_upstream = {}
        self.results_downstream = {}

        self.plot_x = []
        self.plot_y = []

        # Vena Contracta
        self.compute_vena_contracta = False

        # Hydraulic Jump
        self.ignore_energy_in_full_flow_calcs = False
        self.hyd_jump_type_b_over_break = False
        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.froude_jump = False
        self.join_curves = False

        self.selected_culvert = -1

        # Results
        self.clear_my_own_results = clear_my_own_results
        self.results = {}

        self.results['Headwater'] = Variable(
            'Headwater elevation', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Station'] = Variable(
            'Station', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['WSE'] = Variable(
            'WSE', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Distance from inlet'] = Variable(
            'Distance from inlet', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Depth'] = Variable(
            'Depth', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Flow area'] = Variable(
            'Flow area', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.results['Wetted perimeter'] = Variable(
            'Wetted perimeter', 'float_list', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Top width'] = Variable(
            'Top width', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic radius'] = Variable(
            'Hydraulic radius', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Manning n'] = Variable(
            'Manning n', 'float_list', 0.0, [], precision=4, unit_type=['coefficient'], native_unit='coefficient',
            us_units=[['coefficient']], si_units=[['coefficient']])
        self.results['Velocity'] = Variable(
            'Velocity', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['yd/s', 'ft/s', 'in/s']], si_units=[['m/s', 'mm/s']])
        self.results['Energy'] = Variable(
            'Energy', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Energy loss'] = Variable(
            'Energy loss', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Energy slope'] = Variable(
            'Energy slope', 'float_list', 0.0, limits=[-10.0, 10.0], precision=6, unit_type=['slope'],
            native_unit='ft/ft', us_units=[['ft/ft']], si_units=[['m/m']])
        self.results['Boundary shear stress'] = Variable(
            'Boundary shear stress', 'float_list', 0.0, precision=2, unit_type=['slope'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])
        self.results['Max shear stress'] = Variable(
            'Max shear stress', 'float_list', 0.0, precision=2, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])

        self.warnings = []

        # plot
        self.plots['profile'] = {}
        self.plots['profile']['Plot name'] = "Cross-section"
        self.plots['profile']['Legend'] = 'best'
        self.plots['profile']['index'] = 1

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        _, null_value = self.get_setting('Null data')

        if self.selected_culvert != null_value and self.selected_culvert >= 0 and \
                self.selected_culvert == self.input['Selected culvert'].get_index(null_value):
            self.culvert_barrels[self.selected_culvert].update_name()

        input_vars['Flows'] = self.input['Flows']

        input_vars['Tailwater data'] = self.input['Tailwater data']

        input_vars['Roadway data'] = self.input['Roadway data']

        # # Verify Number of barrels
        # num_barrels = self.input['Number of barrels'].get_val()
        # if num_barrels < 1:
        #     num_barrels = 1
        # self.input['Number of barrels'].set_val(num_barrels)
        # input_vars['Number of barrels'] = self.input['Number of barrels']

        # # initial_size_of_culvert_list = len(self.culvert_barrels)
        # # # Increase the data to match the interface for number of barrels
        # # # Only increase the size as needed.  We will only save the number selected and that
        # # # will remove culvert barrels.  This will be forgiving for users to change the number of barrels without
        # # # losing data - easier to change and go back and forth
        # # culvert_copy = CulvertBarrelData(False, 0, True, app_data=self.app_data, model_name=self.model_name,
        # #                                  project_uuid=self.project_uuid)
        # # copied_once = False
        # # for index in range(initial_size_of_culvert_list, num_barrels):
        # #     if index > 0 and not copied_once:
        # #         culvert_copy = copy.deepcopy(self.culvert_barrels[index - 1])
        # #         copied_once = True
        # #     culvert_copy.set_name(index)
        # #     culvert_copy.update_input_name()
        # #     self.culvert_barrels.append(copy.deepcopy(culvert_copy))
        # # Set the culvert data to the selected culvert barrel

        # # # Get and set the names for the culvert selection list
        # # name_list = []
        # # for index in range(len(self.culvert_barrels)):
        # #     name_list.append(self.culvert_barrels[index].name)

        # # # Verify the selected barrels
        # # selected_culvert_name = self.input['Selected culvert'].get_val()
        # # self.selected_culvert = self.input['Selected culvert'].get_index(null_value)
        # # if selected_culvert_name in name_list:
        # #     self.selected_culvert = name_list.index(selected_culvert_name)
        # # if self.selected_culvert < 0:
        # #     self.selected_culvert = 0
        # # if self.selected_culvert > num_barrels:
        # #     self.selected_culvert = num_barrels
        # # self.input['Selected culvert'].set_val(self.selected_culvert)
        # # self.input['Selected culvert'].value_options = name_list
        # # input_vars['Selected culvert'] = self.input['Selected culvert']

        # # # Set the selected culvert in the culvert data variable
        # # self.input['Culvert data'].set_val(self.culvert_barrels[self.selected_culvert])
        # # input_vars['Culvert data'] = self.input['Culvert data']
        # input_vars['Culvert data'] = self.input['Culvert data']

        # # Maybe ?
        # input_vars['Display channel slope as flat'] = self.input[
        #     'Display channel slope as flat']

        return input_vars

    def get_input_dual_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        # input_vars['Number of barrels'] = self.input['Number of barrels']

        input_vars['Culvert data'] = self.input['Culvert data']

        # Maybe ?
        input_vars['Display channel slope as flat'] = self.input[
            'Display channel slope as flat']

        return input_vars

    def get_result_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if not self.get_can_compute():
            return result_vars

        if self.input['Display channel slope as flat'].get_val():
            result_vars['Distance from inlet'] = self.results['Distance from inlet']
            result_vars['Depth'] = self.results['Depth']
        else:
            result_vars['Station'] = self.results['Station']
            result_vars['WSE'] = self.results['WSE']
        result_vars['Flow area'] = self.results['Flow area']
        result_vars['Wetted perimeter'] = self.results['Wetted perimeter']
        result_vars['Top width'] = self.results['Top width']
        result_vars['Hydraulic radius'] = self.results['Hydraulic radius']
        # result_vars['Manning n'] = self.results['Manning n']
        result_vars['Velocity'] = self.results['Velocity']
        # result_vars['Energy'] = self.results['Energy']
        # result_vars['Energy loss'] = self.results['Energy loss']
        result_vars['Energy slope'] = self.results['Energy slope']
        result_vars['Boundary shear stress'] = self.results[
            'Boundary shear stress']
        result_vars['Max shear stress'] = self.results['Max shear stress']

        return result_vars

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def setup_hw_to_flow_lists_per_culvert_barrel(self,
                                                  tw_elevation,
                                                  max_flow=None):
        """Setup a headwater to flow lists for each culvert.

        Args:
            tw_elevation (float): elevation of the tailwater
            max_flow (float): maximum flow to compute
        """
        # Need to complete full flow calcs, then add remainder of the computations
        flows = [
            1.0,
            2.0,
            4.0,
            8.0,
            16,
        ]  # 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768, 65536]
        num_culverts = self.input['Number of barrels'].get_val()
        self.culv_hw_flow_list = num_culverts * [SortedDict()]
        for culvert_index in range(num_culverts):
            max_hw = 0.0
            invert_elevation = self.culvert_barrels[culvert_index].input['Site data'].get_val().\
                input['Inlet invert elevation'].get_val()
            # TODO figure out why in some cases, then computed HW is less than TW
            min_elevation = invert_elevation
            # min_elevation = max(invert_elevation, tw_elevation)
            self.culv_hw_flow_list[culvert_index][0.0] = min_elevation
            for flow in flows:
                result = self.run_single_culvert_run(culvert_index, flow, flow)
                if result:
                    computed_hw = self.culvert_barrels[culvert_index].hw
                    self.culv_hw_flow_list[culvert_index][computed_hw] = flow
                    if max_hw < computed_hw:
                        max_hw = computed_hw

            # Continue adding to the culvert data until we have reached at least 4 * rise of culvert barrel
            rise = self.culvert_barrels[culvert_index].rise
            inlet_elev = self.culvert_barrels[
                culvert_index].inlet_invert_elevation
            flow = flows[-1]
            while max_hw < 4.0 * rise + inlet_elev and (max_flow is None or flow < max_flow):
                flow *= 2.0
                self.culvert_barrels[culvert_index].input['Flows'] = [flow]
                result = self.run_single_culvert_run(culvert_index, flow, flow)
                if result:
                    computed_hw = self.culvert_barrels[culvert_index].hw
                    self.culv_hw_flow_list[culvert_index][computed_hw] = flow
                    if max_hw < computed_hw:
                        max_hw = computed_hw

    def setup_hw_to_flow_lists_for_crossing(self,
                                            tw_elevation,
                                            max_flow=None):
        """Setup a headwater to flow lists for each culvert.

        Args:
            tw_elevation (float): elevation of the tailwater
            max_flow (float): maximum flow to compute
        """
        min_elevation = self.cross_invert_elevation
        # if min_elevation < tw_elevation:
        #     min_elevation = tw_elevation
        # TODO Put this in the settings calc!
        number_of_rises_to_support = 3.0  # data backs up 2-3 * the culvert rise;
        # max_elevation = self.max_rise * number_of_rises_to_support + min_elevation
        num_increments = 20
        distance_increment = (self.max_rise * number_of_rises_to_support) / num_increments
        if distance_increment < 0.25:
            distance_increment = 0.25
        # max_hw_elev = self.max_rise * number_of_rises_to_support

        for index in range(num_increments):
            elevation = min_elevation + index * distance_increment + distance_increment  # Don't start at zero
            result, flow = self.compute_crossing_flow_for_hw(elevation)
            if result:
                self.cross_hw_flow[flow] = elevation
            if max_flow is not None and flow > max_flow:
                return
