"""BcComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.components.display.xms_display_message import DrawType
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream

# 4. Local modules
from xms.adcirc.components import bc_component_display as bc_disp
from xms.adcirc.components.adcirc_component import AdcircComponent
from xms.adcirc.data.bc_data import BcData
from xms.adcirc.data.bc_data_manager import BcDataManager
from xms.adcirc.data.mapped_bc_data import MAPPED_BC_MAIN_FILE
from xms.adcirc.data.xms_data import XmsData
from xms.adcirc.dmi.bc_component_queries import BcComponentQueries
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.bc_mapper import BcMapper


class BcComponent(AdcircComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = BcData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [
            ('Forcing Options...', 'open_forcing_options'),
            ('Display Options...', 'open_display_options'),
        ]
        self.arc_commands = [('Assign BC...', 'open_assign_bc')]
        self.point_commands = [('Assign Pipe...', 'open_assign_pipe')]

        comp_dir = os.path.dirname(self.main_file)
        self.disp_opts_files = [
            os.path.join(comp_dir, bc_disp.BC_JSON),
            os.path.join(comp_dir, bc_disp.BC_POINT_JSON),
        ]
        display_helper = bc_disp.BcComponentDisplay(self)
        display_helper.ensure_display_option_files_exist()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the the component is locked for editing. Do not change the files
                if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_options_action()]
        return messages, action_requests

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.
                - messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            helper = BcDataManager(self)
            helper.update_display_options_file(new_main_file, new_path)
        elif save_type == 'UNLOCK':
            self.data.migrate_relative_paths()

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Should contain coverage component id maps as dumped by XMS.

        Returns:
            Empty message and ActionRequest lists

        """
        helper = bc_disp.BcComponentDisplay(self)
        helper.get_initial_display_options(query, params)
        return [], []

    def open_assign_bc(self, query, params, win_cont):
        """Opens the Assign BC dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        helper = bc_disp.BcComponentDisplay(self, query)
        return helper.assign_bc(params, win_cont)

    def open_assign_pipe(self, query, params, win_cont):
        """Opens the Assign Pipe dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        helper = bc_disp.BcComponentDisplay(self)
        return helper.assign_pipe(params, win_cont)

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`):
            params (:obj:`list` of :obj:`str`):
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        helper = bc_disp.BcComponentDisplay(self)
        return helper.display_options(win_cont, DrawType.draw_at_ids, self.cov_uuid)

    def open_forcing_options(self, query, params, win_cont):
        """Shows the tidal and flow forcing options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`):
            params (:obj:`list` of :obj:`str`):
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        helper = bc_disp.BcComponentDisplay(self)
        return helper.forcing_options(win_cont, query)

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list[tuple]`): tuple containing:

                main_file (:obj:`str`): The absolute path to the main file of the old component this
                component is being merged from.

                id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'. Each value is a tuple
                that may have two absolute file paths or none. The first file is for the ids in XMS on the coverage.
                The second file contains the ids the old component used for those objects. Both id files should be
                equal in length. This dictionary is only applicable if the component derives from CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        helper = BcDataManager(self)
        return helper.handle_merge(merge_list)

    def copy_external_files(self, new_main_file):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_main_file (:obj:`str`): The location of the new component main file in the package.


        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        new_data = BcData(new_main_file)
        new_data.copy_external_files()
        new_data.commit()
        return ''

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (:obj:`str`): The location of the new main file.
            convert_filepaths (:obj:`bool`): False if only the project directory should be updated.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        helper = BcDataManager(self)
        return helper.update_proj_dir(new_main_file, convert_filepaths)

    def apply_to_sim(self, query, sim_duration):
        """Map the BC arcs and their attributes to a mesh linked to a simulation.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            sim_duration (float): Duration of simulation in days.
        """
        XmLog().instance.info('Creating directory in SMS temporary folder for applied boundary condition data.')
        # Make a new folder for the mapped component next to ours
        comp_dir = os.path.join(os.path.dirname(os.path.dirname(self.main_file)), str(uuid.uuid4()))
        os.makedirs(comp_dir, exist_ok=True)
        new_main_file = os.path.join(comp_dir, MAPPED_BC_MAIN_FILE)

        # Query XMS for data required for mapping
        XmLog().instance.info('Querying SMS for data required to apply boundary conditions.')
        query_helper = BcComponentQueries(self, query)
        xms_data = query_helper.get_xms_data_for_mapping()
        if LogEchoQSignalStream.logged_error:
            query_helper.unlink_bc_cov_on_error()
            return

        # Perform the mapping
        try:
            XmLog().instance.info('Performing the mapping operation.')
            # import cProfile, pstats, io
            # pr = cProfile.Profile()
            # pr.enable()

            from xms.adcirc.mapping.bc_mapping_manager import BcMappingManager  # Avoid circular dependency.
            manager = BcMappingManager(query=query, sim_uuid=xms_data['sim_uuid'])
            mapper = BcMapper(
                new_main_file, self, xms_data['mesh'], xms_data['cov'], xms_data['flow_geoms'], xms_data['flow_amps'],
                xms_data['flow_phases'], manager.is_main, sim_duration, XmsData(query=query)
            )
            mapped_bc_comp, flow_comp = mapper.map_data()

            # pr.disable()
            # s = io.StringIO()
            # sortby = 'cumulative'
            # ps = pstats.Stats(pr, stream=s).sort_stats(sortby)
            # ps.print_stats()
            # XmLog().instance.debug(f'\n************PROFILE*************\n{s.getvalue()}\n***************************\n')

            if LogEchoQSignalStream.logged_error:
                mapped_bc_comp = None  # Don't add the mapped component if we raised an error.
            # Add the new component to the Query to be sent to XMS.
            XmLog().instance.info('Preparing applied boundary condition data for loading into SMS.')
            query_helper.add_xms_data_for_mapping(
                xms_data, mapped_bc_comp, flow_comp, mapper.mapped_comp.get_display_options(), manager.delete_uuids
            )
        except Exception:
            query_helper.unlink_bc_cov_on_error()
            XmLog().instance.exception('Boundary conditions could not be applied to the ADCIRC simulation.')

    def clean_unused_comp_ids(self):
        """Cleanup unused component ids from the data.

        Notes:
            You must load all component ids before calling this method.
        """
        # Cleanup BC points
        used_comp_ids = self.comp_to_xms.get(self.cov_uuid, {}).get(TargetType.point, {}).keys()
        used_comp_ids = np.array(list(used_comp_ids))
        mask = self.data.pipes.comp_id.isin(used_comp_ids)
        self.data.pipes = self.data.pipes.where(mask, drop=True)
        # Cleanup BC arcs
        used_comp_ids = self.comp_to_xms.get(self.cov_uuid, {}).get(TargetType.arc, {}).keys()
        used_comp_ids = np.array(list(used_comp_ids))
        mask = self.data.arcs.comp_id.isin(used_comp_ids)
        self.data.arcs = self.data.arcs.where(mask, drop=True)
        # Cleanup levee table
        mask = self.data.levees.comp_id.isin(used_comp_ids)
        self.data.levees = self.data.levees.where(mask, drop=True)
        self.data.commit()
