"""MappedBcComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.dialogs.message_box import message_with_ok, message_with_ok_cancel
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.adcirc.components import bc_component_display as bc_disp
from xms.adcirc.components.adcirc_component import AdcircComponent
from xms.adcirc.data.mapped_bc_data import MappedBcData
from xms.adcirc.data.mapped_bc_data_manager import MappedBcDataManager
from xms.adcirc.feedback.bc_unmapping_worker_thread import BcUnMappingWorkerThread
from xms.adcirc.model_checks.model_checker import grid_files_match


class MappedBcComponent(AdcircComponent):
    """A hidden Dynamic Model Interface (DMI) component for the mapped BC coverage.

    Note: A copy of the source BC data file must be present in this component instance's directory
          before the instance is constructed for the first time.

    """
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = MappedBcData(self.main_file)
        self.tree_commands = [
            ('Boundary Conditions...', 'view_boundary_conditions'), ('Display Options...', 'open_display_options'),
            ('Convert to Coverage', 'convert_to_coverage')
        ]
        self.disp_opts_files = [
            os.path.join(os.path.dirname(self.main_file), bc_disp.BC_JSON),
        ]

    def _mesh_is_out_of_date(self, query, parent):
        """Check if the linked mesh has been edited since we were mapped.

        Args:
            query (:obj:`Query`): The XMS interprocess communicator
            parent (:obj:`QObject`): The Qt parent window

        Returns:
            (:obj:`data_objects.parameters.UGrid`):
                The linked UGrid data_object if it is not out of date or the user chose to
                continue anyway.

                None if the mesh was out of date and the user canceled.
        """
        try:
            from xms.adcirc.dmi.mapped_bc_component_queries import MappedBcComponentQueries
            query_helper = MappedBcComponentQueries(self, query)
            comp_item = tree_util.find_tree_node_by_uuid(query.project_tree, self.uuid)
            do_ugrid = query_helper.get_linked_do_ugrid(comp_item)
            cogrid_file = do_ugrid.cogrid_file
        except Exception:
            msg = "Cannot retrieve the simulation's linked mesh. Operation has been aborted."
            message_with_ok(parent=parent, message=msg, app_name='SMS', icon='Critical', win_icon=QIcon(get_xms_icon()))
            return None  # No linked mesh?

        if not grid_files_match(cogrid_file, self.data.info.attrs['grid_crc']):
            msg = 'The mesh currently linked to the simulation has been edited since these applied boundary ' \
                  'conditions were created. The operation may produce an invalid result. See ' \
                  'https://www.xmswiki.com/wiki/SMS:ADCIRC for more information. Would you like to continue?'
            if not message_with_ok_cancel(
                parent=parent, message=msg, app_name='SMS', icon='Question', win_icon=QIcon(get_xms_icon())
            ):
                return None  # User canceled, abort the operation
        return do_ugrid

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK:

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

            new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
            if necessary.

            messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
            tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
            text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            helper = MappedBcDataManager(self)
            action_requests.append(helper.update_display_options_file(new_main_file, new_path))
        elif save_type in ['SAVE_AS', 'PACKAGE']:
            action_requests.append(self.get_display_options_action())
        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        # Send the default display options list to XMS.
        self.display_option_list = [
            XmsDisplayMessage(file=self.disp_opts_files[0], draw_type=DrawType.draw_at_locations),  # arcs
        ]
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`):
            params (:obj:`list` of :obj:`str`):
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        helper = bc_disp.BcComponentDisplay(self)
        return helper.display_options(win_cont, DrawType.draw_at_locations, '')  # May want to tie list to mesh

    def view_boundary_conditions(self, query, params, win_cont):
        """Display the Assign BC dialog and persist data if accepted.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): The ActionRequest parameter map
            win_cont (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        helper = bc_disp.BcComponentDisplay(self)
        return helper.view_mapped_bc(query, win_cont)

    def convert_to_coverage(self, query, params, win_cont):
        """Convert a mapped BC coverage object to a source BC coverage.

        Note that this has not been implemented yet. Here for demonstration and testing purposes while I
        was adding the functionality to xms.api. Pretty much the only thing correct here is the Query::Add
        calls. Must send coverage and hidden components to ModelBuilder.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): The ActionRequest parameter map
            win_cont (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform..
        """
        do_ugrid = self._mesh_is_out_of_date(query, win_cont)
        if not do_ugrid:
            return [], []
        worker = BcUnMappingWorkerThread(query, self, do_ugrid, win_cont)
        display_text = {
            'title': 'Converting Applied Boundary Conditions',
            'working_prompt': 'Converting applied boundary conditions to a coverage. Please wait...',
            'error_prompt':
                'Error(s) encountered converted applied boundary conditions to a coverage. Review log '
                'output for more details.',
            'warning_prompt':
                'Warning(s) encountered converting applied boundary conditions to a coverage. Review '
                'log output for more details.',
            'success_prompt':
                'Successfully converted applied boundary conditions to an ADCIRC Boundary Conditions '
                'coverage. Close this dialog to load the coverage into SMS Map Module.',
            'note':
                'Converting applied ADCIRC boundary conditions to an ADCIRC Boundary Conditions coverage in the '
                'Map Module.',
            'auto_load': 'Auto load converted coverage into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        feedback_dlg.exec()
        return [], []
