"""Reader for ADCIRC fort.15 control files."""

# 1. Standard Python modules
import copy
import datetime
import math
import os
import re
import sys
import uuid

# 2. Third party modules
import h5py
import numpy as np
from PySide2.QtCore import QDate, QDateTime, QTime
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.data_objects.parameters import Coverage, Dataset, datetime_to_julian, Point, Simulation
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime
from xms.tides.data import tidal_data as tid

# 4. Local modules
from xms.adcirc.components.adcirc_component import get_component_data_object
from xms.adcirc.data import mapped_flow_data as mfd
from xms.adcirc.data import mapped_tidal_data as mtd
from xms.adcirc.data import sim_data as sid
from xms.adcirc.data import station_data as std
from xms.adcirc.data.adcirc_data import file_exists
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort13_reader import Fort13Reader
from xms.adcirc.file_io.fort14_reader import Fort14Reader
from xms.adcirc.file_io.fort20_reader import Fort20Reader
from xms.adcirc.file_io.fort22_reader import Fort22Reader
from xms.adcirc.gui import gui_util

DEFAULT_FILENAMES = {13: "fort.13", 14: "fort.14", 15: "fort.15", 19: "fort.19", 20: "fort.20", 22: "fort.22"}


class DataForThe14:
    """Container for holding values read from the fort.15 that are needed by the fort.14 reader."""
    def __init__(self):
        """Constructs the container."""
        self.mesh_uuid = "CCCCCCCC-CCCC-CCCC-CCCC-CCCCCCCCCCCC"
        self.vert_units = "METERS"  # Determined by gravitational constant, needs to be passed to the fort.14 reader.
        self.geo_coords = True  # If ICS = 2, the fort.14 reader needs to build the mesh with a geographic projection.
        self.mesh_name = ""
        self.comp_dir = ''


class DataForThe22:
    """Container for holding values read from the fort.15 that are needed by the fort.22 reader."""
    def __init__(self):
        """Constructs the container."""
        self.nws_type = 0
        self.reftime = 0.0  # Need to pass this to the fort.22 reader when NWS=3
        self.nwlat = 0
        self.nwlon = 0
        self.latmax = 0.0
        self.lonmin = 0.0
        self.latinc = 0.0
        self.loninc = 0.0
        self.wtiminc = 1.0
        self.numnodes = 0
        self.vert_units = "METERS"
        self.mesh_uuid = ''
        self.grid_uuid = ''
        self.cov_uuid = ''
        self.dset_uuids = []
        self.projection = None
        self.temp_dir = None


class Fort15Reader:
    """Reads an ADCIRC fort.15 (control) file. Model parameters and boundary conditions."""
    def __init__(
        self,
        filename,
        query,
        comp_dir,
        global_time,
        geom_uuid=None,
        sim_name=None,
        storm_track_uuid=None,
        alt_filenames=None,
        overwrite_mc_widgets=None,
        read_project_dsets=False
    ):
        """Initializes the reader. The fort.15 is the starting point for a model-native read.

        Notes:
            Reading a fort.15 could potentially trigger the read of several other files. The reading
            of some of the optional input files will trigger the read of the fort.15 because of
            cross-file dependencies.

        Args:
            filename (:obj:`str`): Full path and filename of the fort.15 file.
            query (:obj:`Query`): Query for communicating with SMS.
            comp_dir (:obj:`str`): Filesystem path to the XMS component temp directory for creating new components.
            global_time (:obj:`QDateTime`): Time to use as reference date if not in ADCIRC input files.
            geom_uuid (:obj:`Optional[str]`): UUID of the ADCIRC mesh, random by default
            sim_name (:obj:`str`): Name to assign the simulation. Only defined when migrating pre-DMI SMS projects.
            storm_track_uuid (:obj:`Optional[str]`) The UUID of the NWS=8,19,20 storm track coverage. If not specified,
                the coverage is read from the fort.22 file.
            alt_filenames (:obj:`Optional[dict]`): dict where key is fort extension (e.g. "fort.14" -> 14) and the
                value is the fort input file relative from the read directory. If not specified, the traditional
                ADCIRC filenames will be assumed in the current working directory.
            overwrite_mc_widgets (:obj:`Optional[dict]`): Use to override values read from the fort.15. Used for project
                migration.
            read_project_dsets (:obj:`Optional[bool]`): If True, script will read all datasets in the project
                "_datasets" folder that belong to the ADCIRC mesh. Used when migrating an old ADCIRC custom
                interface project.
        """
        self.filename = filename
        self.comp_dir = comp_dir
        self.query = query
        self.global_time = global_time
        self.sim_name = sim_name
        self.sim_uuid = str(uuid.uuid4())
        self.lines = None
        self.curr_line = 0
        self.num_ocean_nodes = 0  # Read from the fort.14
        self.num_river_nodes = 0  # Read from the fort.14

        # New component datas we will be building
        self.tidal_data = None  # Created when we read tidal forcing parameters
        self.bc_data = None  # Created by the fort.14 reader
        self.sim_data = None  # Created when we add the hidden simulation component

        self.tidal_potential = {}  # {name: (amp, freq, etrf, nf, eq_arg)}
        self.fort22_data = DataForThe22()
        self.fort14_data = DataForThe14()
        self.fort14_data.comp_dir = self.comp_dir
        self.mapped_flow_data = None  # Only created if we have periodic flow boundaries
        # UUID of the mesh needs to be passed to the fort.13, fort.14, and fort.22 readers.
        self.fort14_data.mesh_uuid = geom_uuid if geom_uuid is not None else str(uuid.uuid4())
        # If a storm track coverage uuid is specified, an existing coverage will be linked to the simulation as
        # opposed to being read from the fort.22.
        self.storm_track_uuid = storm_track_uuid

        # Read wind data if it exists. Need the reader variable to stay in scope until after the Query::Send. If
        # building a wind coverage dump, we need it to stay in memory until the API writes the dump file, which
        # happens when packaging up the data to send to XMS. Dumb of the API, but whatever.
        self.fort22reader = None
        self.read13 = False
        self.read19 = False
        self.read20 = False
        self.read22 = False
        self.read23 = False
        self.read24 = False
        self.read_netcdf_options = False
        self.read_ref_date = False  # True if we read a reference datetime for wind track or wind grid NWS types
        self.read_project_dsets = read_project_dsets
        self.alt_filenames = alt_filenames if alt_filenames else DEFAULT_FILENAMES  # Use to override default filenames
        self.overwrite_mc_widgets = {}
        if overwrite_mc_widgets:
            self.overwrite_mc_widgets = overwrite_mc_widgets  # Use to add or override values in the fort.15
        self.project_dataset_file = None
        self.current_project_file = ""
        self.built_data = {'proj_dsets': []}  # Imported data to send back to SMS
        self.out_filenames = []  # For testing so we don't have to use Query

    def _get_line(self):
        """Get the next line in the fort.15 file."""
        try:
            line = self.lines[self.curr_line]
            self.curr_line += 1
            return line
        except IndexError:
            return 'x'

    def _split_line(self):
        """Get the next fort.15 line split on whitespace."""
        return self._get_line().split()

    def _get_first_token(self):
        """Get the first element of the next fort.15 line split on whitespace (as a str)."""
        return self._split_line()[0]

    def _get_first_int(self):
        """Get the next line of the fort.15 and extract the first int value on the line."""
        return int(self._get_first_token())

    def _get_first_float(self):
        """Get the next line of the fort.15 and extract the first float value on the line."""
        return float(self._get_first_token())

    def set_mesh_name(self, mesh_name):
        """Set the name that should be assigned to the imported ADCIRC mesh.

        Args:
            mesh_name (:obj:`str`): Name of the ADCIRC mesh
        """
        self.fort14_data.mesh_name = mesh_name

    def _create_sim_data_component(self):
        """Create the hidden simulation component with default data."""
        # Add the hidden simulation component
        comp_uuid = str(uuid.uuid4())
        sim_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(sim_comp_dir, exist_ok=True)
        sim_mainfile = os.path.join(sim_comp_dir, sid.SIM_COMP_MAIN_FILE)
        self.sim_data = sid.SimData(sim_mainfile)
        self.sim_data.timing.attrs['ref_date'] = qdatetime_to_datetime(self.global_time).strftime(ISO_DATETIME_FORMAT)
        self.built_data['sim_comp'] = get_component_data_object(sim_mainfile, comp_uuid, 'Sim_Component')

    def _create_tidal_data_component(
        self, ocean_node_ids, con_names, con_names_for_props, potential_amplitude, frequencies, nodal_factors, eq_args,
        potential_etrf, amplitudes, phases
    ):
        """Create a mapped component for periodic flow boundaries.

        Args:
            ocean_node_ids (:obj:`list`): List of the ocean boundary 1-based node ids in the order they
                appear in the fort.14
            con_names (:obj:`list`): List of the tidal forcing constituent names
            con_names_for_props (:obj:`list`): List of the tidal forcing and potential constituent names. Should be the
                same, but historical files may be mismatched.
            potential_amplitude (:obj:`list`): List of the tidal constituent potential amplitudes
            frequencies (:obj:`list`): List of the tidal constituent frequencies
            nodal_factors (:obj:`list`): List of the tidal constituent nodal factors
            eq_args (:obj:`list`): List of the tidal constituent equilibrium arguments
            potential_etrf (:obj:`list`): List of the tidal constituent earth tide reduction factors
            amplitudes (:obj:`list`): List of the tidal boundary node amplitudes
            phases (:obj:`list`): List of the tidal boundary node phases
        """
        # Create a folder for the new mapped tidal component.
        comp_uuid = str(uuid.uuid4())
        mapped_tidal_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(mapped_tidal_dir)
        tidal_mainfile = os.path.join(mapped_tidal_dir, mtd.MAPPED_TIDAL_MAIN_FILE)
        self.tidal_data = mtd.MappedTidalData(tidal_mainfile)

        con_props_data = {
            'amplitude': ('con', potential_amplitude),
            'frequency': ('con', frequencies),
            'nodal_factor': ('con', nodal_factors),
            'equilibrium_argument': ('con', eq_args),
            'earth_tide_reduction_factor': ('con', potential_etrf),
        }
        con_props_coords = {'con': con_names_for_props}
        self.tidal_data.cons = xr.Dataset(data_vars=con_props_data, coords=con_props_coords).sortby('con')

        con_data = {
            'amplitude': (('con', 'node_id'), amplitudes),
            'phase': (('con', 'node_id'), phases),
        }
        con_coords = {
            'con': con_names,
            'node_id': ocean_node_ids,
        }
        self.tidal_data.values = xr.Dataset(data_vars=con_data, coords=con_coords).sortby('con')
        # Create a default source tidal data file and set the extraction source to user defined. Set the
        # reference date.
        source_data = tid.TidalData(os.path.join(mapped_tidal_dir, tid.DEFAULT_TIDAL_DATA_FILE))
        source_data.info.attrs['source'] = tid.USER_DEFINED_INDEX
        source_data.info.attrs['reftime'] = qdatetime_to_datetime(self.global_time).strftime(ISO_DATETIME_FORMAT)
        self.tidal_data.commit()
        source_data.commit()
        self.built_data['tidal_comp'] = get_component_data_object(
            tidal_mainfile, comp_uuid, 'Mapped_Tidal_Component', 'Tidal Constituents (applied)'
        )

    def _create_flow_data_component(
        self, river_node_ids, flow_names, frequencies, nodal_factors, eq_args, amplitudes, phases
    ):
        """Create a mapped component for periodic flow boundaries.

        Args:
            river_node_ids (:obj:`list`): List of the river boundary 1-based node ids in
                the order they appear in the fort.14
            flow_names (:obj:`list`): List of the flow constituent names
            frequencies (:obj:`list`): List of the flow constituent frequencies
            nodal_factors (:obj:`list`): List of the flow constituent nodal factors
            eq_args (:obj:`list`): List of the flow constituent equilibrium arguments
            amplitudes (:obj:`list`): List of the flow boundary node amplitudes (unit flow rate)
            phases (:obj:`list`): List of the flow boundary node phases
        """
        # Create a new mapped flow component for periodic flow boundaries.
        flow_comp_uuid = str(uuid.uuid4())
        flow_comp_dir = os.path.join(self.comp_dir, flow_comp_uuid)
        os.makedirs(flow_comp_dir)
        flow_mainfile = os.path.join(flow_comp_dir, mfd.MAPPED_FLOW_MAIN_FILE)
        self.mapped_flow_data = mfd.MappedFlowData(flow_mainfile)
        flow_con_coords = {'con': flow_names}
        flow_con_data = {
            'frequency': ('con', np.array(frequencies, dtype=np.float64)),
            'nodal_factor': ('con', np.array(nodal_factors, dtype=np.float64)),
            'equilibrium_argument': ('con', np.array(eq_args, dtype=np.float64)),
        }
        self.mapped_flow_data.cons = xr.Dataset(data_vars=flow_con_data, coords=flow_con_coords)

        flow_value_coords = {
            'con': flow_names,
            'node_id': river_node_ids,
        }
        flow_value_data = {
            'amplitude': (('con', 'node_id'), amplitudes),
            'phase': (('con', 'node_id'), phases),
        }
        self.mapped_flow_data.values = xr.Dataset(data_vars=flow_value_data, coords=flow_value_coords).fillna(0.0)
        self.mapped_flow_data.commit()

        # Create the data_objects component
        self.built_data['flow_comp'] = get_component_data_object(
            flow_mainfile, flow_comp_uuid, 'Mapped_Flow_Component', 'Flow Constituents (applied)'
        )

    def _build_recording_station_coverage(self, mesh_name, recording_pts):
        """Create the recording stations coverage and its hidden component.

        Args:
            mesh_name (:obj:`str`): Name of the ADCIRC mesh
            recording_pts (:obj:`dict`): Dictionary where keys are stringified point x,y locations and the values are
                lists containing the elevation, velocity, and wind toggles of each key point.
        """
        # Build the station points and get all their widget values ready.
        cov_pts = [None for _ in recording_pts]
        elevs = [0 for _ in range(len(recording_pts))]
        velocities = [0 for _ in range(len(recording_pts))]
        winds = [0 for _ in range(len(recording_pts))]
        for idx, (pt_loc, toggles) in enumerate(recording_pts.items()):
            pt = Point(float(pt_loc[0]), float(pt_loc[1]), feature_id=idx + 1)
            cov_pts[idx] = pt
            elevs[idx] = toggles[0]
            velocities[idx] = toggles[1]
            winds[idx] = toggles[2]

        # Create the coverage geometry.
        cov_uuid = str(uuid.uuid4())
        cov = Coverage()
        cov.name = f'{mesh_name} - Stations'
        cov.set_points(cov_pts)
        cov.uuid = cov_uuid
        cov.projection = self.fort22_data.projection
        cov.complete()

        # Create a directory for the hidden recording station coverage component.
        comp_uuid = str(uuid.uuid4())
        station_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(station_comp_dir, exist_ok=True)
        station_mainfile = os.path.join(station_comp_dir, std.STATION_MAIN_FILE)

        # Write the component data main file.
        station_dict = {
            'elevation': ('comp_id', np.array(elevs, dtype=np.int32)),
            'velocity': ('comp_id', np.array(velocities, dtype=np.int32)),
            'wind': ('comp_id', np.array(winds, dtype=np.int32)),
        }
        coords = {'comp_id': [idx for idx in range(len(elevs))]}
        station_data = std.StationData(station_mainfile)
        station_data.stations = xr.Dataset(data_vars=station_dict, coords=coords)
        station_data.info.attrs['next_comp_id'] = len(elevs)
        station_data.info.attrs['cov_uuid'] = cov_uuid
        station_data.info.attrs['native_import'] = 1  # Need to set component ids when XMS initializes display.
        station_data.commit()
        self.built_data['stations'] = cov, get_component_data_object(station_mainfile, comp_uuid, 'Station_Component')

    def _read_run_options(self):
        """Reads the general run options in the fort.15 (RUNDES - IDEN)."""
        self.sim_data.general.attrs['RUNDES'] = self._get_line()[:32].strip()  # (32 character max)
        self.sim_data.general.attrs['RUNID'] = self._get_line()[:24].strip()  # (24 character max)
        self.sim_data.general.attrs['NFOVER'] = self._get_first_int()
        self.sim_data.general.attrs['NABOUT'] = self._get_first_int()
        self.sim_data.general.attrs['NSCREEN'] = self._get_first_int()
        ihot = self._get_first_int()
        self.sim_data.general.attrs['IHOT'] = ihot
        if self.sim_data.general.attrs['IHOT'] != 0:
            # Link the hot start file to the file selector widget
            file_name = ""
            if ihot == 17:
                file_name = "fort.17"
            elif ihot == 67:
                file_name = "fort.67"
            elif ihot == 68:
                file_name = "fort.68"
            elif ihot == 367:
                file_name = "fort.67.nc"
            elif ihot == 368:
                file_name = "fort.68.nc"
            elif ihot == 567:
                file_name = "fort.67.nc"
            elif ihot == 568:
                file_name = "fort.68.nc"
            file_name = os.path.join(os.getcwd(), file_name)
            if file_exists(file_name):
                self.sim_data.general.attrs['IHOT_file'] = file_name
        if self._get_first_int() != 2:  # ICS - if == 2 geographic coordinates
            self.fort14_data.geo_coords = False
        im = self._get_first_int()
        self.sim_data.formulation.attrs['IM'] = im
        if im == 21:  # Baroclinic 3D
            # Read IDEN if IC == Baroclinic 3D
            self.sim_data.formulation.attrs['IDEN'] = self._get_first_int()

    def _read_friction_options(self):
        """Reads the friction options in the fort.15 (NOLIBF - NWP).

        Also will set the flag to read the fort.13 if any nodal attributes have been specified.
        """
        # Don't set cbxNOLIBF value yet. Although the export text is the same, the combobox options for friction
        # nodal attributes are different. We'll find out if we're using one later.
        self.sim_data.general.attrs['NOLIBF'] = self._get_first_int()
        self.sim_data.formulation.attrs['NOLIFA'] = self._get_first_int()
        self.sim_data.formulation.attrs['NOLICA'] = self._get_first_int()
        self.sim_data.formulation.attrs['NOLICAT'] = self._get_first_int()

        nwp = self._get_first_int()  # NWP
        self.sim_data.formulation.attrs['NWP'] = nwp
        for _ in range(nwp):
            att_line = self._get_first_token()
            # Listing the nodal attributes here is somewhat redundant. All this data is in the fort.13, but it is
            # convenient to setup toggles
            if "surface_submergence_state" == att_line:
                self.sim_data.nodal_atts.attrs['surface_submergence_state_on'] = 1
            elif "surface_directional_effective_roughness_length" == att_line:
                self.sim_data.nodal_atts.attrs['surface_directional_effective_roughness_length_on'] = 1
            elif "surface_canopy_coefficient" == att_line:
                self.sim_data.nodal_atts.attrs['surface_canopy_coefficient_on'] = 1
            elif "sea_surface_height_above_geoid" == att_line:
                self.sim_data.nodal_atts.attrs['sea_surface_height_above_geoid_on'] = 1
            elif "average_horizontal_eddy_viscosity_in_sea_water_wrt_depth" == att_line:
                self.sim_data.nodal_atts.attrs['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on'] = 1
            elif "wave_refraction_in_swan" == att_line:
                self.sim_data.nodal_atts.attrs['wave_refraction_in_swan_on'] = 1
            elif "bridge_pilings_friction_paramenters" == att_line:
                self.sim_data.nodal_atts.attrs['bridge_pilings_friction_paramenters_on'] = 1
            elif "elemental_slope_limiter" == att_line:
                self.sim_data.nodal_atts.attrs['elemental_slope_limiter_on'] = 1
            elif "advection_state" == att_line:
                self.sim_data.nodal_atts.attrs['advection_state_on'] = 1
            elif "initial_river_elevation" == att_line:
                self.sim_data.nodal_atts.attrs['initial_river_elevation_on'] = 1
            elif "quadratic_friction_coefficient_at_sea_floor" == att_line:
                self.sim_data.general.attrs['NOLIBF'] = 3
            elif "mannings_n_at_sea_floor" == att_line:
                self.sim_data.general.attrs['NOLIBF'] = 4
            elif "chezy_friction_coefficient_at_sea_floor" == att_line:
                self.sim_data.general.attrs['NOLIBF'] = 5
            elif "bottom_roughness_length" == att_line:
                self.sim_data.general.attrs['NOLIBF'] = 6
        if nwp > 0:  # Read the fort.13 if nodal attributes were listed.
            self.read13 = True

    def _read_nws_type(self):
        """Reads the NWS type and a couple of lines preceeding it (NCOR - NWS)."""
        self.sim_data.general.attrs['NCOR'] = self._get_first_int()
        # TODO: If NTIP = 2, need to read a fort.24.
        self.read24 = self._get_first_int() == 2
        self.fort22_data.nws_type = self._get_first_int()
        # In python, modulo of negative numbers is not what you expect.
        nws_simple = abs(self.fort22_data.nws_type) % 100
        self.sim_data.wind.attrs['NWS'] = nws_simple
        if self.fort22_data.nws_type < 0:
            self.sim_data.wind.attrs['first_matches_hot_start'] = 1
        if nws_simple not in [0, 10, 11]:  # No fort.22 to read
            self.read22 = True
        elif nws_simple == 10:
            self.sim_data.nws10_files = self._gather_wind_files('AVN File')
            if self.sim_data.nws10_files.sizes['dim_0'] == 0:
                file_exists('No AVN wind files found')
        elif nws_simple == 11:
            self.sim_data.nws11_files = self._gather_wind_files('ETA File')
            if self.sim_data.nws11_files.sizes['dim_0'] == 0:
                file_exists('No ETA wind files found')
        # NWS=7 type is not supported, change to a NWS=6
        if nws_simple == 7:
            nws_simple = 6
        # Don't read fort.22 if wind track coverage is already in the map file (custom project migration)
        if nws_simple in [8, 19, 20] and self.storm_track_uuid:
            self.read22 = False

    def _gather_wind_files(self, variable_name):
        """Add all fort.xxx files in the fort.15 file's directory. (NWS=10,11).

        Args:
            variable_name (:obj:`str`): Name of the variable to fill in the dataset

        Returns:
            (:obj:`xarray.Dataset`): Dataset with passed in variable's array filled with NWS=10,11 met files
        """
        filenames = []
        model_path = os.path.dirname(self.filename)
        if not model_path:
            model_path = os.getcwd()  # fort.15 filename passed as relative path. cwd is location of import files.
        pattern = re.compile(r'fort\.[0-9][0-9][0-9]')
        for _, _, files in os.walk(model_path):
            for file in files:  # Loop through the files of the top directory, no recurse.
                match = re.fullmatch(pattern, file)
                if match:
                    filenames.append(os.path.normpath(os.path.join(model_path, file)))
        # Build a dataset of the met files in the fort.15's directory
        return xr.Dataset(data_vars={variable_name: xr.DataArray(data=np.array(filenames, dtype=object))})

    def _parse_wtiminc(self):
        """Parse the WTIMINC line based on the NWS type.

        The WTIMINC line only exists if NWS > 1
        """
        # Determine if we are using wave radiation and/or ice forcing. Also find out if this SWAN or STWAVE coupled.
        positive_nws = abs(self.fort22_data.nws_type)
        if positive_nws < 100:  # meteorological forcing only
            use_ice = False
            self.read23 = False  # defined in a fort.23 file
            swan_coupled = False
            stwave_coupled = False
        elif positive_nws < 12000:  # no ice
            use_ice = False
            if positive_nws < 300:  # Wave radiation and meteorological forcing
                self.read23 = True
                swan_coupled = False
                stwave_coupled = False
            else:
                self.read23 = False
                if positive_nws < 400:  # SWAN coupled with meteorological forcing
                    swan_coupled = True
                    stwave_coupled = False
                else:  # STWAVE coupled with meteorological forcing
                    stwave_coupled = True
                    swan_coupled = False
        else:  # using ice
            use_ice = True
            if positive_nws < 12100:  # Ice and meteorological forcing
                self.read23 = False
                swan_coupled = False
                stwave_coupled = False
            elif positive_nws < 12300:  # Ice, wave radiation, and meteorological forcing
                self.read23 = True
                swan_coupled = False
                stwave_coupled = False
            else:
                self.read23 = False
                if positive_nws < 12400:  # SWAN coupled with ice and meteorological forcing
                    swan_coupled = True
                    stwave_coupled = False
                else:  # STWAVE coupled with ice and meteorological forcing
                    swan_coupled = False
                    stwave_coupled = True

        nws_simple = positive_nws % 100
        vals = self._split_line()
        rstiminc = None
        wtiminc = None
        icetiminc = None
        if nws_simple in [1, 11]:  # no wtiminc, but maybe others
            rstiminc = float(vals[0])
            if use_ice:  # Not available for NWS=1 types
                icetiminc = float(vals[1])
        elif nws_simple in [2, 4, 5, 7, 10, 12, 15, 16]:
            wtiminc = float(vals[0])  # First value is WTIMINC
            if self.read23 or swan_coupled or stwave_coupled:  # Second value is RSTIMINC if used
                rstiminc = float(vals[1])  # This is a float (sec)
                if use_ice:  # Third value is CICE_TIMINC if used
                    icetiminc = float(vals[2])
            elif use_ice:  # Second value is CICE_TIMINC if used
                icetiminc = float(vals[1])
        elif nws_simple in [8, 19, 20]:  # Wind track types
            # Set the date here as the interpolation reference date. It is not necessarily the beginning of the
            # storm track.
            year = int(vals[0])
            month = int(vals[1])
            day = int(vals[2])
            hour = int(vals[3])
            qdate = QDate(year, month, day)
            qtime = QTime(hour, 0, 0)
            ref_date = QDateTime(qdate, qtime)
            self.sim_data.timing.attrs['ref_date'] = qdatetime_to_datetime(ref_date).strftime(ISO_DATETIME_FORMAT)
            self.read_ref_date = True
            try:
                self.sim_data.wind.attrs['bladj'] = float(vals[5])
            except ValueError:
                pass  # default if value not present
            idx = 6
            if nws_simple == 20:  # extra value for NWS=20
                self.sim_data.wind.attrs['geofactor'] = int(vals[idx])
                idx += 1

            if self.read23 or swan_coupled or stwave_coupled:
                rstiminc = float(vals[idx])
                idx += 1
                if use_ice:
                    icetiminc = float(vals[idx])
            elif use_ice:
                icetiminc = float(vals[idx])
        elif nws_simple in [3, 6]:
            if nws_simple == 3:
                # parse the reference date
                year = int(vals[0])
                month = int(vals[1])
                day = int(vals[2])
                hour = int(vals[3])
                minute = int(vals[4])
                # There are files in the wild where this is a float. Not sure if ADCIRC uses the precision or not.
                sec = int(float(vals[5]))
                # This is presumptuous and stupid, but not as much as using two digit years in your file format.
                if year > 50:
                    year += 1900
                else:
                    year += 2000
                # Create a data_objects DateTimeLiteral for converting to Julian datetime using our standard method.
                ref_date = datetime.datetime(year=year, month=month, day=day, hour=hour, minute=minute, second=sec)
                self.sim_data.timing.attrs['ref_date'] = ref_date.strftime(ISO_DATETIME_FORMAT)
                self.read_ref_date = True
                self.fort22_data.reftime = datetime_to_julian(ref_date)
                if self.read23 or swan_coupled or stwave_coupled:
                    rstiminc = float(vals[6])
                    if use_ice:
                        icetiminc = float(vals[7])
                elif use_ice:
                    icetiminc = float(6)
                vals = self._split_line()
            # Parse the grid definition
            self.fort22_data.nwlat = int(vals[0])
            self.fort22_data.nwlon = int(vals[1])
            self.fort22_data.latmax = float(vals[2])
            self.fort22_data.lonmin = float(vals[3])
            self.fort22_data.latinc = float(vals[4])
            self.fort22_data.loninc = float(vals[5])
            wtiminc = float(vals[6])
            if nws_simple == 6:
                if self.read23 or swan_coupled or stwave_coupled:
                    rstiminc = float(vals[7])
                    if use_ice:
                        icetiminc = float(vals[8])
                elif use_ice:
                    icetiminc = float(vals[7])
        elif nws_simple == 7:
            pass

        # add the widget values in the appropriate places.
        if wtiminc:
            self.sim_data.wind.attrs['WTIMINC'] = wtiminc
            self.fort22_data.wtiminc = wtiminc
        if rstiminc:
            self.sim_data.wind.attrs['RSTIMINC'] = rstiminc
            if swan_coupled:
                self.sim_data.wind.attrs['wave_radiation'] = 300  # SWAN coupled
            elif stwave_coupled:
                self.sim_data.wind.attrs['wave_radiation'] = 400  # STWAVE coupled
            else:
                self.sim_data.wind.attrs['wave_radiation'] = 100  # Forcing file
        if icetiminc:
            self.sim_data.wind.attrs['use_ice'] = 1
            self.sim_data.wind.attrs['CICE_TIMINC'] = icetiminc

    def _read_ramp_and_time(self):
        """Read the ramp and run time options (NRAMP - DRAMP).

        TAU0 also gets read in here
        """
        nramp = self._get_first_int()
        self.sim_data.timing.attrs['NRAMP'] = nramp
        gravity = self._get_first_float()  # G
        if math.isclose(gravity, 32.2):  # Set vertical units for geometry/mesh projection based on G constant.
            self.fort14_data.vert_units = "FEET (INTERNATIONAL)"
            self.fort22_data.vert_units = "FEET (INTERNATIONAL)"
        tau0 = self._get_first_float()  # TAU0 - 1 in the tenths place changes ADCIRC output
        if 0.0 < tau0 < 1.0:
            self.sim_data.formulation.attrs['TAU0'] = 1
            self.sim_data.formulation.attrs['TAU0_specified'] = tau0
        else:
            self.sim_data.formulation.attrs['TAU0'] = int(tau0)
            if self.sim_data.formulation.attrs['TAU0'] == -5:
                # The next line is the domain. Only if TAU0 = -5
                line = self._split_line()
                self.sim_data.formulation.attrs['Tau0FullDomainMin'] = float(line[0])
                self.sim_data.formulation.attrs['Tau0FullDomainMax'] = float(line[1])

        self.sim_data.timing.attrs['DTDP'] = self._get_first_float()
        self.fort22_data.wtiminc = self.sim_data.timing.attrs['DTDP']  # Defaulting WTIMINC to DTDP for NWS = 1
        self.curr_line += 2  # STATIM and REFTIM - only exists for historical reasons in ADCIRC
        if abs(self.fort22_data.nws_type) > 1 and abs(self.fort22_data.nws_type) != 11:
            self._parse_wtiminc()
        self.sim_data.timing.attrs['RUNDAY'] = self._get_first_float()

        line = self._split_line()  # DRAMP
        ramp_vars = [
            'DRAMP', 'DRAMPExtFlux', 'FluxSettlingTime', 'DRAMPIntFlux', 'DRAMPElev', 'DRAMPTip', 'DRAMPMete',
            'DRAMPWRad', 'DUnRampMete'
        ]
        nramp_vals = nramp if nramp == 1 else nramp + 1
        for idx in range(nramp_vals):
            self.sim_data.timing.attrs[ramp_vars[idx]] = float(line[idx])

    def _read_coords(self):
        """Read the coordinate data (A00 - CORI).

        TAU/CF also gets read in here
        """
        line = self._split_line()
        self.sim_data.formulation.attrs['A00'] = float(line[0])
        self.sim_data.formulation.attrs['B00'] = float(line[1])
        self.sim_data.formulation.attrs['C00'] = float(line[2])
        line = self._split_line()
        self.sim_data.formulation.attrs['H0'] = float(line[0])
        if self.sim_data.formulation.attrs['NOLIFA'] > 1:
            self.sim_data.formulation.attrs['VELMIN'] = float(line[3])
        line = self._split_line()
        self.sim_data.general.attrs['SLAM0'] = float(line[0])
        self.sim_data.general.attrs['SFEA0'] = float(line[1])
        self.sim_data.general.attrs['calc_center_coords'] = 0
        line = self._split_line()
        self.sim_data.general.attrs['CF'] = float(line[0])
        if self.sim_data.general.attrs['NOLIBF'] == 2:
            self.sim_data.general.attrs['HBREAK'] = float(line[1])
            self.sim_data.general.attrs['FTHETA'] = float(line[2])
            self.sim_data.general.attrs['FGAMMA'] = float(line[3])
        read_eslm = self.sim_data.formulation.attrs['IM'] % 10 in [0, 1, 2, 10]
        if read_eslm:
            # I don't think 2 and 10 exist any more, but I'll read them just in case.
            self.sim_data.formulation.attrs['ESLM'] = self._get_first_float()
            if self.sim_data.formulation.attrs['IM'] == 10:
                # TODO: ESLC? docs say only if IM=10, Not in interface
                self.sim_data.formulation.attrs['ESLC'] = float(line[1])
        self.sim_data.general.attrs['CORI'] = self._get_first_float()

    def _read_tidal_potential(self):
        """Read the tidal potential data (NTIF and data).

        Even if the data was pulled from a database, we will recreate them as user defined because that
        information is not available in the model-native files. Tidal potential constituents are assumed
        to match the names and orders of the tidal forcing constituents
        """
        num_tidal_pots = self._get_first_int()
        for _ in range(num_tidal_pots):
            con_name = self._get_first_token().upper()
            line = self._split_line()
            self.tidal_potential[con_name] = (
                float(line[0]),  # Amplitude
                float(line[1]),  # Frequency
                float(line[2]),  # Earth tide reduction facto
                float(line[3]),  # Nodal factor
                float(line[4]),  # Equilibrium arg
            )

    def _read_tidal_forcing(self):
        """Read the tidal forcing constituents on ocean boundaries (NBFR and data).

        The fort.14 must be read before this happens. We need to know the number of ocean boundary nodes
        and their locations.

        Even if the data was pulled from a database, we will recreate it as a mapped tidal constituent component
        because that information is not available in the model-native files.

        If NBFR = 0 and the number of ocean boundary nodes > 0 (always true I think), non-periodic forcing
        will be read from the fort.19
        """
        # Create the mapped tidal constituent component
        num_cons = self._get_first_int()  # NBFR
        # Get the node ids of ocean boundary mesh nodes in the order they appear in the fort.14
        ocean_node_ids, _ = self.bc_data.get_ocean_node_ids()
        self.num_ocean_nodes = len(ocean_node_ids)
        if num_cons > 0:  # Periodic tidal forcing specified in the fort.15
            # Need a list of the tidal constituent identifiers to know their order because the identifiers in the
            # second loop do not need to match.
            con_names = ['' for _ in range(num_cons)]
            potential_amplitude = [0.0 for _ in range(num_cons)]
            frequencies = [0.0 for _ in range(num_cons)]
            nodal_factors = [0.0 for _ in range(num_cons)]
            eq_args = [0.0 for _ in range(num_cons)]
            potential_etrf = [0.0 for _ in range(num_cons)]
            # Read the forcing frequency, nodal factor, and equilibrium argument for each constituent.
            for i in range(num_cons):
                con_names[i] = self._get_first_token().upper()
                line = self._split_line()
                frequencies[i] = float(line[0])
                nodal_factors[i] = float(line[1])
                eq_args[i] = float(line[2])
                # Default tidal potential amplitude and ETRF if not read in
                # self.tidal_potential = {name: (amp, freq, etrf, nf, eq_arg)}
                potential_vals = self.tidal_potential.pop(con_names[i], (0.0, 0.0, 0.0))
                potential_amplitude[i] = potential_vals[0]
                potential_etrf[i] = potential_vals[2]

            # Add any constituents listed in tidal potential section but not in the tidal forcing section. We
            # do not think this makes any sense. AKZ has discussed this with the ADCIRC developers and they
            # sheepishly admit that the constituents in the tidal potential section must match the constituents
            # and order listed in the tidal forcing section. However, it is possible to use tidal potential without
            # any ocean boundaries. We also know that we may encounter some invalid fort.15 files in the wild. Handle
            # it.
            con_names_for_props = copy.deepcopy(con_names)
            for con_name, potential_vals in self.tidal_potential.items():
                con_names_for_props.append(con_name)
                potential_amplitude.append(potential_vals[0])
                frequencies.append(potential_vals[1])
                potential_etrf.append(potential_vals[2])
                nodal_factors.append(potential_vals[3])
                eq_args.append(potential_vals[4])

            # Read the amplitude and phase datasets for each constituent.
            amplitudes = [[0.0 for _ in range(self.num_ocean_nodes)] for _ in range(num_cons)]
            phases = [[0.0 for _ in range(self.num_ocean_nodes)] for _ in range(num_cons)]
            for i in range(num_cons):
                self.curr_line += 1  # Skip the descriptor here, it is not reliable.
                # Read the data for each ocean boundary node.
                for j in range(self.num_ocean_nodes):
                    line = self._split_line()
                    amplitudes[i][j] = float(line[0])
                    phases[i][j] = float(line[1])

            self._create_tidal_data_component(
                ocean_node_ids, con_names, con_names_for_props, potential_amplitude, frequencies, nodal_factors,
                eq_args, potential_etrf, amplitudes, phases
            )
        elif len(ocean_node_ids) > 0:  # Non-periodic tidal forcing specified in the fort.19
            XmLog().instance.warning(
                'This simulation uses non-periodic water surface elevation forcing on ocean nodes.\nThis only has '
                'limited support in SMS at this time.\nPlease contact technical support for more details.'
            )
            self.read19 = True  # Not reading fort.19 files for now

    def _read_flow(self):
        """Read the tidal periodic forcing constituents on land boundaries (ANGINN - NFFR and data).

        The fort.14 must be read before this happens. We need to know the number of land boundary nodes
        and their locations.
        """
        self.sim_data.general.attrs['ANGINN'] = self._get_first_float()
        river_node_ids = self.bc_data.get_river_node_ids()
        self.num_river_nodes = len(river_node_ids)
        if self.num_river_nodes:
            num_periodic = self._get_first_int()  # NFFR
            if num_periodic < 1:  # Non-periodic flow, we need to read the fort.20
                if num_periodic == 0:  # Offset by hotstart time if -1
                    self.bc_data.source_data.info.attrs['hot_start_flow'] = 0
                    self.bc_data.source_data.commit()
                self.read20 = True  # Not reading fort.20 files for now
            else:  # Periodic forcing
                XmLog().instance.info('Creating applied periodic flow component...')
                # Need a list of the periodic flow identifiers to know their order because the identifiers in the
                # second loop do not need to match
                flow_names = ['' for _ in range(num_periodic)]
                frequencies = [0.0 for _ in range(num_periodic)]
                nodal_factors = [0.0 for _ in range(num_periodic)]
                eq_args = [0.0 for _ in range(num_periodic)]

                # Read the forcing frequency, nodal factor, and equilibrium argument for each flow frequency.
                for i in range(num_periodic):
                    flow_names[i] = self._get_first_token()
                    line = self._split_line()
                    frequencies[i] = float(line[0])
                    nodal_factors[i] = float(line[1])
                    eq_args[i] = float(line[2])

                # Read the amplitude and phase datasets for each flow frequency.
                amplitudes = [[0.0 for _ in range(self.num_river_nodes)] for _ in range(num_periodic)]
                phases = [[0.0 for _ in range(self.num_river_nodes)] for _ in range(num_periodic)]
                for i in range(num_periodic):
                    self.curr_line += 1  # Skip the descriptor here, it is not reliable.
                    # Read the data for each river boundary node.
                    for j in range(self.num_river_nodes):
                        line = self._split_line()
                        amplitudes[i][j] = float(line[0])
                        phases[i][j] = float(line[1])

                self._create_flow_data_component(
                    river_node_ids, flow_names, frequencies, nodal_factors, eq_args, amplitudes, phases
                )

    def _read_output_parameters(self, mesh_name):
        """Read the output parameters (NOUTE - NOUTGW).

        Args:
            mesh_name (:obj:`str`): Name of the mesh previously built by the fort.14 reader
        """
        recording_pts = {}  # key=(str(x),str(y)) value=[Elevation, Velocity, Met]
        min_per_ts = self.sim_data.timing.attrs['DTDP'] / 60.0  # convert output intervals to minutes

        # Read the recording station output options.
        for idx, station_output in enumerate(['NOUTE', 'NOUTV', 'NOUTW', 'NOUTGE', 'NOUTGV', 'NOUTGW']):
            wind_output = idx in [2, 5]
            if wind_output:  # 'NOUTW', 'NOUTGW'
                # IM = 10 no longer exists. This is just here for backwards compatibility. Used to have another
                # type of output between velocity and meteorological.
                if self.sim_data.formulation.attrs['IM'] == 10:
                    self.curr_line += 1  # NOUTC
                    if idx < 3:  # Skip over location lines if this is the station output variety.
                        num_pts = self._get_first_int()
                        self.curr_line += num_pts
                if self.fort22_data.nws_type == 0:
                    continue  # Do not read meteorological output lines if no wind enabled.
            line = self._split_line()
            opt = int(line[0])
            abs_opt = abs(opt)
            if opt > 0:  # Append vs. overwrite
                self.sim_data.output['File'].loc[station_output] = 0
                hot_start_files = sid.OUTPUT_FILENAMES[station_output]
                if abs_opt in [3, 5]:  # Append NetCDF file extension if needed.
                    for i in range(len(hot_start_files)):
                        hot_start_files[i] += '.nc'
                # Link up hot start files to selectors.
                self.sim_data.output['Hot Start'].loc[station_output] = os.path.normpath(
                    os.path.join(os.getcwd(), hot_start_files[0])
                )
                if wind_output:
                    self.sim_data.output['Hot Start (Wind Only)'].loc[station_output] = os.path.normpath(
                        os.path.join(os.getcwd(), hot_start_files[1])
                    )

            if abs_opt in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                self.read_netcdf_options = True
            self.sim_data.output['Override __new_line__ Default'].loc[station_output] = 1  # Overwrite default row
            self.sim_data.output['Output'].loc[station_output] = abs_opt
            self.sim_data.output['Start __new_line__ (days)'].loc[station_output] = float(line[1])
            self.sim_data.output['End __new_line__ (days)'].loc[station_output] = float(line[2])
            self.sim_data.output['Increment __new_line__ (min)'].loc[station_output] = float(line[3]) * min_per_ts
            if idx < 3:  # Read recording station locations if not a global output.
                num_stations = self._get_first_int()
                for _ in range(num_stations):
                    line = self._split_line()
                    pt_key = (line[0], line[1])
                    if pt_key not in recording_pts:
                        recording_pts[pt_key] = [0, 0, 0]
                    recording_pts[pt_key][idx] = 1

        # Build the recording station coverage if points were read in.
        if recording_pts:
            self._build_recording_station_coverage(mesh_name, recording_pts)

    def _read_harmonic_constituents(self):
        """Read the harmonic analysis constituent data (NFREQ and data - NHAGV).

        Even if the data was pulled from a database, we will recreate them as user defined because that
        information is not available in the model-native files.
        """
        num_harm_cons = self._get_first_int()
        cons = ['User defined' for _ in range(num_harm_cons)]
        names = ['' for _ in range(num_harm_cons)]
        frequencies = [0.0 for _ in range(num_harm_cons)]
        nodal_factors = [0.0 for _ in range(num_harm_cons)]
        eq_args = [0.0 for _ in range(num_harm_cons)]
        for i in range(num_harm_cons):
            names[i] = self._get_first_token()
            line = self._split_line()
            frequencies[i] = float(line[0])
            nodal_factors[i] = float(line[1])
            eq_args[i] = float(line[2])
        harm_data = {
            'Constituent': xr.DataArray(data=cons),
            'Constituent Name': xr.DataArray(data=names),
            'Frequency (HAREQ)': xr.DataArray(data=frequencies),
            'Nodal Factor (HAFF)': xr.DataArray(data=nodal_factors),
            'Equilibrium (HAFACE)': xr.DataArray(data=eq_args),
        }

        line = self._split_line()
        harm_attrs = self.sim_data.harmonics.attrs
        harm_attrs['THAS'] = float(line[0])
        harm_attrs['THAF'] = float(line[1])
        harm_attrs['NHAINC'] = int(float(line[2]))
        harm_attrs['FMV'] = float(line[3])

        line = self._split_line()
        harm_attrs['NHASE'] = int(line[0])
        harm_attrs['NHASV'] = int(line[1])
        harm_attrs['NHAGE'] = int(line[2])
        harm_attrs['NHAGV'] = int(line[3])

        self.sim_data.harmonics = xr.Dataset(data_vars=harm_data, attrs=harm_attrs)

    def _read_iter_control(self):
        """Read the hot start and iteration control options (NHSTAR - ITMAX)."""
        min_per_ts = self.sim_data.timing.attrs['DTDP'] / 60.0  # Convert output intervals to minutes

        line = self._split_line()
        # We do not have a Windows build of ADCIRC that supports NetCDF4. If we encounter this in a model native
        # file, switch to NetCDF.
        self.sim_data.output.attrs['NHSTAR'] = int(line[0]) if int(line[0]) != 5 else 3
        if self.sim_data.output.attrs['NHSTAR'] in [3, 5]:  # 3=NetCDF, 5=NetCDF4:
            self.read_netcdf_options = True  # 3=NetCDF, 5=NetCDF4
        self.sim_data.output.attrs['NHSINC'] = float(line[1]) * min_per_ts  # convert to minutes
        line = self._split_line()
        self.sim_data.formulation.attrs['ITITER'] = int(line[0])
        self.sim_data.formulation.attrs['ISLDIA'] = int(line[1])
        self.sim_data.formulation.attrs['CONVCR'] = float(line[2])
        self.sim_data.formulation.attrs['ITMAX'] = int(line[3])

    def _read_netcdf_lines(self):
        """Read the optional NetCDF lines from the fort.15 if any of the output file formats are NetCDF."""
        # The NetCDF options are optional, so we need to make sure we don't read the annotation as the value.
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCPROJ'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCINST'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCSOUR'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCHIST'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCREF'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCCOM'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCHOST'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCCONV'] = '' if value.startswith('!') else value
        value = self._get_line().split('!')[0].strip()
        self.sim_data.output.attrs['NCCONT'] = '' if value.startswith('!') else value
        # Parse the date out of the next string.
        line = self._get_line().strip()
        try:
            year = int(line[:4])
            month = int(line[5:7])
            day = int(line[8:10])
            hour = int(line[11:13])
            minute = int(line[14:16])
            second = int(line[17:19])
            qdate = QDate(year, month, day)
            qtime = QTime(hour, minute, second)
            nc_date = QDateTime(qdate, qtime)
        except Exception:
            nc_date = QDateTime.currentDateTime()
        nc_date = qdatetime_to_datetime(nc_date).strftime(ISO_DATETIME_FORMAT)
        # Make sure the simulation reference time is consistent with the NetCDF reference time.
        if self.read_ref_date and nc_date != self.sim_data.timing.attrs['ref_date']:
            XmLog().instance.warning(
                f'Inconsistent reference times: {self.sim_data.timing.attrs["ref_date"]} != {nc_date}. Defaulting to '
                f'the NetCDF reference time.'
            )
            self.sim_data.timing.attrs['ref_date'] = nc_date
        if not self.read_ref_date:
            self.sim_data.timing.attrs['ref_date'] = nc_date
        self.sim_data.output.attrs['NCDATE'] = nc_date

    def _read_namelists(self):
        """Read the optional namelists at the end of the fort.15.

        Supported namelists: timeBathyControl

        """
        while self.curr_line < len(self.lines):  # Keep reading namelists until the end of the file
            namelist = self._get_line()
            if "timeBathyControl" in namelist:
                self._read_time_bathy()

    def _read_time_bathy(self):
        """Read the optional timeBathyControl namelist."""
        line = self._split_line()  # NDDT
        nddt = int(line[2])
        if nddt < 0:
            self.sim_data.advanced.attrs['NDDT'] = -1
        self.sim_data.advanced.attrs['NDDT'] = nddt
        line = self._split_line()
        self.sim_data.advanced.attrs['BTIMINC'] = int(line[2])
        line = self._split_line()
        self.sim_data.advanced.attrs['BCHGTIMINC'] = int(line[2])

    def _find_datasets(self, name):
        """Returns a view of the dataset in an H5 group path or None if it is not a valid XMDF dataset.

        Args:
            name (:obj:`str`): Group name to check for a dataset in
        Returns:
            (:obj:`xms.data_objects.parameters.Dataset`): See description
        """
        current_grp = self.project_dataset_file["Datasets/" + name]
        # If the group has a DatasetCompression attribute assume it is an XMDF dataset
        if "DatasetCompression" in current_grp.attrs:
            dataset = Dataset()
            dataset.filename = self.current_project_file
            dataset.group_path = "Datasets/" + name
            dataset.geom_uuid = self.fort14_data.mesh_uuid
            # Works but puts datasets in simulation folder
            self.built_data['proj_dsets'].append(dataset)
        return None

    def _read_project_datasets(self):
        """Read datasets referenced in the project file."""
        proj_prefix = os.path.basename(os.path.normpath(os.path.join(os.getcwd(), "../..")))
        proj_folder = os.path.normpath(os.path.join(os.getcwd(), "../../.."))
        dset_folder = os.path.join(proj_folder, proj_prefix + "_datasets")
        if os.path.isdir(dset_folder):
            for _, _, files in os.walk(dset_folder):
                for file in files:
                    self.current_project_file = os.path.join(dset_folder, file)
                    try:
                        with h5py.File(self.current_project_file, "r") as self.project_dataset_file:
                            dset_grp = self.project_dataset_file.require_group("Datasets")

                            if dset_grp.__contains__("Guid"):  # Look for the MultiDatasets group
                                geom_uuid = str(dset_grp["Guid"][0].decode('UTF-8'))
                                # Load XMDF project datasets that belong on the ADCIRC mesh
                                if geom_uuid == self.fort14_data.mesh_uuid:
                                    try:
                                        dset_grp.visit(self._find_datasets)
                                    except Exception as e:
                                        raise e
                    except Exception as e:
                        raise e

    def add_built_data(self, send=True):
        """Add data built during import to Query Context and sends data to SMS."""
        # Add the simulation
        sim_name = self.sim_name if self.sim_name else 'Sim'
        sim = Simulation(name=sim_name, model='ADCIRC', sim_uuid=self.sim_uuid)
        self.query.add_simulation(sim, [self.built_data['sim_comp']])

        # Add the domain mesh
        if 'domain_mesh' in self.built_data:
            self.query.add_ugrid(self.built_data['domain_mesh'])
            self.query.link_item(sim.uuid, self.built_data['domain_mesh'].uuid)

        # Add the mapped BC component
        if 'mapped_bc' in self.built_data:
            self.query.add_component(self.built_data['mapped_bc'])
            self.query.link_item(sim.uuid, self.built_data['mapped_bc'].uuid)

        # Add the mapped tidal constituent component (if there is one)
        if 'tidal_comp' in self.built_data:
            self.query.add_component(self.built_data['tidal_comp'])
            self.query.link_item(sim.uuid, self.built_data['tidal_comp'].uuid)

        # Add the mapped flow constituent component (if there is one)
        if 'flow_comp' in self.built_data:
            self.query.add_component(self.built_data['flow_comp'])
            self.query.link_item(sim.uuid, self.built_data['flow_comp'].uuid)

        # Add the recording stations coverage and its component to the Context.
        if 'stations' in self.built_data:
            station_cov = self.built_data['stations'][0]
            station_comp = self.built_data['stations'][1]
            self.query.add_coverage(
                station_cov, model_name='ADCIRC', coverage_type='Recording Stations', components=[station_comp]
            )
            self.query.link_item(sim.uuid, station_cov.uuid)

        # Take a storm track coverage that is defined in an old SMS project's map file
        if self.storm_track_uuid and not self.read22:
            self.query.link_item(sim.uuid, self.storm_track_uuid)

        # Add datasets read from an old project file.
        if 'proj_dsets' in self.built_data:
            for dset in self.built_data['proj_dsets']:
                self.query.add_dataset(dset)

        # Add nodal attributes read from the fort.13
        if 'nodal_atts' in self.built_data:
            for nodal_att in self.built_data['nodal_atts']:
                self.query.add_dataset(nodal_att)

        # -- Wind data --
        # Wind grid
        if 'wind_grid' in self.built_data:
            self.query.add_ugrid(self.built_data['wind_grid'])
            self.query.link_item(sim.uuid, self.built_data['wind_grid'].uuid)

        # Wind datasets (on mesh or wind grid)
        if 'wind_dsets' in self.built_data:
            for wind_dset in self.built_data['wind_dsets']:
                self.query.add_dataset(wind_dset)

        # Storm track coverage
        if 'wind_cov' in self.built_data:
            self.query.add_coverage(self.built_data['wind_cov'])
            self.query.link_item(sim.uuid, self.built_data['wind_cov'].m_cov.uuid)

        if send:
            self.query.send()

    def read(self):
        """Top level function that triggers the read of a native ADCIRC project."""
        if not os.path.isfile(self.filename) or os.path.getsize(self.filename) == 0:  # pragma: no cover
            sys.stderr.write(f"Error reading fort.15: File not found - {self.filename}")
            return

        XmLog().instance.info('Reading fort.15 ADCIRC control file...')
        with open(self.filename, "r") as f:
            self.lines = f.readlines()

        # Create the hidden simulation component
        XmLog().instance.info('Creating simulation component data...')
        self._create_sim_data_component()

        XmLog().instance.info('Parsing run options...')
        self._read_run_options()
        XmLog().instance.info('Parsing friction options...')
        self._read_friction_options()
        XmLog().instance.info('Parsing NWS type...')
        self._read_nws_type()
        XmLog().instance.info('Parsing ramp up options...')
        self._read_ramp_and_time()
        XmLog().instance.info('Parsing projection options...')
        self._read_coords()
        XmLog().instance.info('Parsing tidal potential...')
        self._read_tidal_potential()

        # Need to read the fort.14 before the tidal forcing section
        mesh_name = 'ADCIRC Mesh'
        fort14file = os.path.join(os.path.dirname(self.filename), 'fort.14')
        fort14file = fort14file if 14 not in self.alt_filenames else os.path.join(
            os.path.dirname(self.filename), self.alt_filenames[14]
        )
        if not os.path.isfile(fort14file):
            fort14file = os.path.splitext(self.filename)[0] + '.grd'
        if os.path.isfile(fort14file):
            reader14 = Fort14Reader(fort14file, self.query, self.fort14_data)
            XmLog().instance.info('Reading fort.14...')
            if self.query:  # pragma: no cover
                reader14.read()
                self.fort22_data.projection = reader14.proj
                self.fort22_data.mesh_uuid = self.fort14_data.mesh_uuid  # Pass along generated mesh UUID
                self.built_data.update(reader14.built_data)
            else:  # testing
                reader14.mapped_bc_dir = self.comp_dir
                reader14.constraint_file = os.path.join(self.comp_dir, 'adcirc_domain.xmc')
                reader14._parse_lines()
                reader14._read_coverages()
                reader14._build_mapped_bc_component()
            mesh_name = reader14.mesh_name
            # Get the number of mesh domain nodes for the fort.22 reader
            nws = abs(self.sim_data.wind.attrs['NWS']) % 100
            nwlon = self.fort22_data.nwlon
            nwlat = self.fort22_data.nwlat
            if nws == 1 or nws == 2 or nws == 4 or nws == 5:
                self.fort22_data.numnodes = reader14.numnodes
            elif nws == 3 or nws == 6 or nws == 7:
                self.fort22_data.numnodes = nwlon * nwlat
            # Get the mapped BC data
            self.bc_data = reader14.mapped_data
            # Get the river node ID order
            self.river_node_ids = reader14.river_boundaries
        elif self.query:  # pragma: no cover
            XmLog().instance.error(
                f'Unable to find fort.14 or matching *.grd geometry file in same directory as {self.filename}.'
            )
            return

        XmLog().instance.info('Parsing tidal forcing values...')
        self._read_tidal_forcing()
        XmLog().instance.info('Parsing flow forcing values...')
        self._read_flow()
        XmLog().instance.info('Parsing output options...')
        self._read_output_parameters(mesh_name)
        XmLog().instance.info('Parsing harmonic analysis parameters...')
        self._read_harmonic_constituents()
        XmLog().instance.info('Parsing iteration control options...')
        self._read_iter_control()
        if self.read_netcdf_options:
            XmLog().instance.info('Parsing NetCDF output options...')
            self._read_netcdf_lines()
        XmLog().instance.info('Parsing time varying bathymetry Fortran namelist...')
        self._read_namelists()

        self.fort22reader = None
        if self.read22:
            fort22file = os.path.join(os.path.dirname(self.filename), 'fort.22')
            fort22file = fort22file if 22 not in self.alt_filenames else os.path.join(
                os.path.dirname(self.filename), self.alt_filenames[22]
            )
            if os.path.isfile(fort22file):
                self.fort22reader = Fort22Reader(fort22file, self.query, self.fort22_data, self.sim_data)
                XmLog().instance.info('Reading wind parameters from fort.22...')
                self.fort22reader.read()
                self.built_data.update(self.fort22reader.built_data)

        if self.read23:
            fort23file = os.path.join(os.path.dirname(self.filename), 'fort.23')
            fort23file = fort23file if 23 not in self.alt_filenames else os.path.join(
                os.path.dirname(self.filename), self.alt_filenames[23]
            )
            if os.path.isfile(fort23file):
                fort22reader = Fort22Reader(fort23file, self.query, self.fort22_data, self.sim_data, radstress=True)
                XmLog().instance.info('Reading radstress parameters from fort.23...')
                fort22reader.read()
                wind_dsets = fort22reader.built_data.get('wind_dsets')
                if wind_dsets:
                    dsetlist = self.built_data.setdefault('wind_dsets', [])
                    dsetlist.extend(wind_dsets)

        # Read the fort.13 if nodal attributes have been specified.
        if self.read13:
            filename = os.path.join(os.path.dirname(self.filename), 'fort.13')
            filename = filename if 13 not in self.alt_filenames else os.path.join(
                os.path.dirname(self.filename), self.alt_filenames[13]
            )
            if os.path.isfile(filename):
                if self.query:  # pragma: no cover
                    reader13 = Fort13Reader(filename, self.query, self.sim_data)
                else:  # testing
                    reader13 = Fort13Reader(filename, True, self.sim_data)
                    reader13.out_filenames = self.out_filenames
                    reader13.dset_uuids = self.fort22_data.dset_uuids
                XmLog().instance.info('Reading nodal attributes from fort.13...')
                reader13.read()
                self.built_data.update(reader13.built_data)

        if self.read19:  # Limited support for non-periodic water level forcing, just keeping a file reference for now
            pass

        if self.read20:  # Limited support for non-periodic flow forcing, just keeping a file reference for now
            filename = os.path.join(os.path.dirname(self.filename), 'fort.20')
            filename = filename if 20 not in self.alt_filenames else os.path.join(
                os.path.dirname(self.filename), self.alt_filenames[20]
            )
            if os.path.isfile(filename):
                if self.query:  # pragma: no cover
                    reader20 = Fort20Reader(filename, self.query, self.sim_data)
                else:  # testing
                    reader20 = Fort20Reader(filename, True, self.sim_data)
                    reader20.out_filenames = self.out_filenames
                    reader20.dset_uuids = self.fort22_data.dset_uuids
                XmLog().instance.info('Reading hydrographs from fort.20...')
                dt, flows_by_node = reader20.read(self.num_river_nodes)
                self.bc_data.set_river_flow_data(dt, flows_by_node, self.river_node_ids)
                self.bc_data.commit()

        # Read datasets from old SMS project file
        if self.read_project_dsets:
            self._read_project_datasets()
        self.sim_data.commit()  # Write simulation component data to disk.

    def send(self):
        """Send built data to XMS."""
        XmLog().instance.info('Sending imported data to XMS...')
        self.add_built_data()


def read_adcirc_simulation():
    """Read an ADCIRC simulation when a fort.15 file is imported into SMS.

    Returns:
        (:obj:`Fort15Reader`): The reader. Need to call send to load imported data into SMS.
    """
    query = Query()

    # Get the SMS global time. Set reference date to this if not in ADCIRC input files.
    global_time = query.global_time
    if global_time is not None:
        global_time = gui_util.datetime_to_qdatetime(global_time)
    else:  # pragma: no cover
        global_time = QDateTime.currentDateTime()

    # Get the XMS component temp directory
    comp_dir = os.path.join(query.xms_temp_directory, 'Components')
    os.makedirs(comp_dir, exist_ok=True)

    # Read the fort.15 as a control file. A new simulation will be created.
    reader = Fort15Reader(query.read_file, query, comp_dir, global_time)
    reader.read()
    return reader
