"""Qt table view for the harmonic analysis inputs."""

# 1. Standard Python modules
import datetime

# 2. Third party modules
from harmonica.tidal_constituents import Constituents
from harmonica.tidal_database import NOAA_SPEEDS
from PySide2.QtCore import QSortFilterProxyModel, Qt
from PySide2.QtWidgets import QAbstractItemView, QComboBox

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget

TBL_COL_CON = 0  # Name if not user defined
TBL_COL_NAME = 1  # Disabled if not user defined
TBL_COL_FREQ = 2  # Disabled if not user defined
TBL_COL_NODAL_FACTOR = 3  # Disabled if not user defined
TBL_COL_EQUILIBRIUM = 4  # Disabled if not user defined

USER_DEFINED_OPTION = 'User defined'


class ConstituentButtonDelegate(QxCbxDelegate):
    """Delegate for the constituent button column."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (:obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self._strings = list(NOAA_SPEEDS.keys())
        self._strings.append('User defined')

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates the combobox and populates it.

        Args:
            parent (:obj:`QWidget`): The parent.
            option (:obj:`QStyleOptionViewItem`): The option
            index (:obj:`QModelIndex`): The index

        Returns:
            (:obj:`QWidget`): See description
        """
        self.cb = QComboBox(parent)
        self.cb.addItems(self._strings)
        self.cb.currentIndexChanged.connect(self.on_index_changed)
        return self.cb


class HarmonicsTableDisableEditModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        col = index.column()
        if col != TBL_COL_CON:
            row = index.row()
            con = index.model().index(row, TBL_COL_CON).data()
            is_user_defined = con == USER_DEFINED_OPTION
            if is_user_defined:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        return ret_flags


class HarmonicsTableWidget(AdcircTableWidget):
    """The harmonic analysis table widget."""
    def __init__(self, data_frame, ref_date, run_days, parent=None):
        """Construct the widget.

        Args:
            data_frame (DataFrame): The model data.
            ref_date (datetime.datetime): The ADCIRC interpolation reference date
            run_days (float): The ADCIRC simulation run duration
            parent (object): The parent object.
        """
        if len(data_frame.index) > 0 and data_frame.index[0] == 0:
            data_frame.index += 1  # Make vertical header 1-based
        super().__init__(
            parent, data_frame, TBL_COL_NAME, {
                'Constituent': 'M2',
                'Constituent Name': 'M2',
                'Frequency (HAREQ)': 0.0,
                'Nodal Factor (HAFF)': 0.0,
                'Equilibrium (HAFACE)': 0.0,
            }
        )
        self.ref_date = ref_date
        self.middle_dt = ref_date + datetime.timedelta(run_days)
        self.tidal_model = Constituents('leprovost')  # Specific model doesn't matter. LeProvost always available.
        self.enable_model = HarmonicsTableDisableEditModel(self)
        self.con_delegate = ConstituentButtonDelegate(self)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        delegates = {TBL_COL_CON: self.con_delegate}
        super()._setup_ui(delegates, False, False, self.enable_model)
        self.model.dataChanged.connect(self.on_constituent_changed)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def on_constituent_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (:obj:`QModelIndex`): Top left index.
            bottom_right_index (:obj:`QModelIndex`): Bottom right index. Same as top_left_index. QxPandasTableModel only
                supports single cell editing.

        """
        if not top_left_index.isValid():
            return

        col = top_left_index.column()
        if col == TBL_COL_CON:
            con_name = self.model.data(top_left_index)
            if con_name != USER_DEFINED_OPTION:
                # Fill in the non-editable columns with constituent properties.
                df = self.tidal_model.get_nodal_factor([con_name], self.ref_date, self.middle_dt)
                con_df = df.loc[con_name]
                freq = con_df['frequency']
                nf = con_df['nodal_factor']
                eq_arg = con_df['equilibrium_argument']
                row = top_left_index.row()
                new_idx = self.model.index(row, TBL_COL_NAME)
                self.model.setData(new_idx, con_name)
                new_idx = self.model.index(row, TBL_COL_FREQ)
                self.model.setData(new_idx, f'{freq:.6f}')
                new_idx = self.model.index(row, TBL_COL_NODAL_FACTOR)
                self.model.setData(new_idx, f'{nf:.6f}')
                new_idx = self.model.index(row, TBL_COL_EQUILIBRIUM)
                self.model.setData(new_idx, f'{eq_arg:.6f}')

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        # Update the constituent properties for the new row.
        row_idx = self.model.rowCount() - 1
        new_index = self.model.index(row_idx, TBL_COL_CON)
        self.on_constituent_changed(new_index, new_index)

    def on_ref_date_changed(self, new_ref_date):
        """Update the constituent properties when the ADCIRC interpolation reference date changes.

        Args:
            new_ref_date (:obj:`QDateTime`): The new interpolation reference date.
        """
        self.ref_date = new_ref_date.toPython()
        for row in range(self.model.rowCount()):
            index = self.model.index(row, TBL_COL_CON)
            self.on_constituent_changed(index, index)
