"""Manages mapping of BC coverages."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util as tr

# 4. Local modules
from xms.adcirc.components.bc_component import BcComponent
from xms.adcirc.components.mapped_bc_component import MappedBcComponent
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.data.mapped_bc_data import MappedBcData


def is_main_coverage(py_comp):
    """Checks if the given coverage component is a main coverage.

    Args:
        py_comp: The component to check. Can be the data or the component class

    Returns:
        bool: True if the coverage component is a main coverage.

    Notes:
        - This function checks for non-levee arcs in the coverage data.
        - If non-levee arcs are found, it considers the coverage component as a main coverage.
        - If the coverage only has levees, it is not considered a main coverage.
    """
    if isinstance(py_comp, BcComponent):
        data = py_comp.data
    elif isinstance(py_comp, bcd.BcData):
        data = py_comp
    elif isinstance(py_comp, MappedBcComponent):  # MappedBcComponent
        data = py_comp.data.source_data
    elif isinstance(py_comp, MappedBcData):
        data = py_comp.source_data
    else:
        raise ValueError('Invalid arguments passed to `is_main_coverage()`')

    non_levee_arcs = data.arcs.where(
        (data.arcs.type != bcd.LEVEE_INDEX) & (data.arcs.type != bcd.UNASSIGNED_INDEX), drop=True
    )
    if non_levee_arcs.sizes['comp_id'] > 0:
        return True
    return False


class BcMappingManager:
    """Class to manage mapping of BC coverages.

    - If I have an ADCIRC BC coverage that contains only levees and I map it to an ADCIRC sim
          if any existing mapped bc component references this coverage it will be deleted.
          A new mapped bc component will be created, and it will have a reference to the source coverage.
    - If I have an ADCIRC BC coverage that contains any other bcs besides levees, and I map it to an ADCIRC sim then
          the main mapped bc component will be deleted and replaced by the data in this coverage
    """
    def __init__(self, sim_uuid, query):
        """Constructor.

        Args:
            sim_uuid (str): UUID of the simulation we are mapping to.
            query (Query): Object for communicating with XMS
        """
        self.delete_uuids = []
        self.is_main = False
        self._initialize(sim_uuid, query)

    def _initialize(self, sim_uuid, query):
        """Get all the junk we need for SMS.

        Args:
            sim_uuid (str): UUID of the simulation we are mapping to.
            query (Query): Object for communicating with XMS
        """
        pe_tree = query.project_tree
        sim_item = tr.find_tree_node_by_uuid(pe_tree, sim_uuid)
        # There should only ever be one BC coverage under the simulation. We actually don't link BC coverages. We just
        # kick them out after mapping.
        cov_item = tr.descendants_of_type(
            sim_item,
            only_first=True,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Boundary Conditions'
        )
        self._check_mapping_coverage(query, cov_item)
        self._check_mapped_coverages(query, sim_item, cov_item.uuid)

    def _check_mapping_coverage(self, query, cov_item):
        """Checks if the coverage being mapped contains anything other than levees.

        Args:
            query (Query): The query
            cov_item (TreeNode): The coverage item to be checked for mapping.
        """
        do_comp = query.item_with_uuid(cov_item.uuid, model_name='ADCIRC', unique_name='Bc_Component')
        py_comp = BcComponent(do_comp.main_file)
        self.is_main = is_main_coverage(py_comp)

    def _check_mapped_coverages(self, query, sim_item, cov_uuid):
        """Checks and manages mapped coverages for a given simulation item.

        Args:
            query (Query): The query object for database operations.
            sim_item (TreeNode): The simulation item to be checked for mapped coverages.
            cov_uuid (str): The UUID of the coverage to be checked.
        """
        # First get all mapped BC coverages under this sim.
        mapped_items = tr.descendants_of_type(
            sim_item, model_name='ADCIRC', unique_name='Mapped_Bc_Component', allow_pointers=True
        )
        if not mapped_items:
            return  # If there are no previously mapped BCs, we are good to go.

        for mapped_item in mapped_items:
            comp_uuid = mapped_item.uuid
            do_comp = query.item_with_uuid(comp_uuid)
            py_comp = MappedBcComponent(do_comp.main_file)
            # If we are mapping the same coverage, always kick out the old mapped item.
            old_cov_uuid = py_comp.data.source_data.info.attrs['cov_uuid']
            if cov_uuid == old_cov_uuid:
                self.delete_uuids.append(comp_uuid)
            else:  # Check for levee types
                if self.is_main and is_main_coverage(py_comp):
                    # If this coverage has anything other than levee arcs, it becomes the "main" mapped BC coverage,
                    # of which there can only be one.
                    self.delete_uuids.append(py_comp.uuid)
