"""Apply tidal constituents to an ADCIRC simulation."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os
import uuid

# 2. Third party modules
import xarray as xr

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.data_objects.parameters import Component
from xms.guipy.time_format import ISO_DATETIME_FORMAT
from xms.tides.data import tidal_data as td
from xms.tides.data.tidal_extractor import TidalExtractor

# 4. Local modules
from xms.adcirc.components.mapped_tidal_component import MappedTidalComponent
from xms.adcirc.data import mapped_tidal_data as mtd
from xms.adcirc.dmi.sim_component_queries import unlink_tidal_comp_on_error
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.mapping_util import linear_interp_with_idw_extrap


class TidalMapper:
    """Class for mapping tidal constituents to the ocean nodes of a mapped BC component."""
    def __init__(
        self, new_main_file, tidal_data, mapped_bc_data, tidal_name, user_amps, user_phases, user_geoms, co_grid
    ):
        """Construct the mapper.

        Args:
            new_main_file (:obj:`str`): File location where the new mapped component's main file should be created.
                Assumed to be in a unique UUID folder in the temp directory.
            tidal_data (:obj:`TidalData`): Data of the source tidal constituent component
            mapped_bc_data (:obj:`MappedBcData`): Data of the mapped BC component
            tidal_name (:obj:`str`): Name of the source tidal component tree item in XMS
            user_amps (:obj:`list`): The user defined amplitude xms.data_objects.parameters.Dataset objects
            user_phases (:obj:`list`): The user defined phase xms.data_objects.parameters.Dataset objects
            user_geoms (:obj:`dict`): The user defined phase and amplitude geometry interpolators. Keyed by geom UUID.
            co_grid (:obj:`CoGrid`): The domain mesh
        """
        self.mapped_tidal_file = new_main_file
        self.mapped_bc_data = mapped_bc_data
        self.tidal_data = tidal_data
        self.tidal_name = tidal_name
        self.user_amps = user_amps
        self.user_phases = user_phases
        self.user_geoms = user_geoms
        self.xmugrid = co_grid.ugrid
        self.mapped_comp = None
        self.idw_interps = {}  # Fall back to IDW if extrapolating

    def map_data(self):
        """Create a mapped tidal component from a source tidal constituent component.

        Returns:
            (:obj:`Component`): The mapped tidal XMS component object, None if errors encountered.
        """
        self.mapped_comp = MappedTidalComponent(self.mapped_tidal_file)

        # Create the mapped BC Python component
        do_comp = None
        if self._create_map_component():
            XmLog().instance.info('Writing applied data to new component files.')
            self.mapped_comp.data.commit()

            # Create the data_objects component
            do_comp = Component(
                name=f'{self.tidal_name} (applied)',
                comp_uuid=os.path.basename(os.path.dirname(self.mapped_tidal_file)),
                main_file=self.mapped_tidal_file,
                model_name='ADCIRC',
                unique_name='Mapped_Tidal_Component',
                locked=False
            )
        return do_comp

    def _create_map_component(self):
        """Create the mapped BC component Python object.

        Returns:
            (:obj:`bool`): True if any errors encountered
        """
        XmLog(
        ).instance.info('Querying harmonica for constituent frequencies, nodal factors, and equilibrium arguments.')
        # Get the properties of the enabled constituents (frequency, nodal factor, and equilibrium argument)
        extractor = TidalExtractor(self.tidal_data)
        self.mapped_comp.data.cons = extractor.get_constituent_properties()
        if self.mapped_comp.data.cons is None:
            XmLog().instance.error(
                'Could not apply tidal constituents to ADCIRC simulation. No user defined '
                'constituents specified.'
            )
        # Get the ocean node ids from the mapped BC component
        XmLog().instance.info('Reading ocean boundary node locations on the domain mesh.')
        ocean_nodes, _ = self.mapped_bc_data.get_ocean_node_ids()
        ocean_nodes_0based = [node_id - 1 for node_id in ocean_nodes]
        if self.tidal_data.info.attrs['source'] == td.USER_DEFINED_INDEX:
            self._map_from_user_defined(ocean_nodes, ocean_nodes_0based)
        else:
            self._map_from_database(extractor, ocean_nodes_0based)
        return True

    def _map_from_user_defined(self, ocean_nodes, ocean_nodes_0based):
        """Map tides using user-defined amplitude and phase datasets.

        Args:
            ocean_nodes (:obj:`list[int]`): 1-base ocean node IDs
            ocean_nodes_0based (:obj:`list[int]`): 0-base ocean node IDs
        """
        XmLog(
        ).instance.info('Interpolating user defined constituent amplitude and phase datasets to ocean boundary nodes.')
        all_amps = []
        all_phases = []
        ocean_locs = [(loc[0], loc[1], loc[2]) for loc in self.xmugrid.get_points_locations(ocean_nodes_0based)]
        # Interpolate the selected datasets to the ocean nodes
        for user_amp, user_phase in zip(self.user_amps, self.user_phases):
            all_amps.append(
                linear_interp_with_idw_extrap(
                    self.user_geoms[user_amp.geom_uuid], ocean_locs, user_amp, self.idw_interps
                )
            )

            all_phases.append(
                linear_interp_with_idw_extrap(
                    self.user_geoms[user_phase.geom_uuid], ocean_locs, user_phase, self.idw_interps
                )
            )
        cons = self.mapped_comp.data.cons['con'].data.tolist()
        tidal_coords = {
            'con': cons,
            'node_id': ocean_nodes,
        }
        tidal_data = {
            'amplitude': (('con', 'node_id'), all_amps),
            'phase': (('con', 'node_id'), all_phases),
        }
        self.mapped_comp.data.values = xr.Dataset(data_vars=tidal_data, coords=tidal_coords).fillna(0.0)

    def _map_from_database(self, extractor, ocean_nodes):
        """Map tides from an external tidal database.

        Args:
            extractor (:obj:`TidalExtractor`): The tidal extractor
            ocean_nodes (:obj:`list[int]`): 0-base ocean node IDs
        """
        XmLog().instance.info('Querying harmonica for database extraction of constituent amplitude and phase values.')
        # Convert (x, y) node locations to (lat, lon) coordinates for harmonica.
        ocean_locs = [(loc[1], loc[0]) for loc in self.xmugrid.get_points_locations(ocean_nodes)]
        # Extract amplitude and phase for selected constituents at ocean boundaries.
        con_dsets = extractor.get_amplitude_and_phase(ocean_locs, ocean_nodes)
        if con_dsets is not None:  # May not be ocean boundaries in coverage.
            self.mapped_comp.data.values = con_dsets.drop_vars('speed').fillna(0.0)
            self.mapped_comp.data.values = self.mapped_comp.data.values.assign_coords(
                {'node_id': self.mapped_comp.data.values.node_id + 1}
            )
        # Check for values outside the domain
        zero_vals = self.mapped_comp.data.values.where(self.mapped_comp.data.values.amplitude == 0.0, drop=True)
        if zero_vals.sizes['node_id'] > 0:
            XmLog().instance.warning('Detected zero amplitude ocean boundaries.')


def _has_constituents(tidal_data):
    """Check if any tidal constituents have been defined on the simulation.

    Args:
        tidal_data (:obj:`TidalData`): The tidal component data.

    Returns:
        (:obj:`bool`): See description.
    """
    src = tidal_data.info.attrs['source']
    no_db_cons = src != td.USER_DEFINED_INDEX and (tidal_data.cons.name.size < 1 or not any(tidal_data.cons.enabled))
    no_user_cons = src == td.USER_DEFINED_INDEX and tidal_data.user.sizes['index'] < 1
    if no_db_cons or no_user_cons:
        XmLog().instance.error(
            'Cannot apply tidal constituents to ADCIRC simulation because no source constituents are defined.'
        )
        return False
    return True


def _check_sim_reftimes(xms_data, tidal_data):
    """Log a warning if the ADCIRC interpolation reference date does not match the tidal simulation's.

    Args:
        xms_data (dict): The XMS data dict
        tidal_data (TidalData): The tidal simulation's data
    """
    adcirc_start = datetime.datetime.strptime(xms_data['sim_reftime'], ISO_DATETIME_FORMAT)
    tidal_start = datetime.datetime.strptime(tidal_data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
    if adcirc_start != tidal_start:
        XmLog().instance.warning(
            f'Tidal simulation reference time ({tidal_start}) does not match the ADCIRC interpolation reference '
            f'time ({adcirc_start}). Constituent properties will be extracted relative to the tidal simulation.'
        )


def map_tides(query, xms_data, mapped_bc_component):
    """Map a tidal constituent component to a mapped BC component.

    Args:
        query (:obj:`Query`): Object for communicating with XMS
        xms_data (:obj:`dict`): Dictionary containing the data retrieved from XMS for tidal mapping
        mapped_bc_component (:obj:`MappedBcComponent`): The applied boundary conditions for the simulation
    """
    # Create a folder for the new mapped component. This assumes we are in the component temp directory.
    XmLog().instance.info('Creating new applied component files.')
    comp_dir = os.path.join(os.path.dirname(os.path.dirname(mapped_bc_component.main_file)), str(uuid.uuid4()))
    os.makedirs(comp_dir, exist_ok=True)
    new_main_file = os.path.join(comp_dir, mtd.MAPPED_TIDAL_MAIN_FILE)

    # Copy the source tidal data to the mapped data folder.
    source_tidal_file = os.path.join(os.path.dirname(new_main_file), os.path.basename(xms_data['tidal_main_file']))
    xfs.copyfile(xms_data['tidal_main_file'], source_tidal_file)
    tidal_data = td.TidalData(source_tidal_file)
    tidal_data.info.attrs['run_duration'] = xms_data['run_duration']
    _check_sim_reftimes(xms_data, tidal_data)

    # Make sure we have a geographic display projection if using a global database.
    from xms.adcirc.dmi.mapped_bc_component_queries import MappedBcComponentQueries
    query_helper = MappedBcComponentQueries(mapped_bc_component, query)
    if tidal_data.info.attrs['source'] != td.USER_DEFINED_INDEX and xms_data['display_projection'] != 'GEOGRAPHIC':
        XmLog().instance.error(
            'Cannot apply tidal constituents to ADCIRC simulation because of an incompatible display projection. '
            'The tidal constituent data source is set to a global database model but the display projection is '
            'not geographic. Only user-defined tidal constituents can be applied to the ADCIRC simulation when in '
            'a non-geographic projection.'
        )
        unlink_tidal_comp_on_error(xms_data['tidal_uuid'], query)
        return

    # Check for dumb users trying to map undefined constituents.
    if not _has_constituents(tidal_data):
        unlink_tidal_comp_on_error(xms_data['tidal_uuid'], query)
        return

    XmLog().instance.info('Retrieving user defined constituent amplitude and phase datasets from SMS.')
    if not query_helper.query_for_tidal_mapping_data(tidal_data, xms_data):
        unlink_tidal_comp_on_error(xms_data['tidal_uuid'], query)
        return

    # Perform the mapping
    XmLog().instance.info('Mapping source constituent values to ocean boundary node locations.')
    mapper = TidalMapper(
        new_main_file, tidal_data, mapped_bc_component.data, xms_data['tidal_name'], xms_data['amp_dsets'],
        xms_data['phase_dsets'], xms_data['geoms'], xms_data['mesh']
    )
    do_comp = mapper.map_data()
    if do_comp is None:
        unlink_tidal_comp_on_error(xms_data['tidal_uuid'], query)
        return

    # Add the mapped component to the Context and send back to XMS.
    XmLog().instance.info('Preparing data for loading into SMS.')
    query_helper.add_xms_data_for_tidal_mapping(xms_data, do_comp)
