"""Classes to lazy load property values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Any

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

_NOT_SET = object()


class LazyField:
    """
    A class that implements a descriptor for lazy loading of fields.

    LazyField allows properties to be computed only when they are accessed,
    rather than being computed during object initialization. This can be
    useful for expensive computations or when the value of a property
    depends on other properties of the object.

    LazyField expects a property function to be provided to give the property
    value. This property function will be called to compute the value of the
    property when it is accessed for the first time. By default, the member
    function is named f"_load_{property_name}".

    Note that LazyField only supports objects with a `__dict__` attribute, as
    it uses the `__dict__` to cache computed property values. Objects that do
    not have a `__dict__` attribute (for example: objects that define
    `__slots__`) cannot be used with LazyField.

    Example usage:

        class MyClass:
            my_property: type_of_field = LazyField()
            def _load_my_property(self):
                # Expensive computation
                return my_computation()

        my_instance = MyClass()
        value = my_instance.my_property # Lazy computation is triggered here
    """
    def __init__(self, load_func_name: str | None = None):
        """
        Initializes a new instance of the __init__ method.

        Args:
            load_func_name: The name of the function to load the property.
                This defaults to "_load_{property_name}".
        """
        self._load_func_name = load_func_name
        self._property_name = None

    def __set_name__(self, _owner_class: Any, property_name: str):
        """
        Set the name of the property.

        Args:
            _owner_class: The class that owns the property.
            property_name: The name of the property being assigned.

        Raises:
            TypeError: If the property is already assigned to a different name.
        """
        if self._property_name is None:
            self._property_name = property_name
        elif property_name != self._property_name:
            raise TypeError(
                "Cannot assign the same LazyField to two different names "
                f"({self._property_name!r} and {property_name!r})."
            )
        if self._load_func_name is None:
            self._load_func_name = f'_load_{property_name}'

    def __get__(self, instance: Any, owner=None):
        """
        Get the value of the property.

        Args:
            instance: The instance of the class on which the property is accessed.
            owner: The class that owns the property.

        Returns:
            The value of the property associated with the instance.

        Raises:
            TypeError: If the `__dict__` attribute on the instance does not support item assignment for caching the
                property value.
        """
        if instance is None:
            return self
        property_dict = self._get_property_dict(instance)
        value = property_dict.get(self._property_name, _NOT_SET)
        if value is _NOT_SET:
            value = property_dict.get(self._property_name, _NOT_SET)
            if value is _NOT_SET:
                value = getattr(instance, self._load_func_name)()
                try:
                    property_dict[self._property_name] = value
                except TypeError:
                    msg = (
                        f"The '__dict__' attribute on {type(instance).__name__!r} instance "
                        f"does not support item assignment for caching {self._property_name!r} property."
                    )
                    raise TypeError(msg) from None
        return value

    def __set__(self, instance: Any, value: Any):
        """
        Set the value of the property.

        Args:
            instance: The instance of the class on which the property is being set.
            value: The value to be set for the property.
        """
        property_dict = self._get_property_dict(instance)
        property_dict[self._property_name] = value

    def __delete__(self, instance: Any):
        """
        Delete the property.

        Args:
            instance: The instance on which the property is being deleted.
        """
        property_dict = self._get_property_dict(instance)
        del property_dict[self._property_name]

    def _get_property_dict(self, instance):
        """
        Get the property dictionary of the instance.

        Args:
            instance: The instance for which to retrieve the property dictionary.

        Returns:
            property_dict: The dictionary containing the property values of the instance.

        Raises:
            TypeError: If the instance does not have a '__dict__' attribute,
                indicating that it cannot store properties.
        """
        if self._property_name is None:
            raise TypeError("Cannot use LazyField instance without calling __set_name__ on it.")
        try:
            property_dict = instance.__dict__
        except AttributeError:
            # instance must have a __dict__ attribute to work
            # not all objects have __dict__ (for example, when the class defines slots)
            msg = (
                f"No '__dict__' attribute on {type(instance).__name__!r} "
                f"instance to cache {self._property_name!r} property."
            )
            raise TypeError(msg) from None
        return property_dict
