"""Converts vessel data to an adhparam object."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from collections import defaultdict
import logging
from typing import Optional

# 2. Third party modules
from adhparam.vessels import Propeller, Vessel, VesselList
import pandas as pd

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc
from xms.gmi.data.generic_model import Group
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.adh.data.model import get_model
from xms.adh.data.xms_data import XmsData


class VesselDataToAdhParam:
    """Used to convert vessel coverages to an ADH param object."""
    def __init__(self, xms_data: 'XmsData', logger: Optional[logging.Logger] = None):
        """Initializes the class."""
        self.xms_data = xms_data
        self._logger = logger
        if self._logger is None:
            self._logger = logging.getLogger('xms.adh')

    @staticmethod
    def _create_vessel(
            vessel_parameters: Group, propeller: Propeller, sailing_segments: pd.DataFrame):
        """Create the adhparam Vessel object.

        Args:
            vessel_parameters: The vessel parameters on the vessel coverage.
            propeller: The adhparam Propeller object.
            sailing_segments: A dataframe containing the ordered arcs in the vessel coverage.

        Returns:
            An adhparam Vessel object.
        """
        first_row = sailing_segments.loc[0]
        vessel = Vessel(
            # If the user draws the arc segments backwards the initial coordinates will be wrong.
            initial_x_coordinate=first_row.START_X_COORDINATE,
            initial_y_coordinate=first_row.START_Y_COORDINATE,
            initial_speed=vessel_parameters.parameter('initial_speed').value,
            vessel_draft=vessel_parameters.parameter('vessel_draft').value,
            vessel_length=vessel_parameters.parameter('vessel_length').value,
            vessel_width=vessel_parameters.parameter('vessel_width').value,
            bow_length_ratio=vessel_parameters.parameter('bow_length_ratio').value,
            stern_length_ratio=vessel_parameters.parameter('stern_length_ratio').value,
            bow_draft_ratio=vessel_parameters.parameter('bow_draft_ratio').value,
            stern_draft_ratio=vessel_parameters.parameter('stern_draft_ratio').value,
            propeller=propeller,
            sailing_segments=sailing_segments,
        )
        return vessel

    @staticmethod
    def _create_propeller(propeller_parameters):
        """
        Create the adhparam Propeller object.

        Args:
            propeller_parameters: The propeller parameters on the Vessel coverage.

        Returns:
            An adhparam Propeller object.
        """
        propeller = None
        if propeller_parameters.is_active:
            propeller = Propeller(
                propeller_type=propeller_parameters.parameter('propeller_type').value,
                propeller_diameter=propeller_parameters.parameter('propeller_diameter').value,
                propeller_center_distance=propeller_parameters.parameter('center_distance').value,
                tow_boat_length=propeller_parameters.parameter('tow_boat_length').value,
                distance_to_stern=propeller_parameters.parameter('distance_to_stern').value,
            )
        return propeller

    @staticmethod
    def _create_sailing_segments(coverage_arcs: list[Arc], first_segment: Arc):
        """Create the sailing segments dataframe from an unordered, bidirectional chain of arcs.

        Args:
            coverage_arcs: Unordered arcs in the coverage.
            first_segment: Arc containing the first segment of the chain.
        """
        # build node -> arcs map
        node_to_arcs = defaultdict(list)
        for arc, _ in coverage_arcs:
            for node in (arc.start_node, arc.end_node):
                coord = (node.x, node.y)
                node_to_arcs[coord].append(arc)

        # start with first_segment
        visited = {first_segment}
        current_arc = first_segment

        # decide which end of first_segment is the true start:
        s_coord = (current_arc.start_node.x, current_arc.start_node.y)
        e_coord = (current_arc.end_node.x, current_arc.end_node.y)

        if len(node_to_arcs[s_coord]) == 1:
            # start_node is the degree-1 endpoint
            current_node, next_node = current_arc.start_node, current_arc.end_node
        elif len(node_to_arcs[e_coord]) == 1:
            # end_node is the degree-1 endpoint
            current_node, next_node = current_arc.end_node, current_arc.start_node
        else:
            # fallback to using start_node
            current_node, next_node = current_arc.start_node, current_arc.end_node

        # walk the chain
        arc_to_params = {arc: params for arc, params in coverage_arcs}
        ordered = []
        seg_num = 1

        while current_arc:
            params = arc_to_params[current_arc]
            group = (
                params.group('first_segment')
                if params.group('first_segment').is_active
                else params.group('segment')
            )

            entry = {
                'SEGMENT_NUMBER': seg_num,
                'SEGMENT_TYPE': 1 if group.parameter('segment_type').value == 'Arc' else 0,
                'START_X_COORDINATE': current_node.x,
                'START_Y_COORDINATE': current_node.y,
                'END_X_COORDINATE': next_node.x,
                'END_Y_COORDINATE': next_node.y,
                'MOTION_VALUE': group.parameter('time_of_arrival').value
            }
            if entry['SEGMENT_TYPE'] == 1:
                entry.update({
                    'ARC_CENTER_X': group.parameter('x-center').value,
                    'ARC_CENTER_Y': group.parameter('y-center').value,
                    'TURN_DIRECTION': group.parameter('turn_direction').value
                })

            ordered.append(entry)
            seg_num += 1

            # pick the next arc off next_node
            coord = (next_node.x, next_node.y)
            next_arc = None
            for arc in node_to_arcs[coord]:
                if arc not in visited:
                    next_arc = arc
                    visited.add(arc)
                    break
            if not next_arc:
                break

            # orient the next arc
            s2 = (next_arc.start_node.x, next_arc.start_node.y)
            if s2 == coord:
                current_node, next_node = next_arc.start_node, next_arc.end_node
            else:
                current_node, next_node = next_arc.end_node, next_arc.start_node

            current_arc = next_arc

        return pd.DataFrame(ordered)

    def get_data(self):
        """Populates a Param AdH model object with data retrieved from SMS."""
        # Create an empty VesselList object
        vessel_list = VesselList(
            use_velocity=False,  # Currently we only support position and time of arrival
            vessels=[]
        )
        # Get selected vessel coverages from the model control
        if 'uuids' not in self.xms_data.adh_data.model_control.vessel_uuids:
            return vessel_list
        if self.xms_data.vessel_components is None:
            return vessel_list
        for coverage, component in zip(self.xms_data.vessel_coverages, self.xms_data.vessel_components):
            data = component.data
            coverage_arcs = []
            first_segment = None
            first_segment_count = 0
            for arc in coverage.arcs:
                values = data.feature_values(TargetType.arc, feature_id=arc.id)

                # Create tuple of arcs and arc parameters
                arc_parameters = get_model().arc_parameters
                arc_parameters.restore_values(values)
                coverage_arcs.append((arc, arc_parameters))

                # find the first segment
                if arc_parameters.group('first_segment').is_active:
                    first_segment = arc
                    first_segment_count += 1
            # If there isn't exactly 1 first segment, raise an error
            if first_segment_count != 1:
                self._logger.error(f'{coverage.name} coverage must contain exactly one first segment.')
                # Skip the coverage
                continue

            # Get vessel coverage parameters
            coverage_parameters = get_model().model_parameters
            coverage_parameters.restore_values(data.vessel_values)
            propeller_parameters = coverage_parameters.group('propeller')
            vessel_parameters = coverage_parameters.group('vessel')

            sailing_segments = self._create_sailing_segments(coverage_arcs, first_segment)
            propeller = self._create_propeller(propeller_parameters)
            vessel = self._create_vessel(vessel_parameters, propeller, sailing_segments)
            vessel_list.vessels.append(vessel)
        return vessel_list
