"""This dialog allows users to assign boundary conditions to an arc."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from adhparam.time_series import TimeSeries
import pandas as pd

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.data.card_info import CardInfo
from xms.adh.gui.bc_arc_attributes_dialog_ui import Ui_ArcAttributeDialog
import xms.adh.gui.resources.adhqrc  # noqa F401
from xms.adh.gui.time_series_editor import TimeSeriesEditor


class BcArcDialog(XmsDlg):
    """A dialog for assigning materials to polygons."""
    def __init__(self, win_cont, title):
        """Allows the user to edit arc boundary conditions.

        Args:
            win_cont (QWidget): Parent window
            title (str): Window title
        """
        super().__init__(win_cont, 'xms.adh.gui.bc_arc_attributes_dialog')
        self.ui = Ui_ArcAttributeDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        self.setWindowTitle(title)
        self.arc_types = CardInfo.arc_types
        self.card_to_index = CardInfo.arc_card_to_index
        self.populate_flow_data = None
        # tuple has card name, series button
        self.CARD = 0
        self.SERIES_BTN = 1
        self.WIDGETS = 2
        self.XY_COLUMNS = 3
        self.index_to_card = [
            ('', None, self.ui.off_page, None), ('OF', None, self.ui.off_page, None),
            ('DIS', (self.ui.dis_discharge, ), self.ui.dis_page, [{
                'X': 'Time (sec)',
                'Y': 'Q'
            }]), ('OVL', (self.ui.ovl_flow, ), self.ui.ovl_page, [{
                'X': 'Time (sec)',
                'Y': 'Vel'
            }]), ('OTW', (self.ui.otw_elevation, ), self.ui.otw_page, [{
                'X': 'Time (sec)',
                'Y': 'WSE'
            }]), ('SPL', (self.ui.spl_flow, ), self.ui.spl_page, [{
                'X': 'Time (sec)',
                'Y': '% Flow'
            }]), ('TID', None, self.ui.off_page, None), ('SDR', None, self.ui.sdr_page, None),
            ('OUT', (self.ui.out_flow, ), self.ui.out_page, [{
                'X': 'Time (sec)',
                'Y': 'Q'
            }]),
            (
                'OVH', (self.ui.ovh_x, self.ui.ovh_y, self.ui.ovh_depth), self.ui.ovh_page, [
                    {
                        'X': 'Time (sec)',
                        'Y': 'Vx'
                    }, {
                        'X': 'Time (sec)',
                        'Y': 'Vy'
                    }, {
                        'X': 'Time (sec)',
                        'Y': 'Depth'
                    }
                ]
            ), ('LDE', (self.ui.lde_elevation, ), self.ui.lde_page, [{
                'X': 'Time (sec)',
                'Y': 'Elevation'
            }]), ('LDH', (self.ui.ldh_depth, ), self.ui.ldh_page, [{
                'X': 'Time (sec)',
                'Y': 'Depth'
            }]), ('LID', (self.ui.lid_draft, ), self.ui.lid_page, [{
                'X': 'Time (sec)',
                'Y': 'Draft'
            }])
        ]
        self.series_data = {}
        self._setup()

    def _setup(self):
        """Sets up various widget properties and actions."""
        self.ui.snapping_type.addItems(CardInfo.snap_options)
        self.ui.arc_type.addItems(self.arc_types)
        self.ui.arc_type.currentIndexChanged[int].connect(self.arc_type_changed)

        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.ui.sdr_coef_a.setValidator(QxDoubleValidator())
        self.ui.sdr_coef_b.setValidator(QxDoubleValidator())
        self.ui.sdr_coef_c.setValidator(QxDoubleValidator())
        self.ui.sdr_coef_d.setValidator(QxDoubleValidator())
        self.ui.sdr_coef_e.setValidator(QxDoubleValidator())
        self.ui.arc_type.setCurrentIndex(self.card_to_index['SDR'])
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())
        self.ui.arc_type.setCurrentIndex(self.card_to_index[''])

    def get_open_series_dialog_lambda(self, idx, series_idx):
        """Gets a function for opening a specific series in a dialog.

        Args:
            idx (int): The card index.
            series_idx (int): The series index for the card.

        Returns:
            A lambda for the open series dialog.
        """
        return lambda: self.open_series_dialog(idx, series_idx)

    def get_open_series_populate_dialog_lambda(self, idx, series_idx):
        """Gets a function for opening a specific series in a dialog.

        Args:
            idx (int): The card index.
            series_idx (int): The series index for the card.

        Returns:
            A lambda for the open series with populate buttons dialog.
        """
        return lambda: self.open_series_dialog(idx, series_idx, True)

    def set_arc_type(self, card_type, is_mixed, allow_nb_out, outflow_arc_id, inflow_arc_id):
        """Sets the current type for this arc.

        Args:
            card_type (str): The current card type. Does not include 'DB' or 'NB'.
            is_mixed (bool): True if multiple things were selected.
            allow_nb_out (bool): True if the user should be able to select NB OUT.
            outflow_arc_id (int): The attribute id of the outflow arc in an NB OUT card.
            inflow_arc_id (int): The attribute id of the inflow arc in an NB OUT card.
        """
        if is_mixed:
            self.ui.warning_label.setText('All selected arcs will be reassigned to the current options on ok.')
            self.ui.warning_label.setStyleSheet("QLabel { color : red; }")
        else:
            self.ui.warning_label.hide()
        self.ui.arc_type.setCurrentIndex(self.card_to_index[card_type])
        if card_type == '':
            self.arc_type_changed(self.card_to_index[card_type])
        if not allow_nb_out:
            self.index_to_card.pop(self.card_to_index['OUT'])
            self.ui.arc_type.removeItem(self.card_to_index['OUT'])
        else:
            option_1 = f'Arc {outflow_arc_id} as outflow, arc {inflow_arc_id} as inflow'
            option_2 = f'Arc {inflow_arc_id} as outflow, arc {outflow_arc_id} as inflow'
            self.ui.out_arc_assignment.addItems([option_1, option_2])
        # connect the button here since the NB OUT index may not be available.
        for idx, card in enumerate(self.index_to_card):
            if card[self.SERIES_BTN] and (card[self.CARD] != 'DIS' or self.populate_flow_data is None):
                for series_idx, btn in enumerate(card[self.SERIES_BTN]):
                    btn.clicked.connect(self.get_open_series_dialog_lambda(idx, series_idx))
            elif card[self.SERIES_BTN] and card[self.CARD] == 'DIS':
                for series_idx, btn in enumerate(card[self.SERIES_BTN]):
                    btn.clicked.connect(self.get_open_series_populate_dialog_lambda(idx, series_idx))

    def set_sdr_data(self, coef_a, coef_b, coef_c, coef_d, coef_e):
        """Sets the values associated with the NB SDR card.

        Args:
            coef_a (float): The a coefficent value.
            coef_b (float): The b coefficent value.
            coef_c (float): The c coefficent value.
            coef_d (float): The d coefficent value.
            coef_e (float): The e coefficent value.
        """
        self.ui.sdr_coef_a.setText(str(coef_a))
        self.ui.sdr_coef_b.setText(str(coef_b))
        self.ui.sdr_coef_c.setText(str(coef_c))
        self.ui.sdr_coef_d.setText(str(coef_d))
        self.ui.sdr_coef_e.setText(str(coef_e))

    def get_sdr_data(self):
        """Gets the values associated with the NB SDR card.

        Returns:
            A list of float coefficients a through e.
        """
        return [
            float(self.ui.sdr_coef_a.text()),
            float(self.ui.sdr_coef_b.text()),
            float(self.ui.sdr_coef_c.text()),
            float(self.ui.sdr_coef_d.text()),
            float(self.ui.sdr_coef_e.text())
        ]

    def set_snapping_type(self, snap):
        """Sets the current snapping type.

        The text comes from CardInfo.snap_options.

        Args:
            snap (str): A string value that is one of 'Point snap', 'Edgestring snap', or 'Midstring snap'.
        """
        self.ui.snapping_type.setCurrentText(snap)

    def get_snapping_type(self):
        """Gets the current snapping type.

        The text comes from CardInfo.snap_options.

        Returns:
            A string value that is one of 'Point snap', 'Edgestring snap', or 'Midstring snap'.
        """
        return self.ui.snapping_type.currentText()

    def set_series(self, all_series):
        """Sets the series of the current card.

        Args:
            all_series (list): A list of all of the series used by the current card.
        """
        if all_series:
            idx = self.ui.arc_type.currentIndex()
            self.series_data[idx] = all_series

    def get_series(self, series_idx):
        """Gets the series for the current card index.

        Args:
            series_idx (int): The series index for the current card.

        Returns:
            An XY series, or None if not found.
        """
        idx = self.ui.arc_type.currentIndex()
        if idx in self.series_data:
            if series_idx < len(self.series_data[idx]):
                return self.series_data[idx][series_idx]
            else:
                return None
        else:
            return None

    def get_arc_card_type(self):
        """Gets the card text for the current arc type.

        Returns:
            Returns text for the current card.
        """
        idx = self.ui.arc_type.currentIndex()
        return self.index_to_card[idx][self.CARD]

    def arc_type_changed(self, index):
        """Called when the arc type combobox selection changed. Shows options for the type and hides all others.

        Args:
            index (int): The index in the arc type combo box.
        """
        is_out = self.index_to_card[index][self.CARD] == 'OUT'
        self.ui.out_arc_assignment_label.setVisible(is_out)
        self.ui.out_arc_assignment.setVisible(is_out)
        is_db = self.index_to_card[index][self.CARD] in ['OVH', 'LDE', 'LDH', 'LID']
        self.ui.snapping_type_label.setVisible(is_db)
        self.ui.snapping_type.setVisible(is_db)
        self.ui.card_options_stack.setCurrentWidget(self.index_to_card[index][self.WIDGETS])

    def get_out_assignment_changed(self):
        """Checks to see if the assignment for arcs of NB OUT changed.

        Returns:
            Returns true if the NB OUT arc assignment changed.
        """
        is_first_index = self.ui.out_arc_assignment.currentIndex() == 0
        return not is_first_index

    def open_series_dialog(self, card_index, series_index, populate=False):
        """Opens the XY series editor.

        Args:
            card_index (int): The current card index.
            series_index (int): The series index for the current card.
            populate (bool): True if populate buttons should be added.
        """
        if card_index not in self.series_data:
            self.series_data[card_index] = []
            for _ in self.index_to_card[card_index][self.SERIES_BTN]:
                self.series_data[card_index].append(TimeSeries())
                df = pd.DataFrame([[0.0, 0.0]], columns=['X', 'Y'])
                self.series_data[card_index][-1].time_series = \
                    pd.concat([self.series_data[card_index][-1].time_series, df])
        series = self.series_data[card_index][series_index].time_series
        column_map = self.index_to_card[card_index][self.XY_COLUMNS][series_index]

        if populate:
            editor = TimeSeriesEditor(
                series, icon=self.windowIcon(), parent=self, populate_flow_data=self.populate_flow_data
            )
        else:
            editor = TimeSeriesEditor(series, icon=self.windowIcon(), parent=self, column_map=column_map)
        if editor.run():
            self.series_data[card_index][series_index].time_series = editor.series

    def get_number_of_series(self):
        """Gets the number of possible series for the current card index.

        Returns:
            The number of XY series used by the card.
        """
        idx = self.ui.arc_type.currentIndex()
        if idx < 0 or idx >= len(self.index_to_card):
            return 0
        if self.index_to_card[idx][self.SERIES_BTN] is None:
            return 0
        return len(self.index_to_card[idx][self.SERIES_BTN])

    def set_populate_flow_data(self, populate_data):
        """Sets the data for the populate flow option.

        Args:
            populate_data (PopulateFlowData): The data for populating a time series from flow. Has active Query.
        """
        self.populate_flow_data = populate_data

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the Ok button is clicked."""
        return super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        return super().reject()
