"""Time series editor."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules
from xms.adh.gui.xy_series_populate_dialog import XySeriesPopulateDialog


class TimeSeriesEditor:
    """Time series editor."""
    def __init__(
        self,
        series,
        series_name='',
        dialog_title='XY Series Editor',
        icon=None,
        parent=None,
        column_map=None,
        populate_flow_data=None
    ):
        """
        Initialise the time series editor.

        Args:
            series: The series to be edited.
            series_name: The series name.
            dialog_title: The dialog title.
            icon: The window icon.
            parent: The window parent.
            column_map: The map that defines the series column names.
            populate_flow_data: Data from the last time populate was used.
        """
        self.series = series
        self._series_name = series_name
        self._dialog_title = dialog_title
        self._icon = icon
        self._parent = parent
        self._column_map = column_map
        self._populate_flow_data = populate_flow_data

    def run(self) -> bool:
        """
        Runs the time series editor.

        Returns:
        - bool: True if the user clicked OK.
        """
        if self._populate_flow_data is not None:
            dlg = XySeriesPopulateDialog(
                self.series,
                series_name=self._series_name,
                icon=self._icon,
                parent=self._parent,
                populate_flow_data=self._populate_flow_data
            )
        else:
            if self._column_map is not None:
                self.series = self.series.rename(columns=self._column_map)
            dlg = XySeriesEditor(
                self.series, self._series_name, dialog_title=self._dialog_title, icon=self._icon, parent=self._parent
            )
        if dlg.exec():
            time_series = dlg.model.data_frame
            if self._column_map is not None:
                reverse_map = {value: key for key, value in self._column_map.items()}
                time_series = time_series.rename(columns=reverse_map)
            # remove any empty rows
            time_series.dropna(inplace=True)
            time_series.reset_index(drop=True, inplace=True)
            time_series.index = time_series.index + 1
            self.series = time_series
            return True
        return False
