"""A base class widget for tables."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules


class AdhTableWidget(BasicTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame, select_col, filter_model, column_delegates, is_fixed=False):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
            select_col (int): Column to select when adding/removing rows
            filter_model (QSortFilterProxyModel): A model for sorting, filtering, and changing how the data is viewed.
            column_delegates (dict): A dictionary with column index as the key and a QStyledItemDelegate as the value.
            is_fixed (bool): True if the table size is fixed.
        """
        super().__init__(parent)
        self.select_col = select_col
        self.filter_model = filter_model
        self.column_delegates = column_delegates
        self.is_fixed_size = is_fixed
        # self.default_values = default_values
        if data_frame is not None:
            if len(data_frame.index) > 0 and data_frame.index[0] == 0:
                data_frame.index += 1  # Make vertical header 1-based
            self.model = QxPandasTableModel(data_frame, self)
            default_values = []
            for col in data_frame.columns:
                if 'CARD' in col:
                    default_values.append('')
                elif 'ID' in col:
                    default_values.append(0)
                else:
                    default_values.append(0.0)
            self.default_values = [default_values]
            self.model.set_default_values(self.default_values)
        self.setup_ui()

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )

    def setup_ui(self):
        """Sets up the base class."""
        super()._setup_ui(self.column_delegates, False, self.is_fixed_size, self.filter_model)
