"""Map locations and attributes of applicable linked coverages to the AdH domain."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.adh.mapping.bc_mapper import BcMapper
from xms.adh.mapping.material_mapper import MaterialMapper


class CoverageMapper:
    """Class for mapping coverages to a mesh for AdH."""
    def __init__(self, generate_snap):
        """Constructor.

        Args:
            generate_snap (bool): True if generating components for snap preview.
        """
        super().__init__()
        self._logger = logging.getLogger('xms.adh')
        self._generate_snap = generate_snap
        self._mesh = None
        self._active_mesh = None
        self._bc_cov = None
        self._mat_cov = None
        self._sed_mat_cov = None
        self._bc_comp = None
        self._mat_comp = None
        self._sed_mat_comp = None
        self._transport_comp = None
        self._sediment_comp = None
        self._cell_activity = None
        self._wkt = ''

        self.mapped_comps = []

    def set_mesh(self, mesh, wkt):
        """Sets the mesh geometry of the domain.

        Args:
            mesh (CoGrid): A mesh geometry of the domain.
            wkt (str): The 'well known text' of the projection.
        """
        self._mesh = mesh
        self._wkt = wkt

    def set_materials(self, material_cov, material_comp):
        """Sets the hydrodynamic materials and active areas of the domain.

        Args:
            material_cov (Coverage): A material coverage.
            material_comp (MaterialConceptualComponent): A hydrodynamic material component.
        """
        self._mat_cov = material_cov
        self._mat_comp = material_comp

    def set_sediment_materials(self, material_cov, material_comp):
        """Sets the sediment materials of the domain.

        Args:
            material_cov (Coverage): A sediment material coverage.
            material_comp (SedimentMaterialConceptualComponent): A sediment material component.
        """
        self._sed_mat_cov = material_cov
        self._sed_mat_comp = material_comp

    def set_boundary_conditions(self, boundary_conditions, component):
        """Sets the boundary conditions coverage.

        Args:
            boundary_conditions (Coverage): A boundary conditions coverage.
            component (BcConceptualComponent): A boundary conditions component belonging to the coverage.
        """
        self._bc_cov = boundary_conditions
        self._bc_comp = component

    def set_transport_constituents(self, component):
        """Sets the transport constituents component used by the boundary conditions coverage.

        Args:
            component (TransportConstituentsComponent): A transport constituents component.
        """
        self._transport_comp = component

    def set_sediment_constituents(self, component):
        """Sets the sediment transport constituents component used by the boundary conditions coverage.

        Args:
            component (SedimentConstituentsComponent): A sediment transport constituents component.
        """
        self._sediment_comp = component

    def do_map(self):
        """Performs the snapping of the coverages to the domain."""
        try:
            self._map_materials()
            self._map_boundary_conditions()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def _map_boundary_conditions(self):
        """Maps the boundary conditions coverage onto the domain."""
        if not self._bc_cov:
            return
        self._logger.info('Mapping boundary conditions coverage to mesh.')
        mapper = BcMapper(self, generate_snap=self._generate_snap)
        do_comps, comps = mapper.do_map()
        for do_comp, comp, in zip(do_comps, comps):
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_component'))
        self._logger.info('Finished mapping boundary conditions coverage to mesh.')

    def _map_materials(self):
        """Maps the material and sediment coverage onto the domain."""
        if not self._mat_cov:
            return
        if self._sed_mat_cov:
            coverage_message = 'hydrodynamic and sediment material coverages'
        else:
            coverage_message = 'material coverage'
        self._logger.info(f'Mapping {coverage_message} to mesh.')
        mapper = MaterialMapper(self, generate_snap=self._generate_snap)
        do_comp, comp, self._active_mesh = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()], 'mapped_component'))
        self._logger.info(f'Finished {coverage_message} to mesh.')
