"""The help pane in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
import pandas as pd
from PySide2.QtWidgets import (QCheckBox, QDialog, QGroupBox, QHBoxLayout, QPushButton, QSizePolicy, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules


class StructureProfileTab:
    """The profiles tab in the structure dialog."""
    def __init__(self, parent, widgets):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
        """
        self._parent = parent
        self._widgets = widgets
        self._mp_figure = {}
        self._mp_canvas = {}
        self._mp_ax = {}
        self._models = {}
        self._default_df = pd.DataFrame({'Distance': [0.0], 'Elevation': [0.0]})
        self._default_df.index += 1
        self._check_all_profiles()  # call before _add_widgets
        self._add_widgets()

    def _check_all_profiles(self):
        """Ensure we have valid profiles."""
        for name, df1 in self._parent.data.curves.items():
            self._parent.data.curves[name] = self._check_profile(df1)

    def _check_profile(self, df):
        """Ensure the profile is valid."""
        df1 = self._default_df.copy()
        if len(df) > 0:
            df1 = df
        df1 = df1.reset_index(drop=True)
        df1.index += 1
        return df1

    def _add_widgets(self):
        """Setup the UI."""
        vlayout = self._widgets['tab0_scroll_area_vert_layout']

        self._models['top'] = QxPandasTableModel(self._parent.data.curves['top_profile'])
        self._models['upstream'] = QxPandasTableModel(self._parent.data.curves['upstream_profile'])
        self._models['downstream'] = QxPandasTableModel(self._parent.data.curves['downstream_profile'])

        # top profile curve
        checked, crest = self._profile_constant('top')
        self._widgets['grp_top'] = QGroupBox('Crest profile')
        self._widgets['vlayout_top'] = QVBoxLayout()
        self._widgets['grp_top'].setLayout(self._widgets['vlayout_top'])
        tog_txt = 'Use constant crest elevation (overwrites existing profile)'
        self._widgets['tog_const_crest'] = QCheckBox(tog_txt)
        self._widgets['tog_const_crest'].stateChanged.connect(self._on_const_crest)
        self._widgets['vlayout_top'].addWidget(self._widgets['tog_const_crest'])
        self._parent.add_floating_point_widget(
            'const_crest', 'Constant bridge crest elevation:', crest, self._widgets['vlayout_top']
        )
        self._widgets['hlayout_top'] = QHBoxLayout()
        self._widgets['vlayout_top'].addLayout(self._widgets['hlayout_top'])
        self._add_mp_figure('top', self._widgets['hlayout_top'])
        self._widgets['btn_edit_top'] = QPushButton('Edit...')
        self._widgets['hlayout_top'].addWidget(self._widgets['btn_edit_top'])
        self._widgets['grp_top'].setLayout(self._widgets['hlayout_top'])
        vlayout.addWidget(self._widgets['grp_top'])
        self._widgets['btn_edit_top'].clicked.connect(self._on_btn_edit_top)
        self._widgets['tog_const_crest'].setChecked(checked)
        self._on_const_crest()

        # upstream profile curve
        checked, up_elev = self._profile_constant('upstream')
        self._widgets['grp_up'] = QGroupBox('Low chord upstream profile')
        self._widgets['vlayout_up'] = QVBoxLayout()
        self._widgets['grp_up'].setLayout(self._widgets['vlayout_up'])
        tog_txt = 'Use constant low chord upstream elevation (overwrites existing profile)'
        self._widgets['tog_const_upstream'] = QCheckBox(tog_txt)
        self._widgets['tog_const_upstream'].stateChanged.connect(self._on_const_upstream)
        self._widgets['vlayout_up'].addWidget(self._widgets['tog_const_upstream'])
        self._parent.add_floating_point_widget(
            'const_upstream', 'Constant bridge low chord upstream elevation:', up_elev, self._widgets['vlayout_up']
        )
        self._widgets['hlayout_up'] = QHBoxLayout()
        self._widgets['vlayout_up'].addLayout(self._widgets['hlayout_up'])
        self._add_mp_figure('upstream', self._widgets['hlayout_up'])
        self._widgets['btn_edit_up'] = QPushButton('Edit...')
        self._widgets['hlayout_up'].addWidget(self._widgets['btn_edit_up'])
        self._widgets['grp_up'].setLayout(self._widgets['hlayout_up'])
        vlayout.addWidget(self._widgets['grp_up'])
        self._widgets['btn_edit_up'].clicked.connect(self._on_btn_edit_upstream)
        self._widgets['tog_const_upstream'].setChecked(checked)
        self._on_const_upstream()

        # toggle for downstream profile curve
        self._widgets['tog_downstream'] = QCheckBox('Specify downstream profile')
        checked = True if self._parent.data.data_dict['bridge_specify_downstream_profile'] else False
        self._widgets['tog_downstream'].setChecked(checked)
        vlayout.addWidget(self._widgets['tog_downstream'])
        self._widgets['tog_downstream'].stateChanged.connect(self._on_specify_downstream)

        # downstream profile curve
        self._widgets['grp_down'] = QGroupBox('Low chord downstream profile')
        self._widgets['hlayout_down'] = QHBoxLayout()
        self._add_mp_figure('downstream', self._widgets['hlayout_down'])
        self._widgets['btn_edit_down'] = QPushButton('Edit...')
        self._widgets['hlayout_down'].addWidget(self._widgets['btn_edit_down'])
        self._widgets['grp_down'].setLayout(self._widgets['hlayout_down'])
        vlayout.addWidget(self._widgets['grp_down'])
        self._widgets['grp_down'].setVisible(checked)
        self._widgets['btn_edit_down'].clicked.connect(self._on_btn_edit_downstream)

        self._add_line_series('top')
        self._add_line_series('upstream')
        self._add_line_series('downstream')

        vlayout.addStretch()

    def _profile_constant(self, name):
        """Determine if the profiles is constant."""
        df = self._models[name].data_frame
        vals = set(df['Elevation'].to_list())
        if len(vals) == 1:
            return True, vals.pop()
        return False, vals.pop()

    def _on_const_crest(self):
        """Called when the specify downstream profile check box is clicked."""
        const_flag = self._widgets['tog_const_crest'].isChecked()
        self._widgets['txt_const_crest'].setVisible(const_flag)
        self._widgets['edt_const_crest'].setVisible(const_flag)
        self._mp_canvas['top'].setVisible(not const_flag)
        self._widgets['btn_edit_top'].setVisible(not const_flag)

    def _on_const_upstream(self):
        """Called when the specify downstream profile check box is clicked."""
        const_flag = self._widgets['tog_const_upstream'].isChecked()
        self._widgets['txt_const_upstream'].setVisible(const_flag)
        self._widgets['edt_const_upstream'].setVisible(const_flag)
        self._mp_canvas['upstream'].setVisible(not const_flag)
        self._widgets['btn_edit_up'].setVisible(not const_flag)

    def _on_specify_downstream(self):
        """Called when the specify downstream profile check box is clicked."""
        self._widgets['grp_down'].setVisible(self._widgets['tog_downstream'].isChecked())

    def _add_mp_figure(self, name, layout):
        """Adds a matplotlib figure to the layout.

        Args:
             name (:obj:`str`): name of the figure
             layout (:obj:`QLayout`): qt widget
        """
        self._mp_figure[name] = Figure()
        self._mp_figure[name].set_layout_engine(layout='tight')
        self._mp_canvas[name] = FigureCanvas(self._mp_figure[name])
        self._mp_canvas[name].setMinimumWidth(100)
        self._mp_canvas[name].setMinimumHeight(150)
        self._mp_canvas[name].setSizePolicy(QSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.MinimumExpanding))
        self._mp_ax[name] = self._mp_figure[name].add_subplot(111)
        self._mp_ax[name].grid(True)
        layout.addWidget(self._mp_canvas[name])

    def _on_btn_edit_top(self):
        """Opens the XY Series Editor to display the bridge top profile."""
        rv, df = self._on_btn_edit('Top', 'top', self._parent.data.curves['top_profile'])
        if rv:
            self._parent.data.curves['top_profile'] = df

    def _on_btn_edit_upstream(self):
        """Opens the XY Series Editor to display the bridge upstream profile."""
        rv, df = self._on_btn_edit('Upstream', 'upstream', self._parent.data.curves['upstream_profile'])
        if rv:
            self._parent.data.curves['upstream_profile'] = df

    def _on_btn_edit_downstream(self):
        """Opens the XY Series Editor to display the bridge downstream profile."""
        rv, df = self._on_btn_edit('Downstream', 'downstream', self._parent.data.curves['downstream_profile'])
        if rv:
            self._parent.data.curves['downstream_profile'] = df

    def _on_btn_edit(self, name, index, data_frame):
        """Opens the XySeriesEditor.

        Args:
            name (:obj:`str`): Name of the xy series.
            index (:obj:`str`): The index into the dictionaries.
            data_frame (:obj:`pandas.DataFrame`): The DataFrame.

        Returns:
            (:obj:`tuple`): tuple containing:

                rv (:obj:`bool`): True on OK, else False.

                modified (:obj:`pandas.DataFrame`): The modified dataframe on OK, else None.
        """
        xy_series_editor = XySeriesEditor(data_frame=data_frame, series_name=name, parent=self._parent)
        if xy_series_editor.exec() == QDialog.Accepted:
            df = xy_series_editor.model.data_frame
            df = df.sort_values(df.columns[0])
            df = self._check_profile(df)
            self._models[index] = QxPandasTableModel(df)
            self._add_line_series(index)
            return True, df
        return False, None

    def _add_line_series(self, name):
        """Add line plot.

        Args:
            name (:obj:`str`): name of the series added
        """
        self._mp_ax[name].clear()
        self._mp_ax[name].grid(True)
        x_column = []
        y_column = []
        if self._models[name].rowCount() > 0:
            # Add data to plot
            x_column = self._models[name].data_frame.iloc[:, 0]
            y_column = self._models[name].data_frame.iloc[:, 1]

        self._mp_ax[name].plot(x_column, y_column)
        self._mp_canvas[name].draw()
