"""Module for the CardWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['CardWriter', 'SEQUENCE']

# 1. Standard Python modules
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

SEQUENCE = object()


class CardWriter:
    """Class for writing cards to a .cmcards file."""
    def __init__(self, file: TextIO):
        """
        Initialize the writer.

        Args:
            file: Where to write cards to.
        """
        self._file = file

    def write(self, card: str, *values, indent: int = 1):
        """
        Write a card.

        The line will be formatted so that the value starts in the same column, regardless of how long the card is.

        Args:
            card: The card to write.
            values: The card's values. If omitted, will not be written. Passing SINGLE_QUOTE or DOUBLE_QUOTE will put
                quote marks around the next value passed.
            indent: Level of indentation. 2*indent spaces will be written before the card.
        """
        card = ('  ' * indent) + card

        processed_values = []
        unprocessed_values = iter(values)
        for unprocessed_value in unprocessed_values:
            if unprocessed_value is SEQUENCE:
                sequence = next(unprocessed_values)
                joined = ' '.join(str(item) for item in sequence)
                processed_values.append(joined)
            else:
                processed_values.append(str(unprocessed_value))

        if processed_values:
            to_write = '  '.join(processed_values)
            self._file.write(f'{card:<36}{to_write}\n')
        else:
            self._file.write(f'{card}\n')

    def write_newline(self):
        """Write a blank line."""
        self._file.write('\n')
