"""This is a table widget for specifying bed layers."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, QModelIndex, Qt
from PySide2.QtWidgets import QAbstractItemView, QComboBox, QHeaderView

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import TreeNode
from xms.guipy.delegates.dataset_button_delegate import DatasetButtonDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.resources import resources_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.cmsflow.gui.cmsflow_table_widget import CmsflowTableWidget


class RowDependentComboboxDelegate(QxCbxDelegate):
    """This delegate changes the combo box options depending on the row."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates the combobox and populates it.

        Args:
            parent (QWidget): The parent.
            option (QStyleOptionViewItem): The option
            index (QModelIndex): The index

        Returns:
            QWidget
        """
        self.cb = QComboBox(parent)
        current_strings = ['Dataset', 'Constant']
        if index.row() == 0:
            current_strings.append('Automatic')
        self.cb.addItems(current_strings)
        self.cb.currentIndexChanged.connect(self.on_index_changed)
        return self.cb


class BedLayerRenameModel(RenameModel):
    """A model to rename header titles."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        super().__init__(column_names, parent)

    def flags(self, index: QModelIndex) -> Qt.ItemFlags:
        """Returns flags for the indexes of the filtered model.

        Args:
            index (QModelIndex): The index for which to get flags.
        """
        _layer_id_col = 0
        _thickness_type_col = 1
        _thickness_dset_col = 2
        _thickness_const_col = 3
        local_flags = super().flags(index)
        if index.column() == _thickness_type_col:
            if self.rowCount() == 1:
                local_flags = local_flags & ~Qt.ItemIsEnabled
        elif index.column() in [_thickness_dset_col, _thickness_const_col]:
            current_method = index.model().index(index.row(), _thickness_type_col, index.parent()).data(Qt.EditRole)

            if self.rowCount() == 1:
                local_flags = local_flags & ~Qt.ItemIsEnabled
            if index.column() == _thickness_dset_col:
                if current_method == 'Dataset':
                    local_flags = local_flags | Qt.ItemIsEnabled
                else:
                    local_flags = local_flags & ~Qt.ItemIsEnabled
            else:
                if current_method == 'Constant':
                    local_flags = local_flags | Qt.ItemIsEnabled
                else:
                    local_flags = local_flags & ~Qt.ItemIsEnabled
        if index.column() == _layer_id_col and index.row() == 0:
            local_flags = local_flags & ~Qt.ItemIsEnabled
        return local_flags


class BedLayersTableWidget(CmsflowTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame, pe_tree):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
            pe_tree (TreeNode): The project explorer tree.
        """
        self.rename_model = None
        self.dset_delegate = None
        self.edit_delegate = None
        self.int_edit_delegate = None
        self.combo_delegate = None
        self.dbl_validator = None
        self.int_validator = None
        self.pe_tree = pe_tree
        default_values = {
            'layer_id': 1,
            'layer_thickness_type': 'Dataset',
            'layer_thickness': '',
            'layer_thickness_const': 0.5,
            'd05': '',
            'd10': '',
            'd16': '',
            'd20': '',
            'd30': '',
            'd35': '',
            'd50': '',
            'd65': '',
            'd84': '',
            'd90': '',
            'd95': ''
        }
        super().__init__(parent, data_frame, 0, default_values)

    def setup_ui(self):
        """Sets up the delegates, validators, and filter model for this table."""
        column_names = [
            'Layer ID', 'Layer Thickness Type', 'Layer Thickness Dataset', 'Layer Thickness Constant', 'D05', 'D10',
            'D16', 'D20', 'D30', 'D35', 'D50', 'D65', 'D84', 'D90', 'D95'
        ]
        self.rename_model = BedLayerRenameModel(column_names, self)
        self.dset_delegate = DatasetButtonDelegate(
            self.pe_tree, _is_scalar_cell_dset, resources_util.get_quadtree_icon, self
        )
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.int_validator = QxIntValidator(parent=self, bottom=1)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.int_edit_delegate = EditFieldValidator(self.int_validator)
        self.combo_delegate = RowDependentComboboxDelegate(self)
        self.combo_delegate.set_strings(['Dataset', 'Constant', 'Automatic'])
        delegates = {
            0: self.int_edit_delegate,
            1: self.combo_delegate,
            2: self.dset_delegate,
            3: self.edit_delegate,
            4: self.dset_delegate,
            5: self.dset_delegate,
            6: self.dset_delegate,
            7: self.dset_delegate,
            8: self.dset_delegate,
            9: self.dset_delegate,
            10: self.dset_delegate,
            11: self.dset_delegate,
            12: self.dset_delegate,
            13: self.dset_delegate,
            14: self.dset_delegate
        }
        super()._setup_ui(delegates, False, False, self.rename_model)
        self.table_view.horizontalHeader().setMinimumSectionSize(100)
        self.table_view.horizontalHeader().setSectionResizeMode(4, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(5, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(6, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(7, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(8, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(9, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(10, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(11, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(12, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(13, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(14, QHeaderView.Stretch)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
        max_id = 0
        for row in range(row_idx):
            layer_id = int(self.model.index(row, self.select_col).data())
            max_id = max(max_id, layer_id)
        self.model.setData(new_index, str(max_id + 1))

    def on_selection_changed(self):
        """Called when the user clicks in the table and changes what cells are selected."""
        selection_list = self.table_view.selectedIndexes()
        selections_exist = len(selection_list) > 0
        empty_table = (self.model.rowCount() <= 1)
        enable_actions = selections_exist and not empty_table
        if enable_actions:
            # check that the first row is not selected
            for selection in selection_list:
                if selection.row() == 0:
                    enable_actions = False
                    break

        self.toolbar.widgetForAction(self.btn_actions[self.delete_icon]).setEnabled(enable_actions)


def _is_scalar_cell_dset(item: TreeNode) -> bool:
    """
    Check if a tree item is scalar cell dataset.

    Args:
        item: The item to check

    Returns:
        Whether the item is a scalar cell dataset.
    """
    if type(item.data) is not xmd.DatasetItem:
        return False

    if item.num_components != 1:
        return False

    if item.data_location != 'CELL':
        return False

    return True
