"""Dialog for editing the meteorological stations wind direction and velocity data."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.cmsflow.gui import wind_parameters_table_widget
from xms.cmsflow.gui.meteorological_wind_dialog_ui import Ui_MeteorologicalWindDialog


class MeteorologicalWindDialog(XmsDlg):
    """A dialog for viewing and editing a meteorological stations wind direction and velocity curve data."""
    def __init__(self, wind_data, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            wind_data (pandas.DataFrame): A dataframe containing three columns (Time, Direction, Velocity).
            parent (Something derived from :obj:`QWidget`): The parent window

        """
        super().__init__(parent, 'xms.cmsflow.gui.meteorological_wind_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:CMS-Flow'
        self.ui = Ui_MeteorologicalWindDialog()
        self.ui.setupUi(self)

        self.wind_table = wind_parameters_table_widget.WindParametersTableWidget(self, wind_data)
        self.wind_table.setObjectName('wind_table')
        self.wind_table.setMinimumHeight(205)  # min height from the XML
        self.ui.vertical_layout.insertWidget(3, self.wind_table)  # 3 is after the labels, before the button box

        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.ui.button_box.helpRequested.connect(self.help_requested)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)
